﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nsd/nsd_ApiForTest.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/pctl/pctl_Api.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/detail/ipc/pctl_IpcConfig.h>
#include <nn/pctl/detail/service/pctl_IpcServer.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/settings/system/settings_Region.h>

static nn::pctl::detail::service::watcher::ServerDeviceId g_DeviceId = 0;

TEST(Network, Initialize)
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::nsd::SetTd1EnvironmentForTest();
#endif

    nn::pctl::detail::service::InitializeAllocatorForServer();
    nn::pctl::detail::service::InitializeMain();
    nn::pctl::detail::service::InitializeWatcher();

    auto& manager = nn::pctl::detail::service::g_pMain->GetSettingsManager();
    manager.DeleteSettings();
    manager.SetPinCode("12356", 6);
    // ダミーで追加
    {
        nn::pctl::FreeCommunicationApplicationInfo info;
        info.applicationId.value = 0x0100000000014000;
        manager.AddFreeCommunicationApplicationList(info.applicationId);
        info.isRestricted = false;
        manager.UpdateFreeCommunicationApplicationList(&info, 1);
        info.applicationId.value = 0x0100000000014001;
        manager.AddFreeCommunicationApplicationList(info.applicationId);
        info.isRestricted = true;
        manager.UpdateFreeCommunicationApplicationList(&info, 1);
    }
}

TEST(Network, TestRequestPairing)
{
    nn::pctl::detail::service::common::AsyncContext* pContext;
    nn::pctl::detail::service::watcher::ServerDeviceId deviceId;
    nn::pctl::detail::service::watcher::ServerDeviceOwner owners[5];
    size_t actualCount = 0;

    g_DeviceId = 0;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::pctl::detail::service::g_pWatcher->GetNetworkManager().RequestPairing(
            &pContext,
            &deviceId,
            &actualCount,
            "909689",
            6,
            owners,
            std::extent<decltype(owners)>::value
            )
        );
    while (NN_STATIC_CONDITION(true))
    {
        nn::Result result;
        if (pContext->IsFinished(&result))
        {
            NNT_EXPECT_RESULT_SUCCESS(result);
            pContext->CloseContext();
            if (result.IsSuccess())
                g_DeviceId = deviceId;
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(30));
    }
}

TEST(Network, TestAuthorizePairing)
{
    if (g_DeviceId == 0)
        return;

    nn::pctl::detail::service::common::AsyncContext* pContext;

    NNT_EXPECT_RESULT_SUCCESS(
        nn::pctl::detail::service::g_pWatcher->GetNetworkManager().RequestAuthorizePairing(
            &pContext,
            g_DeviceId
            )
        );
    while (NN_STATIC_CONDITION(true))
    {
        nn::Result result;
        if (pContext->IsFinished(&result))
        {
            NNT_EXPECT_RESULT_SUCCESS(result);
            pContext->CloseContext();
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(30));
    }
}

TEST(Network, TestSynchronizeSettings)
{
    if (g_DeviceId == 0)
        return;

    nn::pctl::detail::service::common::AsyncContext* pContext;

    // Etagなしの取得
    NNT_EXPECT_RESULT_SUCCESS(
        nn::pctl::detail::service::g_pWatcher->GetNetworkManager().RequestSynchronizeSettingsForeground(
            &pContext,
            g_DeviceId,
            true
            )
        );
    while (NN_STATIC_CONDITION(true))
    {
        nn::Result result;
        if (pContext->IsFinished(&result))
        {
            NNT_EXPECT_RESULT_SUCCESS(result);
            pContext->CloseContext();
            if (result.IsFailure())
                return;
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(30));
    }

    // Etag込みの取得(304 で成功になるはず)
    NNT_EXPECT_RESULT_SUCCESS(
        nn::pctl::detail::service::g_pWatcher->GetNetworkManager().RequestSynchronizeSettingsForeground(
            &pContext,
            g_DeviceId,
            false
            )
        );
    while (NN_STATIC_CONDITION(true))
    {
        nn::Result result;
        if (pContext->IsFinished(&result))
        {
            NNT_EXPECT_RESULT_SUCCESS(result);
            pContext->CloseContext();
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(30));
    }
}

// 設定変更をバックグラウンド処理が検知してアップロードするテスト(やや暫定実装)
TEST(Network, ChangeSettings)
{
    if (g_DeviceId == 0)
        return;

    nn::settings::system::RegionCode region;
    nn::settings::system::GetRegionCode(&region);
    if (region == nn::settings::system::RegionCode_Japan)
    {
        nn::settings::system::SetRegionCode(nn::settings::system::RegionCode_Usa);
    }
    else
    {
        nn::settings::system::SetRegionCode(nn::settings::system::RegionCode_Japan);
    }
    // 設定変更を待つ
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(20));

    NNT_EXPECT_RESULT_SUCCESS(
        nn::pctl::detail::service::g_pWatcher->GetNetworkManager().WaitForBackgroundTaskForTest()
    );
}

TEST(Network, TestUnlink)
{
    if (g_DeviceId == 0)
        return;

    nn::pctl::detail::service::common::AsyncContext* pContext;

    NNT_EXPECT_RESULT_SUCCESS(
        nn::pctl::detail::service::g_pWatcher->GetNetworkManager().RequestUnlinkDevice(
            &pContext,
            g_DeviceId,
            false
            )
        );
    while (NN_STATIC_CONDITION(true))
    {
        nn::Result result;
        if (pContext->IsFinished(&result))
        {
            NNT_EXPECT_RESULT_SUCCESS(result);
            pContext->CloseContext();
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(30));
    }
}

TEST(Network, Finalize)
{
    nn::pctl::detail::service::FinalizeWatcher();
    nn::pctl::detail::service::FinalizeMain();
}
