﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nifm.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/pctl/pctl_Api.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/detail/ipc/pctl_IpcConfig.h>
#include <nn/pctl/detail/service/pctl_IpcServer.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_Constants.h>
#include <nn/pctl/detail/service/common/pctl_FileSystem.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/util/util_StringUtil.h>


using namespace nn::pctl;
using namespace nn::pctl::detail::service;

class ManagerTest : public ::testing::Test
{
public:
    static void SetUpTestCase()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());

        nn::nifm::SubmitNetworkRequestAndWait();

        NN_PRAGMA_PUSH_WARNINGS
        NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS // EnsureNetworkClockAvailability deprecated warning 回避
        auto result = nn::time::EnsureNetworkClockAvailability();
        NN_PRAGMA_POP_WARNINGS

        g_IsNetworkClockAvailable = result.IsSuccess();
        if (result.IsFailure())
        {
            NN_LOG("Network clock is not available (0x%08lX). Some tests may be skipped...\n",
                result.GetInnerValueForDebug());
        }

        InitializeAllocatorForServer();
        InitializeMain();
        InitializeWatcher();

        // BGスレッド処理のため少し待機する
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        // イベント処理を止める
        g_pWatcher->GetWatcherEventManager().DisableEventProcess();

        // 一旦破棄する
        auto& storage = g_pWatcher->GetWatcherEventStorage();
        storage.DiscardAllEvents();
    }

    static void TearDownTestCase()
    {
        g_pWatcher->GetNetworkManager().ClearPairingInfo(false);

        FinalizeWatcher();
        FinalizeMain();

        nn::time::Finalize();
    }

    static bool g_IsNetworkClockAvailable;
};
bool ManagerTest::g_IsNetworkClockAvailable = false;

TEST_F(ManagerTest, StoreSettings_CheckChanged)
{
    system::Settings settings;
    system::FreeCommunicationApplicationSettings appSettings;
    system::ExemptApplicationSettings exemptionSettings;
    using namespace nn::ovln::format;
    PctlSettingTypeFlagSet changedType;

    auto& manager = g_pMain->GetSettingsManager();

    std::memset(&settings, 0, sizeof(settings));
    std::memset(&appSettings, 0, sizeof(appSettings));
    std::memset(&exemptionSettings, 0, sizeof(exemptionSettings));

    changedType.Reset();
    nn::util::Strlcpy(settings.current.pinCode, "1234", system::MaxPinCodeLength);
    settings.current.safetyLevel = system::SizedSafetyLevel::SizedSafetyLevel_None;
    settings.current.isDefaultRatingOrganizationSet = 1;
    manager.StoreAllSettings(&changedType, &settings, &appSettings, nullptr, nullptr, &exemptionSettings);
    manager.SetPinCode(nullptr, 0);

    changedType.Reset();
    nn::util::Strlcpy(settings.current.pinCode, "1234", system::MaxPinCodeLength);
    manager.StoreAllSettings(&changedType, &settings, &appSettings, nullptr, nullptr, &exemptionSettings);
    EXPECT_TRUE(changedType.Test<PctlSettingTypeFlag::PinCode>());
    EXPECT_EQ(1, changedType.CountPopulation());

    changedType.Reset();
    settings.current.safetyLevel = system::SizedSafetyLevel::SizedSafetyLevel_Custom;
    manager.StoreAllSettings(&changedType, &settings, &appSettings, nullptr, nullptr, &exemptionSettings);
    EXPECT_TRUE(changedType.Test<PctlSettingTypeFlag::SafetyLevel>());
    EXPECT_EQ(1, changedType.CountPopulation());

    changedType.Reset();
    settings.custom.isSnsPostRestricted = true;
    manager.StoreAllSettings(&changedType, &settings, &appSettings, nullptr, nullptr, &exemptionSettings);
    EXPECT_TRUE(changedType.Test<PctlSettingTypeFlag::SnsPost>());
    EXPECT_EQ(1, changedType.CountPopulation());

    changedType.Reset();
    settings.custom.ratingAge = 15;
    manager.StoreAllSettings(&changedType, &settings, &appSettings, nullptr, nullptr, &exemptionSettings);
    EXPECT_TRUE(changedType.Test<PctlSettingTypeFlag::Rating>());
    EXPECT_EQ(1, changedType.CountPopulation());

    changedType.Reset();
    settings.custom.isSnsPostRestricted = false;
    settings.custom.isFreeCommunicationRestrictedByDefault = true;
    manager.StoreAllSettings(&changedType, &settings, &appSettings, nullptr, nullptr, &exemptionSettings);
    EXPECT_TRUE(changedType.Test<PctlSettingTypeFlag::SnsPost>());
    EXPECT_TRUE(changedType.Test<PctlSettingTypeFlag::FreeCommunication>());
    EXPECT_EQ(2, changedType.CountPopulation());

    changedType.Reset();
    nn::util::Strlcpy(settings.current.pinCode, "4234", system::MaxPinCodeLength);
    nn::ncm::ApplicationId id1 = { 0x0100000000001000 };
    nn::ncm::ApplicationId id2 = { 0x0100000000001001 };
    manager.AddFreeCommunicationApplicationList(id1);
    manager.AddFreeCommunicationApplicationList(id2);
    appSettings.AddRestrictedValue(id2, false);
    appSettings.AddRestrictedValue(id1, false);
    manager.StoreAllSettings(&changedType, &settings, &appSettings, nullptr, nullptr, &exemptionSettings);
    EXPECT_TRUE(changedType.Test<PctlSettingTypeFlag::PinCode>());
    EXPECT_TRUE(changedType.Test<PctlSettingTypeFlag::FreeCommunication>());
    EXPECT_EQ(2, changedType.CountPopulation());

#if 0
    changedType.Reset();
    nn::util::Strlcpy(settings.current.pinCode, "6789", system::MaxPinCodeLength);
    nn::ncm::ApplicationId id3 = { 0x0100000000001002 };
    nn::ncm::ApplicationId id4 = { 0x0100000000001003 };
    manager.AddExemptApplicationList(id3);
    manager.AddExemptApplicationList(id4);
    exemptionSettings.AddExemptedValue(id3, true);
    exemptionSettings.AddExemptedValue(id4, true);
    manager.StoreAllSettings(&changedType, &settings, &appSettings, nullptr, nullptr, &exemptionSettings);
    EXPECT_TRUE(changedType.Test<PctlSettingTypeFlag::ExemptAppList>());
    EXPECT_EQ(3, changedType.CountPopulation());
#endif

    nn::settings::system::RegionCode region;
    nn::settings::system::GetRegionCode(&region);
    auto safetyLevel = system::SizedSafetyLevel::SizedSafetyLevel_YoungTeens;
    auto& presetSettings = (region == nn::settings::system::RegionCode::RegionCode_Japan ?
        common::SafetyLevelPresetSettingsForJapan[safetyLevel] :
        common::SafetyLevelPresetSettings[safetyLevel]);
    int diffCount = 0;

    changedType.Reset();
    manager.GetSettings(settings);

    if (settings.current.ratingAge != presetSettings.ratingAge)
    {
        ++diffCount;
    }
    if ((settings.current.isSnsPostRestricted != 0) != presetSettings.snsPostRestriction)
    {
        ++diffCount;
    }
    ++diffCount; // SafetyLevel: Custom -> YoungTeens

    settings.current.safetyLevel = safetyLevel;
    manager.StoreAllSettings(&changedType, &settings, &appSettings, nullptr, nullptr, &exemptionSettings);
    if (settings.current.ratingAge != presetSettings.ratingAge)
    {
        EXPECT_TRUE(changedType.Test<PctlSettingTypeFlag::Rating>());
    }
    if ((settings.current.isSnsPostRestricted != 0) != presetSettings.snsPostRestriction)
    {
        EXPECT_TRUE(changedType.Test<PctlSettingTypeFlag::SnsPost>());
    }
    EXPECT_TRUE(changedType.Test<PctlSettingTypeFlag::SafetyLevel>());
    EXPECT_EQ(diffCount, changedType.CountPopulation());

    manager.DeleteSettings();
    manager.ClearFreeCommunicationApplicationList();
    manager.ClearExemptApplicationList();
} // NOLINT(impl/function_size)
