﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/pctl/pctl_Api.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/pctl_ApiForDebug.h>

TEST(LaunchCheck, Initialize)
{
    nn::pctl::SetPinCode(nullptr);
    nn::pctl::ClearFreeCommunicationApplicationListForDebug();
}

// ConfirmLaunchApplicationPermission

TEST(LaunchCheck, TestConfirmLaunchApplicationPermission_NoAppRestriction)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 18;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = 0;
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 0;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmLaunchApplicationPermission(appId, prop));
}

TEST(LaunchCheck, TestConfirmLaunchApplicationPermission_NoSettings)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = false;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmLaunchApplicationPermission(appId, prop));
}

TEST(LaunchCheck, TestConfirmLaunchApplicationPermission_HasRatingRestriction)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultRestrictedByRating, nn::pctl::ConfirmLaunchApplicationPermission(appId, prop));
}

// USKを利用する設定でROMにUSK設定がない場合は制限あり、のチェック
TEST(LaunchCheck, TestConfirmLaunchApplicationPermission_USK_WithNoRatingRestriction)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    // USKの設定は無しにする
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::USK)] = -1;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::USK);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultRestrictedByRating, nn::pctl::ConfirmLaunchApplicationPermission(appId, prop));
}

// ROMに全く設定がない場合は制限あり、のチェック
TEST(LaunchCheck, TestConfirmLaunchApplicationPermission_WithNoAnyRatingRestriction)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    // すべての設定を無しにする
    for (auto& age : prop.ratingAge)
    {
        age = -1;
    }

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultRestrictedByRating, nn::pctl::ConfirmLaunchApplicationPermission(appId, prop));
}

// ROMに別団体の設定のみ場合はそれを用いる、のチェック1
TEST(LaunchCheck, TestConfirmLaunchApplicationPermission_WithOtherRatingRestriction_Restricted)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    // すべての設定を無しにする
    for (auto& age : prop.ratingAge)
    {
        age = -1;
    }
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::ESRB)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultRestrictedByRating, nn::pctl::ConfirmLaunchApplicationPermission(appId, prop));
}

// ROMに別団体の設定のみ場合はそれを用いる、のチェック2
TEST(LaunchCheck, TestConfirmLaunchApplicationPermission_WithOtherRatingRestriction_Pass)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    // すべての設定を無しにする
    for (auto& age : prop.ratingAge)
    {
        age = -1;
    }
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::ESRB)] = 12;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmLaunchApplicationPermission(appId, prop));
}

TEST(LaunchCheck, TestConfirmLaunchApplicationPermission_HasRatingRestriction_TemporaryUnlocked)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0123"));
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmLaunchApplicationPermission(appId, prop));

    nn::pctl::RevertRestrictionTemporaryUnlocked();
}

TEST(LaunchCheck, TestConfirmLaunchApplicationPermission_IgnoreFreeCommunication)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmLaunchApplicationPermission(appId, prop));
}

TEST(LaunchCheck, TestConfirmLaunchApplicationPermission_ExemptApplicationSetting)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::pctl::ExemptApplicationInfo exemptionInfo;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    exemptionInfo.applicationId = appId;
    exemptionInfo.isExempted = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToExemptApplicationListForDebug(appId);
    nn::pctl::UpdateExemptApplicationListForDebug(&exemptionInfo, 1);

    // 年齢（15）による起動制限（18）を無視して起動可能
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmLaunchApplicationPermission(appId, prop));

    nn::pctl::DeleteFromExemptApplicationListForDebug(appId);
}

// ConfirmResumeApplicationPermission

TEST(LaunchCheck, TestConfirmResumeApplicationPermission_NoAppRestriction)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 18;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = 0;
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 0;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));
}

TEST(LaunchCheck, TestConfirmResumeApplicationPermission_NoSettings)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = false;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));
}

TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasRatingRestriction)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = false;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultRestrictedByRating, nn::pctl::ConfirmResumeApplicationPermission(appId, prop));
}

TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasRatingRestriction_TemporaryUnlocked)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = false;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0123"));
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::RevertRestrictionTemporaryUnlocked();
}

TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasFreeCommunicationRestriction_UsingDefaultValue)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultFreeCommunicationRestricted, nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::DeleteFromFreeCommunicationApplicationListForDebug(appId);
}

TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasExemption_UsingDefaultValue)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToExemptApplicationListForDebug(appId);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultFreeCommunicationRestricted, nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::DeleteFromExemptApplicationListForDebug(appId);
}

TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasFreeCommunicationRestriction_UsingAppSpecificValue)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::pctl::FreeCommunicationApplicationInfo appInfo;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    appInfo.applicationId = appId;
    appInfo.isRestricted = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);
    // 個別設定を適用する
    nn::pctl::UpdateFreeCommunicationApplicationList(&appInfo, 1);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultFreeCommunicationRestricted, nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::DeleteFromFreeCommunicationApplicationListForDebug(appId);
}

TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasExemption_UsingAppSpecificValue)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::pctl::ExemptApplicationInfo exemptInfo;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    exemptInfo.applicationId = appId;
    exemptInfo.isExempted = false;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToExemptApplicationListForDebug(appId);
    // 個別設定を適用する
    nn::pctl::UpdateExemptApplicationListForDebug(&exemptInfo, 1);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultFreeCommunicationRestricted, nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::DeleteFromExemptApplicationListForDebug(appId);
}

TEST(LaunchCheck, TestConfirmResumeApplicationPermission_NoFreeCommunicationRestriction_ByDefaultSettings)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::pctl::FreeCommunicationApplicationInfo appInfo;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = false; // 既定値が false

    appInfo.applicationId = appId;
    appInfo.isRestricted = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);
    // 個別設定を適用する
    nn::pctl::UpdateFreeCommunicationApplicationList(&appInfo, 1);
    // 既定値が false なので通過
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::DeleteFromFreeCommunicationApplicationListForDebug(appId);
}

TEST(LaunchCheck, TestConfirmResumeApplicationPermission_NoFreeCommunicationRestriction_ByDefaultSettingsExempt)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::pctl::ExemptApplicationInfo exemptionInfo;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = false; // 既定値が false

    exemptionInfo.applicationId = appId;
    exemptionInfo.isExempted = false;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);
    // 個別設定を適用する
    nn::pctl::UpdateExemptApplicationListForDebug(&exemptionInfo, 1);
    // 既定値が false なので通過
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::DeleteFromExemptApplicationListForDebug(appId);
}

TEST(LaunchCheck, TestConfirmResumeApplicationPermission_NoFreeCommunicationRestriction_UsingAppSpecificValue)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::pctl::FreeCommunicationApplicationInfo appInfo;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    appInfo.applicationId = appId;
    appInfo.isRestricted = false;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);
    // 個別設定を適用する
    nn::pctl::UpdateFreeCommunicationApplicationList(&appInfo, 1);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::DeleteFromFreeCommunicationApplicationListForDebug(appId);
}

TEST(LaunchCheck, TestConfirmResumeApplicationPermission_NoFreeCommunicationRestriction_UsingAppSpecificValueExempt)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::pctl::ExemptApplicationInfo exemptionInfo;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    exemptionInfo.applicationId = appId;
    exemptionInfo.isExempted = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToExemptApplicationListForDebug(appId);
    // 個別設定を適用する
    nn::pctl::UpdateExemptApplicationListForDebug(&exemptionInfo, 1);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::DeleteFromExemptApplicationListForDebug(appId);
}

// タイトル別制限リストと制限対象外リストの共存
TEST(LaunchCheck, TestConfirmResumeApplicationPermission_FreeCommunicationRestriction_Exemption)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::pctl::FreeCommunicationApplicationInfo appInfo;
    nn::pctl::ExemptApplicationInfo exemptionInfo;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    appInfo.applicationId = appId;
    appInfo.isRestricted = false;
    exemptionInfo.applicationId = appId;
    exemptionInfo.isExempted = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);
    nn::pctl::AddToExemptApplicationListForDebug(appId);

    // 制限あり＋制限対象内（共に規定値の場合）
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultFreeCommunicationRestricted, nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    // 制限なし＋制限対象内
    nn::pctl::UpdateFreeCommunicationApplicationList(&appInfo, 1);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    // 制限なし＋制限対象外
    nn::pctl::UpdateExemptApplicationListForDebug(&exemptionInfo, 1);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    // 制限あり＋制限対象外
    appInfo.isRestricted = true;
    nn::pctl::UpdateFreeCommunicationApplicationList(&appInfo, 1);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::DeleteFromFreeCommunicationApplicationListForDebug(appId);
    nn::pctl::DeleteFromExemptApplicationListForDebug(appId);
}

TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasFreeCommunicationRestriction_TemporaryUnlocked)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0123"));
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::RevertRestrictionTemporaryUnlocked();
    nn::pctl::DeleteFromFreeCommunicationApplicationListForDebug(appId);
}

// レーティング制限設定が制限なしからありに変わった場合
TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasRatingRestriction_SettingsChangedToRestrict)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = false;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    customSettings.ratingAge = 15;
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultRestrictedByRating, nn::pctl::ConfirmResumeApplicationPermission(appId, prop));
}

// レーティング制限設定が制限なしからありに変わったが一時解除状態である場合
TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasRatingRestriction_SettingsChangedToRestrict_WithTemporaryUnlocked)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = false;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);

    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0123"));
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    customSettings.ratingAge = 15;
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::RevertRestrictionTemporaryUnlocked();
}

// レーティング制限設定が制限ありからなしに変わった場合
TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasRatingRestriction_SettingsChangedToUnrestrict)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = false;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultRestrictedByRating, nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    customSettings.ratingAge = 0;
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));
}

// UGC利用制限の既定値設定が制限なしからありに変わった場合
TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasFreeCommunicationRestriction_UsingDefaultValue_SettingsChangedToRestrict)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = false;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    // Addしない(本来はないはず？)
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    customSettings.freeCommunicationRestriction = true;
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultFreeCommunicationRestricted, nn::pctl::ConfirmResumeApplicationPermission(appId, prop));
}

// UGC利用制限の既定値設定が制限なしからありに変わったが一時解除状態である場合
TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasFreeCommunicationRestriction_UsingDefaultValue_SettingsChangedToRestrict_WithTemporaryUnlocked)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = false;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    // Addしない(本来はないはず？)

    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0123"));
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    customSettings.freeCommunicationRestriction = true;
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::RevertRestrictionTemporaryUnlocked();
}

// UGC利用制限の既定値設定が制限ありからなしに変わった場合
TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasFreeCommunicationRestriction_UsingDefaultValue_SettingsChangedToUnrestrict)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    // Addしない(本来はないはず？)
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultFreeCommunicationRestricted, nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    customSettings.freeCommunicationRestriction = false;
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));
}

// UGC利用制限のアプリ別設定が制限なしからありに変わった場合
TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasFreeCommunicationRestriction_UsingAppSpecificValue_SettingsChangedToRestrict)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::pctl::FreeCommunicationApplicationInfo appInfo;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    appInfo.applicationId = appId;
    appInfo.isRestricted = false;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);
    nn::pctl::UpdateFreeCommunicationApplicationList(&appInfo, 1);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    appInfo.isRestricted = true;
    nn::pctl::UpdateFreeCommunicationApplicationList(&appInfo, 1);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultFreeCommunicationRestricted, nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::DeleteFromFreeCommunicationApplicationListForDebug(appId);
}

// UGC利用制限のアプリ別設定が制限なしからありに変わったが一時解除状態である場合
TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasFreeCommunicationRestriction_UsingAppSpecificValue_SettingsChangedToRestrict_WithTemporaryUnlocked)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::pctl::FreeCommunicationApplicationInfo appInfo;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    appInfo.applicationId = appId;
    appInfo.isRestricted = false;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);
    nn::pctl::UpdateFreeCommunicationApplicationList(&appInfo, 1);

    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0123"));
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    appInfo.isRestricted = true;
    nn::pctl::UpdateFreeCommunicationApplicationList(&appInfo, 1);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::RevertRestrictionTemporaryUnlocked();
    nn::pctl::DeleteFromFreeCommunicationApplicationListForDebug(appId);
}

// UGC利用制限のアプリ別設定が制限ありからなしに変わった場合
TEST(LaunchCheck, TestConfirmResumeApplicationPermission_HasFreeCommunicationRestriction_UsingAppSpecificValue_SettingsChangedToUnrestrict)
{
    nn::pctl::SafetyLevelSettings customSettings;
    nn::pctl::FreeCommunicationApplicationInfo appInfo;
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    appInfo.applicationId = appId;
    appInfo.isRestricted = true;

    // 通過状態にする
    nn::pctl::SetPinCode(nullptr);
    EXPECT_TRUE(nn::pctl::CheckFreeCommunicationPermission(false));

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);
    nn::pctl::UpdateFreeCommunicationApplicationList(&appInfo, 1);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultFreeCommunicationRestricted, nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    appInfo.isRestricted = false;
    nn::pctl::UpdateFreeCommunicationApplicationList(&appInfo, 1);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmResumeApplicationPermission(appId, prop));

    nn::pctl::DeleteFromFreeCommunicationApplicationListForDebug(appId);
}

// ConfirmPlayableApplicationVideo

TEST(LaunchCheck, TestConfirmPlayableApplicationVideo_NoAppRestriction)
{
    nn::pctl::SafetyLevelSettings customSettings;
    int8_t ratingAge[32];
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 18;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 0;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmPlayableApplicationVideo(appId, ratingAge));
}

TEST(LaunchCheck, TestConfirmPlayableApplicationVideo_NoSettings)
{
    nn::pctl::SafetyLevelSettings customSettings;
    int8_t ratingAge[32];
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = false;

    ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmPlayableApplicationVideo(appId, ratingAge));
}

TEST(LaunchCheck, TestConfirmPlayableApplicationVideo_HasRatingRestriction)
{
    nn::pctl::SafetyLevelSettings customSettings;
    int8_t ratingAge[32];
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultPlayingApplicationVideoRestricted, nn::pctl::ConfirmPlayableApplicationVideo(appId, ratingAge));
}

// USKを利用する設定でROMにUSK設定がない場合は制限あり、のチェック
TEST(LaunchCheck, TestConfirmPlayableApplicationVideo_USK_WithNoRatingRestriction)
{
    nn::pctl::SafetyLevelSettings customSettings;
    int8_t ratingAge[32];
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    // USKの設定は無しにする
    ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;
    ratingAge[static_cast<int>(nn::ns::RatingOrganization::USK)] = -1;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::USK);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultPlayingApplicationVideoRestricted, nn::pctl::ConfirmPlayableApplicationVideo(appId, ratingAge));
}

// ROMに全く設定がない場合は制限あり、のチェック
TEST(LaunchCheck, TestConfirmPlayableApplicationVideo_WithNoAnyRatingRestriction)
{
    nn::pctl::SafetyLevelSettings customSettings;
    int8_t ratingAge[32];
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    // すべての設定を無しにする
    for (auto& age : ratingAge)
    {
        age = -1;
    }

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultPlayingApplicationVideoRestricted, nn::pctl::ConfirmPlayableApplicationVideo(appId, ratingAge));
}

// ROMに別団体の設定のみ場合はそれを用いる、のチェック1
TEST(LaunchCheck, TestConfirmPlayableApplicationVideo_WithOtherRatingRestriction_Restricted)
{
    nn::pctl::SafetyLevelSettings customSettings;
    int8_t ratingAge[32];
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    // すべての設定を無しにする
    for (auto& age : ratingAge)
    {
        age = -1;
    }
    ratingAge[static_cast<int>(nn::ns::RatingOrganization::ESRB)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultPlayingApplicationVideoRestricted, nn::pctl::ConfirmPlayableApplicationVideo(appId, ratingAge));
}

// ROMに別団体の設定のみ場合はそれを用いる、のチェック2
TEST(LaunchCheck, TestConfirmPlayableApplicationVideo_WithOtherRatingRestriction_Pass)
{
    nn::pctl::SafetyLevelSettings customSettings;
    int8_t ratingAge[32];
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    // すべての設定を無しにする
    for (auto& age : ratingAge)
    {
        age = -1;
    }
    ratingAge[static_cast<int>(nn::ns::RatingOrganization::ESRB)] = 12;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmPlayableApplicationVideo(appId, ratingAge));
}

TEST(LaunchCheck, TestConfirmPlayableApplicationVideo_HasRatingRestriction_TemporaryUnlocked)
{
    nn::pctl::SafetyLevelSettings customSettings;
    int8_t ratingAge[32];
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily("0123"));
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmPlayableApplicationVideo(appId, ratingAge));

    nn::pctl::RevertRestrictionTemporaryUnlocked();
}

TEST(LaunchCheck, TestConfirmPlayableApplicationVideo_IgnoreFreeCommunication)
{
    nn::pctl::SafetyLevelSettings customSettings;
    int8_t ratingAge[32];
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmPlayableApplicationVideo(appId, ratingAge));
}

// 制限対象外リスト
TEST(LaunchCheck, TestConfirmPlayableApplicationVideo_HasRatingRestrictionWithExemptionList)
{
    nn::pctl::SafetyLevelSettings customSettings;
    int8_t ratingAge[32];
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    customSettings.ratingAge = 15;
    customSettings.snsPostRestriction = true;
    customSettings.freeCommunicationRestriction = true;

    ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::SetPinCode("0123");
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultPlayingApplicationVideoRestricted, nn::pctl::ConfirmPlayableApplicationVideo(appId, ratingAge));

    // 制限対象外リストに登録後は動画再生も可能
    nn::pctl::ExemptApplicationInfo exemptionInfo;
    exemptionInfo.applicationId = appId;
    exemptionInfo.isExempted = true;

    nn::pctl::AddToExemptApplicationListForDebug(appId);
    nn::pctl::UpdateExemptApplicationListForDebug(&exemptionInfo, 1);

    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmPlayableApplicationVideo(appId, ratingAge));
}
