﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/pctl/pctl_Api.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/pctl_ApiForDebug.h>

TEST(FreeCommunicationApplicationList, Test_Empty)
{
    nn::pctl::ClearFreeCommunicationApplicationListForDebug();

    nn::pctl::FreeCommunicationApplicationInfo appInfo;
    EXPECT_EQ(0, nn::pctl::GetFreeCommunicationApplicationListCount());
    EXPECT_EQ(0, nn::pctl::GetFreeCommunicationApplicationList(&appInfo, 0, 1));
}

TEST(FreeCommunicationApplicationList, Test_AddOne_Restricted)
{
    nn::pctl::ClearFreeCommunicationApplicationListForDebug();

    const nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = true; // 既定値

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::FreeCommunicationApplicationInfo appInfo;
    nn::pctl::SetPinCode("0123");
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);

    EXPECT_EQ(1, nn::pctl::GetFreeCommunicationApplicationListCount());
    EXPECT_EQ(1, nn::pctl::GetFreeCommunicationApplicationList(&appInfo, 0, 1));
    EXPECT_EQ(appId, appInfo.applicationId);
    EXPECT_EQ(true, appInfo.isRestricted);
}

TEST(FreeCommunicationApplicationList, Test_AddOne_RestrictedEvenIfDefaultNotRestricted)
{
    nn::pctl::ClearFreeCommunicationApplicationListForDebug();

    const nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = false; // 既定値

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::FreeCommunicationApplicationInfo appInfo;
    nn::pctl::SetPinCode("0123");
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);

    EXPECT_EQ(1, nn::pctl::GetFreeCommunicationApplicationListCount());
    EXPECT_EQ(1, nn::pctl::GetFreeCommunicationApplicationList(&appInfo, 0, 1));
    EXPECT_EQ(appId, appInfo.applicationId);
    EXPECT_EQ(true, appInfo.isRestricted); // customSettings.freeCommunicationRestriction は無関係
}

// 制限のないタイトルは追加されないテスト
TEST(FreeCommunicationApplicationList, Test_AddOne_NotAdded)
{
    nn::pctl::ClearFreeCommunicationApplicationListForDebug();

    const nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = true; // 既定値

    prop.parentalControlFlag = 0; // フラグがないと追加されないはず
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::FreeCommunicationApplicationInfo appInfo;
    nn::pctl::SetPinCode("0123");
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);

    EXPECT_EQ(0, nn::pctl::GetFreeCommunicationApplicationListCount());
    EXPECT_EQ(0, nn::pctl::GetFreeCommunicationApplicationList(&appInfo, 0, 1));
}

TEST(FreeCommunicationApplicationList, Test_UpdateOne_Restricted)
{
    nn::pctl::ClearFreeCommunicationApplicationListForDebug();

    const nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = false; // 既定値

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::FreeCommunicationApplicationInfo appInfo;
    nn::pctl::SetPinCode("0123");
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);

    appInfo.applicationId = appId;
    appInfo.isRestricted = true;
    nn::pctl::UpdateFreeCommunicationApplicationList(&appInfo, 1);

    std::memset(&appInfo, 0, sizeof(appInfo));
    EXPECT_EQ(1, nn::pctl::GetFreeCommunicationApplicationListCount());
    EXPECT_EQ(1, nn::pctl::GetFreeCommunicationApplicationList(&appInfo, 0, 1));
    EXPECT_EQ(appId, appInfo.applicationId);
    EXPECT_EQ(true, appInfo.isRestricted);
}

TEST(FreeCommunicationApplicationList, Test_UpdateOne_NotRestricted)
{
    nn::pctl::ClearFreeCommunicationApplicationListForDebug();

    const nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = true; // 既定値

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::FreeCommunicationApplicationInfo appInfo;
    nn::pctl::SetPinCode("0123");
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);

    appInfo.applicationId = appId;
    appInfo.isRestricted = false;
    nn::pctl::UpdateFreeCommunicationApplicationList(&appInfo, 1);

    std::memset(&appInfo, 0, sizeof(appInfo));
    EXPECT_EQ(1, nn::pctl::GetFreeCommunicationApplicationListCount());
    EXPECT_EQ(1, nn::pctl::GetFreeCommunicationApplicationList(&appInfo, 0, 1));
    EXPECT_EQ(appId, appInfo.applicationId);
    EXPECT_EQ(false, appInfo.isRestricted);
}

TEST(FreeCommunicationApplicationList, Test_DeleteSettings_ResetToRestricted)
{
    nn::pctl::ClearFreeCommunicationApplicationListForDebug();

    const nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId
    nn::pctl::SafetyLevelSettings customSettings;
    nn::ns::ApplicationControlProperty prop;

    customSettings.ratingAge = 0;
    customSettings.snsPostRestriction = false;
    customSettings.freeCommunicationRestriction = true; // 既定値

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    nn::pctl::FreeCommunicationApplicationInfo appInfo;
    nn::pctl::SetPinCode("0123");
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);
    nn::pctl::AddToFreeCommunicationApplicationList(appId, prop);

    appInfo.applicationId = appId;
    appInfo.isRestricted = false;
    nn::pctl::UpdateFreeCommunicationApplicationList(&appInfo, 1);

    nn::pctl::DeleteSettings();

    std::memset(&appInfo, 0, sizeof(appInfo));
    EXPECT_EQ(1, nn::pctl::GetFreeCommunicationApplicationListCount());
    EXPECT_EQ(1, nn::pctl::GetFreeCommunicationApplicationList(&appInfo, 0, 1));
    EXPECT_EQ(appId, appInfo.applicationId);
    EXPECT_EQ(true, appInfo.isRestricted);
}
