﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nnt.h>

#include <nn/pcie/pcie.h>

nn::pcie::ClassDriverHandle g_ClassDriverHandle = nn::pcie::InvalidClassDriverHandle;
nn::os::SystemEventType g_RegistrationEvent;
nn::os::SystemEventType g_PmEvent;
nn::os::SystemEventType g_IrqEvent;
nn::pcie::FunctionState g_FuncStates[nn::pcie::MaxEnumeratedDeviceFunctions];
nn::pcie::FunctionState* g_pFuncUnderTest = NULL;
uint8_t NN_ALIGNAS(nn::pcie::MinimumDmaAddressAlignment) g_BigDmaBuffer[0x100000];


void QueryFunctionTest(bool reset)
{
    int32_t numReturned = 0;
    if(reset){g_pFuncUnderTest=NULL;}
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::QueryFunctions(g_FuncStates, &numReturned, g_ClassDriverHandle,
                                                       sizeof(g_FuncStates)));
    for(int32_t i = 0; i < numReturned; i++)
    {
        nn::pcie::FunctionState *pF = g_FuncStates + i;
        NN_LOG("Query out %d\n",i);
        NN_LOG("  functionHandle = 0x%llx\n", static_cast<uint64_t>(pF->functionHandle));
        NN_LOG("  vendorId       = 0x%x\n", pF->vendorId);
        NN_LOG("  deviceId       = 0x%x\n", pF->deviceId);
        if(pF->isAcquired)
        {
            NN_LOG("There is a problem. Registered QueryFunctions() API should never return info about acquired function!\n");
            ADD_FAILURE();
            return;
        }
        if(g_pFuncUnderTest==NULL){g_pFuncUnderTest = pF;}
    }
}

TEST( PcieServer_Full, Initialize)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::Initialize());
}

TEST( PcieServer_Full, RegisterClassDriver)
{
    // accept any device for this test
    nn::pcie::ClassDriverConfig cfg = {0};
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::RegisterClassDriver(&g_ClassDriverHandle, &g_RegistrationEvent, &cfg,
                                                            nn::os::EventClearMode_ManualClear));
}

TEST( PcieServer_Full, WaitForDeviceFunction)
{
    NN_LOG("Waiting for device function availability...\n");
    nn::os::WaitSystemEvent(&g_RegistrationEvent);
    NN_LOG("GOT EVENT.\n");
    nn::os::ClearSystemEvent(&g_RegistrationEvent);
}

TEST( PcieServer_Full, QueryFunctions)
{
    QueryFunctionTest(true);
}

TEST( PcieServer_Full, BarProfile)
{
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}
    for(uint8_t bar=0; bar < nn::pcie::StandardConfigMaxBaseAddresses; bar++)
    {
        nn::pcie::BarProfile barProfile;
        if(nn::pcie::GetBarProfile(&barProfile, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, bar).IsSuccess())
        {
            NN_LOG("BAR %d\n",bar);
            NN_LOG("  flags = 0x%x\n", barProfile.flags);
            NN_LOG("  size  = 0x%x\n", barProfile.size);
        }
    }
}

TEST( PcieServer_Full, AcquireFunction)
{
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}
    NN_LOG("Trying to acquire functionHandle 0x%llx...\n",g_pFuncUnderTest->functionHandle);
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::AcquireFunction(&g_PmEvent, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                        nn::os::EventClearMode_ManualClear));
}

TEST( PcieServer_Full, ResetFunction)
{
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}
    NN_LOG("Trying to reset functionHandle 0x%llx...\n",g_pFuncUnderTest->functionHandle);

    for(int i=0; i < 500; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::pcie::ResetFunction(g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                  nn::pcie::FunctionResetOption_DoExternalReset));
        NNT_EXPECT_RESULT_SUCCESS(nn::pcie::ReleaseFunction(&g_PmEvent, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle));
        QueryFunctionTest(true);
        NNT_EXPECT_RESULT_SUCCESS(nn::pcie::AcquireFunction(&g_PmEvent, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                            nn::os::EventClearMode_ManualClear));
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::ReleaseFunction(&g_PmEvent, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle));
}

TEST( PcieServer_Full, ReAcquire)
{
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}
    QueryFunctionTest(true);
    NN_LOG("Trying to acquire functionHandle 0x%llx...\n",g_pFuncUnderTest->functionHandle);
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::AcquireFunction(&g_PmEvent, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                        nn::os::EventClearMode_ManualClear));
}

TEST( PcieServer_Full, ExtendedCapabilityId_DeviceSerialNumber)
{
    size_t offset = 0;
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}
    if (nn::pcie::FindExtendedCapability(&offset, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                         nn::pcie::ExtendedCapabilityId_DeviceSerialNumber).IsSuccess())
    {
        uint8_t buf[nn::pcie::ExtendedCapabilityDeviceSerialNumberSize] = { 0 };
        int32_t i;
        offset += 4;
        for (i = 0; i < nn::pcie::ExtendedCapabilityDeviceSerialNumberSize; i++)
        {
            if(!nn::pcie::ReadConfig8(&buf[i], g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, offset + i).IsSuccess())
            {
                break;
            }
        }
        if(i == nn::pcie::ExtendedCapabilityDeviceSerialNumberSize)
        {
            NN_LOG("Device Serial Number: %02X%02X%02X%02X%02X%02X%02X%02X\n",
                   buf[7], buf[6], buf[5], buf[4],
                   buf[3], buf[2], buf[1], buf[0]);
        }
    }
    else
    {
        NN_LOG("ExtendedCapabilityId_DeviceSerialNumber unsupported\n");
    }
}

TEST( PcieServer_Full, GetFunctionState)
{
    nn::pcie::FunctionState localFuncState;
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::GetFunctionState(&localFuncState, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle));
    if((0x10ee != localFuncState.vendorId) &&  /* Our HostBridge */
       (0x14e4 != localFuncState.vendorId))    /* Broadcom device */
    {
        ADD_FAILURE();
    }
}

TEST( PcieServer_Full, SingleDmaMap)
{
    nn::Result result = nn::ResultSuccess();
    nn::pcie::BusAddress busAddr = 0;
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}
    NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::MapDma(&busAddr, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                        nn::pcie::DmaDirection_BiDirectional,
                                                        g_BigDmaBuffer, nn::pcie::MinimumDmaAddressAlignment));
    NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::UnmapDma(g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, busAddr));
}

TEST(PcieServer_Full, DmaMapsWithBusAddressessCheck)
{
    nn::Result result = nn::ResultSuccess();
    static nn::pcie::BusAddress busAddrs[sizeof(g_BigDmaBuffer) / nn::pcie::MinimumDmaAddressAlignment] = {0LL};
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}

    // Map an area using most number of calls, once per minimum alignment
    for (size_t offset = 0, index = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
    {
        NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::MapDma(&busAddrs[index], g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                            nn::pcie::DmaDirection_BiDirectional,
                                                            g_BigDmaBuffer + offset, nn::pcie::MinimumDmaAddressAlignment));
        if (!result.IsSuccess())
        {
            NN_SDK_LOG("Failed mapping at offset 0x%x\n", offset);
            ADD_FAILURE();
            return;
        }
        index++;
    }

    // Check mapping by asking for bus address throughout mapped range
    for (size_t offset = 0; offset < sizeof(g_BigDmaBuffer); )
    {
        nn::pcie::BusAddress reportedBusAddress = 0;
        nn::pcie::BusAddress expectedBusAddress = busAddrs[0] + offset;
        NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::GetDmaBusAddress(&reportedBusAddress, g_ClassDriverHandle,
                                                                      g_pFuncUnderTest->functionHandle,
                                                                      g_BigDmaBuffer + offset, 1));
        if (expectedBusAddress != reportedBusAddress)
        {
            NN_SDK_LOG("Expected bus address 0x%llx, but PCI driver provided 0x%llx\n",
                       expectedBusAddress, reportedBusAddress);
            ADD_FAILURE();
        }

        if (!result.IsSuccess())
        {
            NN_SDK_LOG("Failed bus address check at offset 0x%x\n", offset);
            ADD_FAILURE();
            return;
        }

        // vary offset step, so the test doesn't take forever
        offset += ((offset % 128) + 1);
    }

    // Now unmap by bus address
    for (size_t offset = 0, index = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
    {
        NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::UnmapDma(g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, busAddrs[index]));
        if (!result.IsSuccess())
        {
            NN_SDK_LOG("Failed mapping at offset 0x%x\n", offset);
            ADD_FAILURE();
            return;
        }
        index++;
    }

    // Map it again
    for (size_t offset = 0, index = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
    {
        NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::MapDma(&busAddrs[index], g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                            nn::pcie::DmaDirection_BiDirectional,
                                                            g_BigDmaBuffer + offset, nn::pcie::MinimumDmaAddressAlignment));
        if (!result.IsSuccess())
        {
            NN_SDK_LOG("Failed mapping at offset 0x%x\n", offset);
            ADD_FAILURE();
            return;
        }
        index++;
    }

    // Now unmap by process virtual address
    for (size_t offset = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
    {
        NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::UnmapDma(g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                              g_BigDmaBuffer + offset));
        if (!result.IsSuccess())
        {
            NN_SDK_LOG("Failed mapping at offset 0x%x\n", offset);
            ADD_FAILURE();
            return;
        }
    }

    SUCCEED();
}

TEST( PcieServer_Full, GetDmaBusRange)
{
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}
    nn::Result result = nn::ResultSuccess();
    nn::pcie::BusAddress base = 0;
    nn::pcie::BusAddress size = 0;
    NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::GetDmaBusAddressRange(&base, &size, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle));
    NN_SDK_LOG("%s: base=0x%08x, size=0x%08x\n", __FUNCTION__, base, size);
}

TEST(PcieServer_Full, DmaUnmapSloppy)
{
    nn::Result result = nn::ResultSuccess();
    static nn::pcie::BusAddress busAddrs[sizeof(g_BigDmaBuffer) / nn::pcie::MinimumDmaAddressAlignment] = {0LL};

    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}

    // Map an area using most number of calls, once per minimum alignment
    for (size_t offset = 0, index = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
    {
        NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::MapDma(&busAddrs[index], g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                            nn::pcie::DmaDirection_BiDirectional,
                                                            g_BigDmaBuffer + offset, nn::pcie::MinimumDmaAddressAlignment));
        if (!result.IsSuccess())
        {
            NN_SDK_LOG("Failed mapping at offset 0x%x\n", offset);
            ADD_FAILURE();
            return;
        }
        index++;
    }

    // Now unmap by bus address, but be sloppy about base address. Driver is smart enough to find corresponding base
    // and free the original mapping.
    for (size_t offset = 0, index = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
    {
        NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::UnmapDma(g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                              busAddrs[index] + (index % nn::pcie::MinimumDmaAddressAlignment)));
        if (!result.IsSuccess())
        {
            NN_SDK_LOG("Failed mapping at offset 0x%x\n", offset);
            ADD_FAILURE();
            return;
        }
        index++;
    }

    // Setup the same mapping again
    for (size_t offset = 0, index = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
    {
        NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::MapDma(&busAddrs[index], g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                            nn::pcie::DmaDirection_BiDirectional,
                                                            g_BigDmaBuffer + offset, nn::pcie::MinimumDmaAddressAlignment));
        if (!result.IsSuccess())
        {
            NN_SDK_LOG("Failed mapping at offset 0x%x\n", offset);
            ADD_FAILURE();
            return;
        }
        index++;
    }

    // Same as above, but do it by process virtual address instead
    for (size_t offset = 0, index = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
    {
        NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::UnmapDma(g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                              g_BigDmaBuffer + offset + (index % nn::pcie::MinimumDmaAddressAlignment)));
        if (!result.IsSuccess())
        {
            NN_SDK_LOG("Failed mapping at offset 0x%x\n", offset);
            ADD_FAILURE();
            return;
        }
        index++;
    }
}

TEST(PcieServer_Full, DmaMapUnsuccessfulSimple)
{
    nn::pcie::BusAddress busAddr;

    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}

    // Misaligned base address failure
    NNT_EXPECT_RESULT_FAILURE(nn::pcie::ResultInvalidAlignment,
                              nn::pcie::MapDma(&busAddr, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                               nn::pcie::DmaDirection_BiDirectional,
                                               g_BigDmaBuffer + 64, nn::pcie::MinimumDmaAddressAlignment));

    // Unpadded size failure
    NNT_EXPECT_RESULT_FAILURE(nn::pcie::ResultInvalidSizePadding,
                              nn::pcie::MapDma(&busAddr, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                               nn::pcie::DmaDirection_BiDirectional,
                                               g_BigDmaBuffer, nn::pcie::MinimumDmaAddressAlignment + 64));
}

TEST(PcieServer_Full, DmaMapUnsuccessfulComplex)
{
    nn::Result result = nn::ResultSuccess();
    static nn::pcie::BusAddress busAddrs[sizeof(g_BigDmaBuffer) / nn::pcie::MinimumDmaAddressAlignment] = {0LL};

    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}

    // Map an area using most number of calls, once per minimum alignment
    for (size_t offset = 0, index = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
    {
        NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::MapDma(&busAddrs[index], g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                            nn::pcie::DmaDirection_BiDirectional,
                                                            g_BigDmaBuffer + offset, nn::pcie::MinimumDmaAddressAlignment));
        if (!result.IsSuccess())
        {
            NN_SDK_LOG("Failed mapping at offset 0x%x\n", offset);
            ADD_FAILURE();
            return;
        }
        index++;
    }

    // Try duplicate mapping, verify proper error code
    for (size_t offset = 0, index = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
    {
        NNT_EXPECT_RESULT_FAILURE(nn::pcie::ResultOverlappedDmaMapping,
                                  nn::pcie::MapDma(NULL, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                   nn::pcie::DmaDirection_BiDirectional,
                                                   g_BigDmaBuffer + offset, nn::pcie::MinimumDmaAddressAlignment));
        index++;
    }

    // Now unmap by bus address
    for (size_t offset = 0, index = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
    {
        NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::UnmapDma(g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                              busAddrs[index]));
        if (!result.IsSuccess())
        {
            NN_SDK_LOG("Failed mapping at offset 0x%x\n", offset);
            ADD_FAILURE();
            return;
        }
        index++;
    }

    // Try to unmap that which no longer exists, by virtual address
    for (size_t offset = 0, index = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
    {
        NNT_EXPECT_RESULT_FAILURE(nn::pcie::ResultNonExistentDmaMapping,
                                  nn::pcie::UnmapDma(g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                     g_BigDmaBuffer + offset));
        index++;
    }

    // Try to unmap that which no longer exists, by bus address
    for (size_t offset = 0, index = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
    {
        NNT_EXPECT_RESULT_FAILURE(nn::pcie::ResultNonExistentDmaMapping,
                                  nn::pcie::UnmapDma(g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                     busAddrs[index]));
        index++;
    }
}

TEST(PcieServer_Full, AbandondedDmaMaps)
{
    nn::Result result = nn::ResultSuccess();
    static nn::pcie::BusAddress busAddrs[sizeof(g_BigDmaBuffer) / nn::pcie::MinimumDmaAddressAlignment] = {0LL};
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}

    // Do some mapping, then release the function. Maps should be removed automatically. If they weren't, we wouldn't
    // be able to re-map the same memory again in this loop.
    for(int32_t i=0; i < 500; i++)
    {
        // Map an area using most number of calls, once per minimum alignment
        for (size_t offset = 0, index = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
        {
            NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::MapDma(&busAddrs[index], g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                                nn::pcie::DmaDirection_BiDirectional,
                                                                g_BigDmaBuffer + offset, nn::pcie::MinimumDmaAddressAlignment));
            if (!result.IsSuccess())
            {
                NN_SDK_LOG("Failed mapping at offset 0x%x\n", offset);
                ADD_FAILURE();
                return;
            }
            index++;
        }
        NNT_EXPECT_RESULT_SUCCESS(result=nn::pcie::ReleaseFunction(&g_PmEvent, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle));
        if(result.IsFailure()) return;
        NNT_EXPECT_RESULT_SUCCESS(result=nn::pcie::AcquireFunction(&g_PmEvent, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                                   nn::os::EventClearMode_ManualClear));
        if(result.IsFailure()) return;
    }

    // Same as above, but this time just unregister the class driver. Acquired functions should be released automatically
    for(int32_t i=0; i < 2; i++)
    {
        // Map an area using most number of calls, once per minimum alignment
        for (size_t offset = 0, index = 0; offset < sizeof(g_BigDmaBuffer); offset += nn::pcie::MinimumDmaAddressAlignment)
        {
            NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::MapDma(&busAddrs[index], g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                                nn::pcie::DmaDirection_BiDirectional,
                                                                g_BigDmaBuffer + offset, nn::pcie::MinimumDmaAddressAlignment));
            if (!result.IsSuccess())
            {
                NN_SDK_LOG("Failed mapping at offset 0x%x\n", offset);
                ADD_FAILURE();
                return;
            }
            index++;
        }

        // Unregister
        NNT_EXPECT_RESULT_SUCCESS(result=nn::pcie::UnregisterClassDriver(&g_RegistrationEvent, g_ClassDriverHandle));
        if(result.IsFailure()) return;

        // Re-Register
        nn::pcie::ClassDriverConfig cfg = {0};
        NNT_EXPECT_RESULT_SUCCESS(nn::pcie::RegisterClassDriver(&g_ClassDriverHandle, &g_RegistrationEvent, &cfg,
                                                                nn::os::EventClearMode_ManualClear));

        // Re-query, but for sure function still there
        QueryFunctionTest(true);

        // Re-acquire
        NNT_EXPECT_RESULT_SUCCESS(nn::pcie::AcquireFunction(&g_PmEvent, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                            nn::os::EventClearMode_ManualClear));
    }
}

TEST(PcieServer_Full, Irq)
{
    nn::Result result = nn::ResultSuccess();
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}

    // Acquire
    NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::AcquireIrq(&g_IrqEvent,
                                                            g_ClassDriverHandle,
                                                            g_pFuncUnderTest->functionHandle,
                                                            nn::os::EventClearMode_ManualClear,
                                                            nn::pcie::IrqType_Msi));
    if(result.IsFailure()) return;

    // Try negative AcquireIrq scenario
    NNT_EXPECT_RESULT_FAILURE(nn::pcie::ResultAlreadyAcquired,
                              nn::pcie::AcquireIrq(&g_IrqEvent,
                                                   g_ClassDriverHandle,
                                                   g_pFuncUnderTest->functionHandle,
                                                   nn::os::EventClearMode_ManualClear,
                                                   nn::pcie::IrqType_Msi));

    #if 0
    //Try to enable all vectors
    NN_LOG("For %d IRQ vectors...\n",g_pFuncUnderTest->maxNumMsiVectors);
    for(int32_t i=0; i < g_pFuncUnderTest->maxNumMsiVectors; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::SetIrqEnable(g_ClassDriverHandle,
                                                                  g_pFuncUnderTest->functionHandle,
                                                                  i,
                                                                  true));
        if(result.IsFailure()) return;
    }

    // Try negative SetIrqEnable scenario
    NNT_EXPECT_RESULT_FAILURE(nn::pcie::ResultInvalidArgument,
                              nn::pcie::SetIrqEnable(g_ClassDriverHandle,
                                                     g_pFuncUnderTest->functionHandle,
                                                     g_pFuncUnderTest->maxNumMsiVectors + 1,
                                                     true));
    #else
    NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::SetIrqEnable(g_ClassDriverHandle,
                                                              g_pFuncUnderTest->functionHandle,
                                                              0,
                                                              true));
    if(result.IsFailure()) return;

    // Try negative SetIrqEnable scenario
    NNT_EXPECT_RESULT_FAILURE(nn::pcie::ResultInvalidArgument,
                              nn::pcie::SetIrqEnable(g_ClassDriverHandle,
                                                     g_pFuncUnderTest->functionHandle,
                                                     1,
                                                     true));
    #endif

    // Release, which will also disable
    NNT_EXPECT_RESULT_SUCCESS(result = nn::pcie::ReleaseIrq(&g_IrqEvent,
                                                            g_ClassDriverHandle,
                                                            g_pFuncUnderTest->functionHandle));
    if(result.IsFailure()) return;

    // Try negative ReleaseIrq scenario
    NNT_EXPECT_RESULT_FAILURE(nn::pcie::ResultIrqNotAcquired,
                              nn::pcie::ReleaseIrq(&g_IrqEvent,
                                                   g_ClassDriverHandle,
                                                   g_pFuncUnderTest->functionHandle));
}

TEST(PcieServer_Full, BasicConfigAccess)
{
    uint32_t data32 = 0;
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}

    // Read known-good
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::ReadConfig32(&data32, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, nn::pcie::StandardConfigOffset_Command));

    // Write known-good
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::WriteConfig32(data32, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, nn::pcie::StandardConfigOffset_Command));

    // Read known-bad
    NNT_EXPECT_RESULT_FAILURE(nn::pcie::ResultInvalidRegisterOffset,
                              nn::pcie::ReadConfig32(&data32, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, 4096));

    // Write known-bad
    NNT_EXPECT_RESULT_FAILURE(nn::pcie::ResultInvalidRegisterOffset,
                              nn::pcie::WriteConfig32(data32, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, 4096));
}


TEST( PcieServer_Full, ReleaseFunction)
{
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::ReleaseFunction(&g_PmEvent, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle));
}

TEST( PcieServer_Full, UnRegisterClassDriver)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::UnregisterClassDriver(&g_RegistrationEvent, g_ClassDriverHandle));
}

TEST( PcieServer_Full, Finalize)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::Finalize());
}
