﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

// テスト対象
#include <nn/ovln/ovln_Receiver.h>
#include <nn/ovln/ovln_Sender.h>
#include <nn/ovln/ovln_ForDevelop.h>

#include <nn/nn_Common.h>
#include "testOvln_Utility.h"

using namespace nnt::ovln;

// 実装
#include <utility>
#include <nn/sf/sf_DefaultAllocationPolicy.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/ovln/ovln_Result.h>
#include <nn/ovln/ovln_ResultPrivate.h>
#include <cstring>

#include <thread>
#include <nn/os/os_Thread.h>

namespace {

inline nn::ovln::Message MakeTestMessage(int seed) NN_NOEXCEPT
{
    nn::ovln::Message ret = {};
    ret.tag = seed;
    ret.dataSize = seed;
    std::memset(&ret.data, seed, ret.dataSize);
    return ret;
}

const nn::ovln::Message Message1 = MakeTestMessage(1);
const nn::ovln::Message Message2 = MakeTestMessage(2);
const nn::ovln::Message Message3 = MakeTestMessage(3);
const nn::ovln::Message Message4 = MakeTestMessage(4);
const nn::ovln::Message Message10 = MakeTestMessage(10);
const nn::ovln::Message Message100 = MakeTestMessage(100);

inline bool Equals(const nn::ovln::Message& lhs, const nn::ovln::Message& rhs) NN_NOEXCEPT
{
    return true
        && lhs.tag == rhs.tag
        && lhs.dataSize == rhs.dataSize
        && std::memcmp(&lhs.data, &rhs.data, lhs.dataSize) == 0;
}

#define NNT_OVLN_ASSERT_RECEIVED_MESSAGE(x, message) \
    do \
    { \
        ASSERT_TRUE((x).HasMessageToReceive()); \
        ::nn::ovln::Message receivedMessage; \
        ASSERT_TRUE((x).TryReceive(&receivedMessage)); \
        ASSERT_TRUE(Equals((message), receivedMessage)); \
    } while (NN_STATIC_CONDITION(false))

#define NNT_OVLN_ASSERT_CANNOT_RECEIVED_MESSAGE(x) \
    do \
    { \
        ASSERT_TRUE(!(x).HasMessageToReceive()); \
        ::nn::ovln::Message receivedMessage; \
        ASSERT_TRUE(!(x).TryReceive(&receivedMessage)); \
    } while (NN_STATIC_CONDITION(false))

TEST(ovln_SenderReceiver, InitializeForDevelop)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_SenderReceiver, InitializeReceiver)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    nn::ovln::InitializeReceiverLibrary();
    nn::ovln::Receiver x1;
    NNT_OVLN_ASSERT_RESULT_SUCCESS(x1.Initialize());
    x1.Finalize();
    NNT_OVLN_ASSERT_RESULT_SUCCESS(x1.Initialize());
    x1.Finalize();
    nn::ovln::Receiver x2;
    nn::ovln::FinalizeReceicerLibrary();
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_SenderReceiver, Receiver_Initialize)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    nn::ovln::InitializeReceiverLibrary();
    {
        nn::ovln::Receiver x1;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(x1.Initialize());
        nn::ovln::Receiver x2;
        nn::ovln::Receiver x3;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(x2.Initialize());
        NNT_OVLN_ASSERT_RESULT_SUCCESS(x3.Initialize());
    }
    nn::ovln::FinalizeReceicerLibrary();
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_SenderReceiver, Receiver_MoveAndSwap)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    nn::ovln::InitializeReceiverLibrary();
    {
        nn::ovln::Receiver x1;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(x1.Initialize());
        EXPECT_TRUE(x1.IsValid());
        EXPECT_TRUE(!x1.HasMessageToReceive());

        NNT_OVLN_ASSERT_RESULT_SUCCESS(x1.AddSource(SourceName1));

        nn::ovln::Receiver x2 = std::move(x1);
        EXPECT_TRUE(!x1.IsValid());
        EXPECT_TRUE(x2.IsValid());
        EXPECT_TRUE(!x2.HasMessageToReceive());

        x2.RemoveSource(SourceName1);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(x2.AddSource(SourceName2));

        x1 = std::move(x2);
        EXPECT_TRUE(x1.IsValid());
        EXPECT_TRUE(!x2.IsValid());
        EXPECT_TRUE(!x1.HasMessageToReceive());

        x1.RemoveSource(SourceName2);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(x1.AddSource(SourceName3));

        using std::swap;
        swap(x1, x2);
        EXPECT_TRUE(!x1.IsValid());
        EXPECT_TRUE(x2.IsValid());
        EXPECT_TRUE(!x2.HasMessageToReceive());

        x2.RemoveSource(SourceName3);
        NNT_OVLN_ASSERT_RESULT_SUCCESS(x2.AddSource(SourceName4));

        x1.swap(x2);
        EXPECT_TRUE(x1.IsValid());
        EXPECT_TRUE(!x2.IsValid());
        EXPECT_TRUE(!x1.HasMessageToReceive());

        x1.RemoveSource(SourceName4);
    }
    nn::ovln::FinalizeReceicerLibrary();
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_SenderReceiver, Receiver_AddRemoveSource)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    nn::ovln::InitializeReceiverLibrary();
    {
        nn::ovln::Receiver x;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(x.Initialize());
        NNT_OVLN_ASSERT_RESULT_SUCCESS(x.AddSource(SourceName1));
        x.RemoveSource(SourceName1);
    }
    nn::ovln::FinalizeReceicerLibrary();
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_SenderReceiver, InitializeSender)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    nn::ovln::InitializeSenderLibrary();
    nn::ovln::Sender sender1;
    NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1.Initialize(SourceName1, MakeQueueAttribute(1)));
    sender1.Finalize();
    NNT_OVLN_ASSERT_RESULT_SUCCESS(sender1.Initialize(SourceName2, MakeQueueAttribute(10)));
    sender1.Finalize();
    nn::ovln::Sender sender2;
    nn::ovln::FinalizeSenderLibrary();
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_SenderReceiver, Sender_Initialize)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    nn::ovln::InitializeSenderLibrary();
    {
        nn::ovln::Sender x1;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(x1.Initialize(SourceName1, MakeQueueAttribute(1)));
        nn::ovln::Sender x2;
        nn::ovln::Sender x3;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(x2.Initialize(SourceName2, MakeQueueAttribute(1)));
        NNT_OVLN_ASSERT_RESULT_SUCCESS(x3.Initialize(SourceName3, MakeQueueAttribute(1)));
    }
    nn::ovln::FinalizeSenderLibrary();
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_SenderReceiver, Sender_MoveAndSwap)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    nn::ovln::InitializeSenderLibrary();
    {
        nn::ovln::Sender x1;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(x1.Initialize(SourceName1, MakeQueueAttribute(1)));
        EXPECT_TRUE(x1.IsValid());

        nn::ovln::Sender x2 = std::move(x1);
        EXPECT_TRUE(!x1.IsValid());
        EXPECT_TRUE(x2.IsValid());

        x1 = std::move(x2);
        EXPECT_TRUE(x1.IsValid());
        EXPECT_TRUE(!x2.IsValid());

        using std::swap;
        swap(x1, x2);
        EXPECT_TRUE(!x1.IsValid());
        EXPECT_TRUE(x2.IsValid());

        x1.swap(x2);
        EXPECT_TRUE(x1.IsValid());
        EXPECT_TRUE(!x2.IsValid());
    }
    nn::ovln::FinalizeSenderLibrary();
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_SenderReceiver, SendReceive1)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    nn::ovln::InitializeSenderLibrary();
    nn::ovln::InitializeReceiverLibrary();
    {
        nn::ovln::Sender sender;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(sender.Initialize(SourceName1, MakeQueueAttribute(1)));
        nn::ovln::Receiver receiver;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver.Initialize());
        NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver.AddSource(SourceName1));
        {
            EXPECT_TRUE(!receiver.HasMessageToReceive());
            nn::ovln::Message message1;
            EXPECT_TRUE(!receiver.TryReceive(&message1));

            EXPECT_TRUE(sender.Send(Message1, SendOptionBackError));

            EXPECT_TRUE(receiver.HasMessageToReceive());
            EXPECT_TRUE(receiver.TryReceive(&message1));
            EXPECT_TRUE(Equals(Message1, message1));

            EXPECT_TRUE(sender.Send(Message2, SendOptionBackError));

            EXPECT_TRUE(receiver.HasMessageToReceive());
            receiver.Receive(&message1);
            EXPECT_TRUE(Equals(Message2, message1));
        }
    }
    nn::ovln::FinalizeReceicerLibrary();
    nn::ovln::FinalizeSenderLibrary();
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_SenderReceiver, SendReceiveMultiThread)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    nn::ovln::InitializeSenderLibrary();
    nn::ovln::InitializeReceiverLibrary();
    {
        std::thread senderThread([]
        {
            nn::ovln::Sender sender;
            NNT_OVLN_ASSERT_RESULT_SUCCESS(sender.Initialize(SourceName1, MakeQueueAttribute(1)));
            for (int i = 0; i < 10; ++i)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
                while (!sender.Send(MakeTestMessage(i), SendOptionBackError))
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
                }
            }
            while (!(sender.GetUnreceivedMessageCount() == 0))
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
        });
        std::thread receiverThread([]
        {
            nn::ovln::Receiver receiver;
            NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver.Initialize());
            NNT_OVLN_ASSERT_RESULT_SUCCESS(receiver.AddSource(SourceName1));
            for (int i = 0; i < 10; ++i)
            {
                nn::ovln::Message message;
                receiver.Receive(&message);
                EXPECT_TRUE(Equals(MakeTestMessage(i), message));
            }
        });
        senderThread.join();
        receiverThread.join();
    }
    nn::ovln::FinalizeReceicerLibrary();
    nn::ovln::FinalizeSenderLibrary();
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

}
