﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

// テスト対象
#include <nn/ovln/ovln_ReceiverForOverlay.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/ovln/ovln_ForDevelop.h>

#include <nn/nn_Common.h>
#include "testOvln_Utility.h"

using namespace nnt::ovln;

// 実装
#include <utility>
#include <nn/sf/sf_DefaultAllocationPolicy.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/ovln/ovln_Result.h>
#include <nn/ovln/ovln_ResultPrivate.h>
#include <cstring>

namespace {

inline nn::ovln::Message MakeTestMessage(int seed) NN_NOEXCEPT
{
    nn::ovln::Message ret = {};
    ret.tag = seed;
    ret.dataSize = seed;
    std::memset(&ret.data, seed, ret.dataSize);
    return ret;
}

const nn::ovln::Message Message1 = MakeTestMessage(1);
const nn::ovln::Message Message2 = MakeTestMessage(2);
const nn::ovln::Message Message3 = MakeTestMessage(3);
const nn::ovln::Message Message4 = MakeTestMessage(4);
const nn::ovln::Message Message10 = MakeTestMessage(10);
const nn::ovln::Message Message100 = MakeTestMessage(100);

inline bool Equals(const nn::ovln::Message& lhs, const nn::ovln::Message& rhs) NN_NOEXCEPT
{
    return true
        && lhs.tag == rhs.tag
        && lhs.dataSize == rhs.dataSize
        && std::memcmp(&lhs.data, &rhs.data, lhs.dataSize) == 0;
}

#define NNT_OVLN_ASSERT_RECEIVED_MESSAGE(x, message) \
    do \
    { \
        ASSERT_TRUE(::nn::ovln::HasMessageToReceive(&(x))); \
        ::nn::ovln::Message receivedMessage; \
        ASSERT_TRUE(::nn::ovln::TryReceive(&receivedMessage, &(x))); \
        if (!Equals((message), receivedMessage)) \
        { \
            continue; \
        } \
        break; \
    } while (NN_STATIC_CONDITION(true))

TEST(ovln_Overlay, InitializeReceiverLibrary)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    {
        NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeReceiverLibraryForOverlay());
        nn::ovln::FinalizeReceicerLibraryForOverlay();
    }
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_Overlay, InitializeReceiver)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeReceiverLibraryForOverlay());
    {
        nn::ovln::ReceiverForOverlayType receiver;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeReceiver(&receiver));
        nn::ovln::FinalizeReceiver(&receiver);
    }
    nn::ovln::FinalizeReceicerLibraryForOverlay();
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_Overlay, Receive)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeReceiverLibraryForOverlay());
    nn::ovln::ReceiverForOverlayType receiver;
    NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeReceiver(&receiver));
    nn::ovln::FinalizeReceiver(&receiver);
    nn::ovln::FinalizeReceicerLibraryForOverlay();
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_Overlay, InitializeSenderLibrary)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    {
        NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeSenderLibraryForOverlay());
        nn::ovln::FinalizeSenderLibraryForOverlay();
    }
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_Overlay, InitializeSender)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeSenderLibraryForOverlay());
    {
        nn::ovln::SenderForOverlayType sender;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeSenderForValue(&sender));
        nn::ovln::FinalizeSender(&sender);
    }
    {
        nn::ovln::SenderForOverlayType sender;
        NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeSenderWithQueue(&sender, 10));
        nn::ovln::FinalizeSender(&sender);
    }
    nn::ovln::FinalizeSenderLibraryForOverlay();
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_Overlay, ForValue)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeSenderLibraryForOverlay());
    NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeReceiverLibraryForOverlay());
    nn::ovln::ReceiverForOverlayType receiver;
    NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeReceiver(&receiver));
    nn::ovln::SenderForOverlayType sender;
    NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeSenderForValue(&sender));
    {
        EXPECT_TRUE(nn::ovln::Send(&sender, Message1));
        NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, Message1);
        EXPECT_TRUE(nn::ovln::Send(&sender, Message2));
        EXPECT_TRUE(nn::ovln::Send(&sender, Message3));
        NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, Message3);
    }
    nn::ovln::FinalizeSender(&sender);
    nn::ovln::FinalizeReceiver(&receiver);
    nn::ovln::FinalizeReceicerLibraryForOverlay();
    nn::ovln::FinalizeSenderLibraryForOverlay();
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

TEST(ovln_Overlay, WithQueue)
{
    ObservedMemoryResource mr;
    ObservedMemoryResource::AllocationInfo info;
    mr.SetAllocationInfo(&info);
    nn::ovln::PrepareSenderAndReceiverForDevelop(&mr);
    NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeSenderLibraryForOverlay());
    NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeReceiverLibraryForOverlay());
    nn::ovln::ReceiverForOverlayType receiver;
    NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeReceiver(&receiver));
    nn::ovln::SenderForOverlayType sender;
    NNT_OVLN_ASSERT_RESULT_SUCCESS(nn::ovln::InitializeSenderWithQueue(&sender, 3));
    {
        EXPECT_TRUE(nn::ovln::Send(&sender, Message1));
        NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, Message1);
        EXPECT_TRUE(nn::ovln::Send(&sender, Message2));
        EXPECT_TRUE(nn::ovln::Send(&sender, Message3));
        NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, Message2);
        NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, Message3);
        EXPECT_TRUE(nn::ovln::Send(&sender, Message1));
        EXPECT_TRUE(nn::ovln::Send(&sender, Message2));
        EXPECT_TRUE(nn::ovln::Send(&sender, Message3));
        EXPECT_TRUE(!nn::ovln::Send(&sender, Message4));
        NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, Message1);
        NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, Message2);
        NNT_OVLN_ASSERT_RECEIVED_MESSAGE(receiver, Message3);
        EXPECT_TRUE(nn::ovln::Send(&sender, Message1));
        EXPECT_TRUE(nn::ovln::Send(&sender, Message2));
        EXPECT_TRUE(nn::ovln::Send(&sender, Message3));
        EXPECT_TRUE(!nn::ovln::Send(&sender, Message4));
    }
    nn::ovln::FinalizeSender(&sender);
    nn::ovln::FinalizeReceiver(&receiver);
    nn::ovln::FinalizeReceicerLibraryForOverlay();
    nn::ovln::FinalizeSenderLibraryForOverlay();
    nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
    ASSERT_EQ(info.allocateCount, info.deallocateCount);
}

}
