﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "os_DemoUtil.h"

namespace {
    const size_t StackSize  = 4096;
    const size_t StackSize1 = 4096;
    const size_t StackSize2 = 4096;

    // NN_ALIGNAS には即値定数しか書けない
    NN_ALIGNAS( 4096 )  char g_stack [ StackSize  ];
    NN_ALIGNAS( 4096 )  char g_stack1[ StackSize1 ];
    NN_ALIGNAS( 4096 )  char g_stack2[ StackSize2 ];
}


/*!
  @struct       ThreadParam
  @brief        ThreadDemoFunc に渡すパラメータの構造体です。
*/
struct ThreadParam
{
    int             threadNum;      //!< スレッド番号です。
    nn::os::EventType*  pWaitEvent;     //!< 待機するイベントのポインタです。
    nn::os::EventType*  pSignalEvent;   //!< 発生させるイベントのポインタです。
};

/*!
  @brief        ThreadDemoで使われるスレッド関数で、下記の処理を行います。

                文字列の出力とスレッドスイッチを含むループを実行します。

  @param[in]    param       スレッドに渡すパラメータのポインタです。

  @return       なし。
*/
void ThreadDemoFunc(void* arg)
{
    ThreadParam* param = reinterpret_cast<ThreadParam*>( arg );

    for(int i = 0; i < 5; ++i)
    {
        nn::os::WaitEvent( param->pWaitEvent );
        NN_LOG("  Thread %d, count = %d\n", param->threadNum, i);
        nn::os::SignalEvent( param->pSignalEvent );
    }
    NN_LOG(" Thread %d END\n", param->threadNum);
}

/*!
  @brief        ThreadDemoで使われるスレッド関数で、下記の処理を行います。

                文字列の出力のみを行います。

  @param[in]    param       使用しません。

  @return       なし。
*/
void SimpleThreadFunc(void* param)
{
    NN_UNUSED(param);
    NN_LOG(" SimpleThreadFunc\n");
}

/*!
  @brief        スレッドの作成と、スレッドの結合を行うデモです。

                １つのスレッドを作成したのち、そのスレッドの終了を待ちます。

  @return       なし。
*/
void ThreadDemo0()
{
    NN_LOG("-- Thread Demo0 --\n");

    nn::os::ThreadType  thread;

    //----------------------------------------------------------------
    // スレッド作成
    nn::os::CreateThread( &thread, SimpleThreadFunc, NULL, g_stack, StackSize, nn::os::DefaultThreadPriority );
    nn::os::StartThread( &thread );

    // スレッドの終了待ち
    nn::os::WaitThread( &thread );

    // スレッドの破棄
    nn::os::DestroyThread( &thread );

    //----------------------------------------------------------------
    // スレッド作成（優先度・コア指定）
    nn::os::CreateThread( &thread, SimpleThreadFunc, NULL, g_stack, StackSize, nn::os::DefaultThreadPriority );
    nn::os::StartThread( &thread );

    // スレッドの優先度変更
    s32 prior = nn::os::GetThreadPriority( &thread );
    nn::os::ChangeThreadPriority( &thread, nn::os::DefaultThreadPriority + 1 );
    NN_LOG("Thread Priority : %d -> %d\n", prior, nn::os::GetThreadPriority( &thread ));

    // スレッドの終了待ち
    nn::os::WaitThread( &thread );

    // スレッドの破棄
    nn::os::DestroyThread( &thread );
}

/*!
  @brief        スレッドのスイッチと、スレッドの結合を行うデモです。

                ２つのスレッドでイベントの待機と発生を行い、
                互いのスレッドを起こし合います。

  @return       なし。
*/
void ThreadDemo1()
{
    NN_LOG("-- Thread Demo1 --\n");

    nn::os::ThreadType  thread1;
    ThreadParam     threadParam1;

    nn::os::ThreadType  thread2;
    ThreadParam     threadParam2;

    // イベント作成
    nn::os::EventType   event1;
    nn::os::InitializeEvent( &event1, false, nn::os::EventClearMode_AutoClear );
    nn::os::EventType   event2;
    nn::os::InitializeEvent( &event2, false, nn::os::EventClearMode_AutoClear );

    // スレッド作成
    threadParam1.threadNum    = 1;
    threadParam1.pWaitEvent   = &event1;
    threadParam1.pSignalEvent = &event2;
    nn::os::CreateThread(&thread1, ThreadDemoFunc, &threadParam1, g_stack1, StackSize1, nn::os::DefaultThreadPriority);

    threadParam2.threadNum    = 2;
    threadParam2.pWaitEvent   = &event2;
    threadParam2.pSignalEvent = &event1;
    nn::os::CreateThread(&thread2, ThreadDemoFunc, &threadParam2, g_stack2, StackSize2, nn::os::DefaultThreadPriority);

    NN_LOG("Thread created.\n");
    DemoUtil::DumpObjectCounts();

    // スレッドを開始する
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );

    // スレッド１を起こす
    // 以降はスレッド１・スレッド２が交互に起きる
    nn::os::SignalEvent( &event1 );

    // スレッドの終了待ち
    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );

    // スレッドの破棄
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );

    // イベントの破棄
    nn::os::FinalizeEvent( &event1 );
    nn::os::FinalizeEvent( &event2 );
}

/*!
  @brief        スレッドに関するデモです。

                スレッドのデモを実行します。

  @return       なし。
*/
void ThreadDemo()
{
    NN_LOG("*** Thread Demo starts. ***\n");

    ThreadDemo0();
    ThreadDemo1();

    NN_LOG("*** End of Thread Demo ***\n");
}

