﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include "os_DemoUtil.h"

// デモで用いるセマフォ
nn::os::SemaphoreType s_Semaphore;


/*!
  @brief        SemaphoreDemo で使われるスレッド関数で、下記の処理を行います。

                セマフォを獲得します。
                文字列の出力とスレッドスイッチを含むループを実行します。
                セマフォを解放し、終了します。

  @param[in]    param       スレッド番号です。

  @return       なし。
*/
void SemaphoreDemoFunc(void* param)
{
    uint32_t threadNum = reinterpret_cast<uintptr_t>( param ) & UINT32_MAX;

    // セマフォを獲得してから、処理を行う
    NN_LOG(" Thread %d, Trying wait semaphore\n", threadNum);
    nn::os::AcquireSemaphore( &s_Semaphore );
    NN_LOG(" Thread %d, Success get semaphore\n", threadNum);
    for(int i = 0; i < 5; ++i)
    {
        NN_LOG("  Thread %d, i = %d\n", threadNum, i);
        nn::TimeSpan::FromMilliSeconds(1);
    }

    NN_LOG(" Thread %d, Release semaphore\n", threadNum);
    // 処理が終わったのでセマフォを解放
    nn::os::ReleaseSemaphore( &s_Semaphore );

    NN_LOG(" Thread %d, END\n", threadNum);
}

/*!
  @brief        セマフォを操作するデモです。

                単一スレッドでセマフォの初期化と解放を行います。

  @return       なし。
*/
void SemaphoreDemo0()
{
    NN_LOG("-- Semaphore Demo0 --\n");

    // セマフォを作成
    // 初期カウントを 0 、最大カウントを 5 とする
    nn::os::InitializeSemaphore( &s_Semaphore, 0, 5 );

    DemoUtil::DumpObjectCounts();

    // セマフォのカウントを変更
    for(int i = 0; i < 5; i++)
    {
        nn::os::ReleaseSemaphore( &s_Semaphore );
        s32 count = nn::os::GetCurrentSemaphoreCount( &s_Semaphore );
#ifdef NN_BUILD_RELEASE
        NN_UNUSED(count);
#endif
        NN_LOG("semaphore count %d -> %d + 1\n", count, count);
    }

    // セマフォの破棄
    nn::os::FinalizeSemaphore( &s_Semaphore );
}

/*!
  @brief        セマフォを操作するデモです。

                複数スレッド間で同一セマフォの獲得を試み、セマフォを獲得できたスレッドから処理を行います。
                セマフォの最大値は スレッド数 - 2 です。

  @return       なし。
*/
void SemaphoreDemo1()
{
    NN_LOG("-- Semaphore Demo1 --\n");

    // セマフォを作成
    // 初期カウントを 0、最大カウントをスレッド数 - 2 とする
    NN_ASSERT(DemoUtil::OsMaxThreadNum - 2 > 0);
    InitializeSemaphore( &s_Semaphore, 0, DemoUtil::OsMaxThreadNum - 2 );

    // スレッドの作成
    nn::os::ThreadType threads[DemoUtil::OsMaxThreadNum];
    DemoUtil::CreateThreads(threads, SemaphoreDemoFunc, DemoUtil::OsMaxThreadNum);

    // セマフォのカウントを変更（複数カウント返却）
    nn::os::ReleaseSemaphore( &s_Semaphore, DemoUtil::OsMaxThreadNum - 2 );

    // スレッドの終了待ち
    DemoUtil::WaitThreads(threads, DemoUtil::OsMaxThreadNum);

    // スレッドの破棄
    DemoUtil::DestroyThreads(threads, DemoUtil::OsMaxThreadNum);

    // セマフォの破棄
    nn::os::FinalizeSemaphore( &s_Semaphore );
}

/*!
  @brief        セマフォに関するデモです。

                セマフォのデモを実行します。

  @return       なし。
*/
void SemaphoreDemo()
{
    NN_LOG("*** Semaphore Demo starts. ***\n");

    SemaphoreDemo0();
    SemaphoreDemo1();

    NN_LOG("*** End of Semaphore Demo ***\n");
}

