﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include "os_DemoUtil.h"

using nn::os::ThreadType;
using nn::os::MutexType;

namespace {

    volatile int g_Counter;
    nn::os::MutexType   g_Lock;
    nn::os::ThreadType  g_Threads1[DemoUtil::OsMaxThreadNum];
    nn::os::ThreadType  g_Threads2[DemoUtil::OsMaxThreadNum];

    void IncrementWithoutMutex(void*)
    {
        // 作為的ではありますが、レジスタに変数をロードした
        // 直後にスリープして他のスレッドに制御が移ります。
        int tmp = g_Counter;
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
        ++tmp;
        g_Counter = tmp;
    }

    void IncrementWithMutex(void*)
    {
        nn::os::LockMutex( &g_Lock );

        // 加算全体を排他するので、g_Counterは
        // 1度に1つのスレッドでしか読み書きされません。
        IncrementWithoutMutex(NULL);

        nn::os::UnlockMutex( &g_Lock );
    }

}


/*!
  @brief    クリティカルセクションに関するデモです。
            複数のスレッドから1つのグローバルカウンタをインクリメントします。
            クリティカルセクションがない場合は正確にカウンタをインクリメントできません。

  @return   なし。
*/

void MutexDemo()
{
    NN_LOG("*** Mutex Demo starts. ***\n");
    // ミューテックスの初期化
    nn::os::InitializeMutex( &g_Lock, false, 0 );

    NN_LOG("g_Counter is to be incremented by %d threads\n", DemoUtil::OsMaxThreadNum);
    // 排他制御をした上でg_Counterを更新します。
    g_Counter = 0;

    // スレッドの作成
    DemoUtil::CreateThreads(g_Threads1, IncrementWithMutex, DemoUtil::OsMaxThreadNum);

    // スレッドの終了待ち
    DemoUtil::WaitThreads(g_Threads1, DemoUtil::OsMaxThreadNum);
    NN_LOG("IncrementWithMutex: g_Counter=%d\n", g_Counter);

    // スレッドの破棄
    DemoUtil::DestroyThreads(g_Threads1, DemoUtil::OsMaxThreadNum);

    // 排他制御せずにg_Counterを更新します。
    g_Counter = 0;

    // スレッドの作成
    DemoUtil::CreateThreads(g_Threads2, IncrementWithoutMutex, DemoUtil::OsMaxThreadNum);

    // スレッドの終了待ち
    DemoUtil::WaitThreads(g_Threads2, DemoUtil::OsMaxThreadNum);
    NN_LOG("IncrementWithoutMutex: g_Counter=%d\n", g_Counter);

    // スレッドの破棄
    DemoUtil::DestroyThreads(g_Threads2, DemoUtil::OsMaxThreadNum);

    // ミューテックスの破棄
    nn::os::FinalizeMutex( &g_Lock );
    NN_LOG("*** End of Mutex Demo ***\n");
}

