﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdint>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

namespace {
    NN_ALIGNAS(4096)    char    g_stack[4096];
}

/*!
  @brief        MessageQueueDemoで使われるスレッド関数で、下記の処理を行います。
                文字列の出力を行いながら、キューにデータを挿入します。

  @param[in]    pQueue      データを挿入するキューのポインタです。

  @return       なし。
*/
void SendMessageQueueThreadFunc(void* p)
{
    nn::os::MessageQueueType* pQueue = reinterpret_cast<nn::os::MessageQueueType*>(p);

    NN_LOG(" SendMessageQueue Thraed Start\n");
    for(int i = 0; i < 10; i++)
    {
        NN_LOG("  Try to SendMessageQueue(%d)\n", i);
        nn::os::SendMessageQueue( pQueue, i );
        NN_LOG("  SendMessageQueue(%d)\n", i);
    }
    NN_LOG(" SendMessageQueue Thraed End\n");
}

/*!
  @brief        MessageQueueDemoで使われるスレッド関数で、下記の処理を行います。

                文字列の出力を行いながら、キューにデータを挿入します。

  @param[in]    pQueue      データを挿入するキューのポインタです。

  @return       なし。
*/
void SendMessageQueueThreadFunc2(void *p)
{
    nn::os::MessageQueueType* pQueue = reinterpret_cast<nn::os::MessageQueueType*>(p);

    NN_LOG(" SendMessageQueue Thraed Start\n");
    for(int i = 0; i < 10; i++)
    {
        NN_LOG("  SendMessageQueue(%d)\n", i);
        nn::os::SendMessageQueue( pQueue, i );
    }
    NN_LOG(" SendMessageQueue Thraed End\n");
}

/*!
  @brief        メッセージキューのデモです。

                スレッドでメッセージキューにデータを挿入し、
                メインスレッドでデータを取り出します。

  @return       なし。
*/
void MessageQueueDemo1()
{
    NN_LOG("-- MessageQueue Demo1 --\n");

    nn::os::MessageQueueType    queue;
    const size_t    Size = 5;
    std::uintptr_t  buffer[Size];
    nn::os::ThreadType  enqueueThread;


    //----------------------------------------------------------------
    // デキュー待ちが発生するメッセージキューのデモ
    NN_LOG("** Waiting ReceiveMessageQueue Demo. **\n");
    // メッセージキューの初期化
    nn::os::InitializeMessageQueue( &queue, buffer, Size );

    // スレッドでキューの末尾にデータを挿入する
    nn::os::CreateThread( &enqueueThread, SendMessageQueueThreadFunc, &queue, g_stack, 4096, nn::os::DefaultThreadPriority - 1);
    nn::os::StartThread( &enqueueThread );

    // キューの先頭からデータを取り出す
    for(int i = 0; i < 10; i++)
    {
        uintptr_t   data;
        nn::os::ReceiveMessageQueue( &data, &queue );
        NN_LOG("ReceiveMessageQueue(%d)\n", data);
    }

    // スレッドの結合・破棄
    nn::os::WaitThread( &enqueueThread );
    nn::os::DestroyThread( &enqueueThread );
    // メッセージキューの破棄
    nn::os::FinalizeMessageQueue( &queue );


    //----------------------------------------------------------------
    // エンキュー待ちが発生するメッセージキューのデモ
    NN_LOG("** Waiting SendMessageQueue Demo. **\n");
    // メッセージキューの初期化
    nn::os::MessageQueueType    queue2;
    nn::os::InitializeMessageQueue( &queue2, buffer, Size );

    // スレッドでキューの末尾にデータを挿入する
    nn::os::CreateThread( &enqueueThread, SendMessageQueueThreadFunc2, &queue2, g_stack, 4096, nn::os::DefaultThreadPriority - 1);
    nn::os::StartThread( &enqueueThread );

    // キューの先頭からデータを取り出す
    for(int i = 0; i < 10; i++)
    {
        NN_LOG("Try to ReceiveMessageQueue(%d)\n", i);
        uintptr_t   data;
        nn::os::ReceiveMessageQueue( &data, &queue2 );
        NN_LOG("ReceiveMessageQueue(%d)\n", data);
    }

    // スレッドの終了待ち
    nn::os::WaitThread( &enqueueThread );

    // スレッドの破棄
    nn::os::DestroyThread( &enqueueThread );

    // メッセージキューの破棄
    nn::os::FinalizeMessageQueue( &queue2 );
}

/*!
  @brief        メッセージキューに関するデモです。

                メッセージキューのデモを実行します。

  @return       なし。
*/
void MessageQueueDemo()
{
    NN_LOG("*** MessageQueue Demo starts. ***\n");

    MessageQueueDemo1();

    NN_LOG("*** End of MessageQueue Demo ***\n");
}


