﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include "os_DemoUtil.h"

// デモで用いるイベント
nn::os::EventType s_Event[DemoUtil::OsMaxThreadNum];


/*!
  @brief        EventDemoで使われるスレッド関数で、下記の処理を行います。

                １．イベントを待ちます。
                ２．文字列の出力とスレッドスイッチを含むループを実行します。
                ３．イベントをクリアします。
                ４．次のスレッドのイベントを発生させます。
                ５．再度イベントを待ちます。
                ６．次のスレッドのイベントを発生させて終了します。

  @param[in]    param       スレッド番号です。

  @return       なし。
*/
void EventDemoFunc(void* param)
{
    uint32_t threadNum = reinterpret_cast<uintptr_t>(param) & UINT32_MAX;

    // イベントを待つ
    NN_LOG(" Thread %d, Waiting event\n", threadNum);
    nn::os::WaitEvent( &s_Event[threadNum] );

    // イベントが発生
    NN_LOG(" Thread %d, Signaled event\n", threadNum);
    for(int i = 0; i < 5; ++i)
    {
        NN_LOG("  Thread %d, i = %d\n", threadNum, i);
        nn::os::YieldThread();
    }

    // イベントをクリア
    nn::os::ClearEvent( &s_Event[threadNum] );

    // 次のイベントを起こす
    NN_LOG(" Thread %d, Signal next event\n", threadNum);
    nn::os::SignalEvent( &s_Event[(threadNum + 1) % DemoUtil::OsMaxThreadNum] );

    // 再びイベントを待つ
    NN_LOG(" Thread %d, Waiting event again\n", threadNum);
    nn::os::WaitEvent( &s_Event[threadNum] );
    NN_LOG(" Thread %d, Signaled event again\n", threadNum);

    // 次のイベントを起こす
    NN_LOG(" Thread %d, Signal next event\n", threadNum);
    nn::os::SignalEvent( &s_Event[(threadNum + 1) % DemoUtil::OsMaxThreadNum] );

    NN_LOG(" Thread %d, END\n", threadNum);
}

/*!
  @brief        EventDemoで使われるスレッド関数で、下記の処理を行います。

                文字列の出力とイベントの待機のみを行います。

  @param[in]    pEvent      待機するイベントのポインタです。

  @return       なし。
*/
void SimpleEventFunc(void* p)
{
    nn::os::EventType* pEvent = reinterpret_cast<nn::os::EventType*>( p );

    // イベントを待つ
    NN_LOG("Waiting event\n");
    nn::os::WaitEvent( pEvent );
    NN_LOG("Thread END\n");
}

/*!
  @brief        イベントを用いるデモです。

                複数のスレッド間で、イベントの待機と発生を行います。

  @return       なし。
*/
void EventDemo1()
{
    NN_LOG("-- Event Demo1 --\n");

    // イベントを作成
    for(u32 i = 0; i < DemoUtil::OsMaxThreadNum; ++i)
    {
        nn::os::InitializeEvent( &s_Event[i], false, nn::os::EventClearMode_ManualClear );
    }

    // スレッドの作成
    nn::os::ThreadType threads[DemoUtil::OsMaxThreadNum];
    DemoUtil::CreateThreads(threads, EventDemoFunc, DemoUtil::OsMaxThreadNum);

    DemoUtil::DumpObjectCounts();

    // 最初のイベントだけ通知しておく
    nn::os::SignalEvent( &s_Event[0] );

    // スレッドの終了待ち
    DemoUtil::WaitThreads(threads, DemoUtil::OsMaxThreadNum);

    // スレッドの破棄
    DemoUtil::DestroyThreads(threads, DemoUtil::OsMaxThreadNum);

    // イベントの破棄
    for(u32 i = 0; i < DemoUtil::OsMaxThreadNum; ++i)
    {
        nn::os::FinalizeEvent( &s_Event[i] );
    }
}


namespace {
    const size_t        StackSize = 4096;
    NN_ALIGNAS(4096)    char    g_stack[StackSize];
}

/*!
  @brief        イベントを用いるデモです。

                イベント待ち状態のスレッドを起こします。

  @return       なし。
*/
void EventDemo0()
{
    NN_LOG("-- Event Demo0 --\n");

    // イベントを作成
    nn::os::EventType   event0;
    nn::os::InitializeEvent( &event0, false, nn::os::EventClearMode_AutoClear );

    // イベント待ちのスレッドを作成
    nn::os::ThreadType  thread;

    nn::os::CreateThread( &thread, SimpleEventFunc, &event0, g_stack, StackSize, nn::os::DefaultThreadPriority );
    nn::os::StartThread( &thread );

    // スレッド作成待ち
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1) );

    // イベント待ちのスレッドに通知する
    NN_LOG("Signal event.\n");
    nn::os::SignalEvent( &event0 );

    // スレッドの終了待ち
    nn::os::WaitThread( &thread );

    // スレッドの破棄
    nn::os::DestroyThread( &thread );

    // イベントの破棄
    nn::os::FinalizeEvent( &event0 );
}

/*!
  @brief        イベントに関するデモです。

                イベントのデモを実行します。

  @return       なし。
*/
void EventDemo()
{
    NN_LOG("*** Event Demo starts. ***\n");

    EventDemo0();
    EventDemo1();

    NN_LOG("*** End of Event Demo ***\n");
}

