﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/init.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/os/os_VirtualAddressMemory.h>
#include <nn/util/util_TinyMt.h>

#include <nnt.h>

// 64bit のみ対応
#if !(defined(NN_OS_CPU_ARM_AARCH64_ARMV8A) || defined(NN_BUILD_CONFIG_CPU_X64))
    #error "This program supports only 64bit architecture."
#endif

//---------------------------------------------------------------------------
// VirtualAddressMemory 機能で断片化を起こすテスト
//---------------------------------------------------------------------------

/*
    仮想アドレスメモリの代わりにヒープの malloc/free を使う場合は USE_MEMORY_HEAP を定義する。
    Horizon 環境で USE_MEMORY_HEAP を利用してテストを実行すると、断片化によりテストは失敗する。
    USE_MEMORY_HEAP を定義せず、仮想アドレスメモリを使う場合は、断片化しないのでテストは成功する。
*/
//#define USE_MEMORY_HEAP

namespace nnt { namespace os { namespace VirtualAddressMemory {

namespace {
    const size_t PageSize = nn::os::MemoryPageSize;
    //const size_t AllocationStepSize = PageSize;
    const size_t AllocationStepSize = 1 * 1024 * 1024UL; // 2回目以降テスト起動に失敗する問題の再現条件
    //const size_t MaxAllocationSize = 2 * 1024 * 1024 * 1024UL;
#if defined(NNT_OS_APPLET)
    // Applet region (~450MB) - heap size - usage of other processes on develop (~90MB) - code/data/bss - margin
    const size_t MaxAllocationSize = 256 * 1024 * 1024UL;
#else
    const size_t MaxAllocationSize = 1024 * 1024 * 1024UL; // 2回目以降テスト起動に失敗する問題の再現条件
#endif
    const size_t AllocationCount = MaxAllocationSize / AllocationStepSize;

    // os ライブラリが管理用に使用するページ数の管理領域の最大値 (AllocationCount * 8 byte)
    const size_t MaxOsLibraryPageCount = ((AllocationCount * 8) + PageSize - 1) / PageSize;

    size_t g_AllocationSizes[AllocationCount];
    uintptr_t g_Addresses[AllocationCount];
    bool g_AddressValidFlags[AllocationCount];

    size_t Select(bool* validFlags, size_t count)
    {
        // Index を奇数に限定して valid を探す
        for (size_t i = 0; i < count; i += 2)
        {
            if (validFlags[i])
            {
                return i;
            }
        }

        // Index を偶数に限定して valid を探す
        for (size_t i = 1; i < count; i += 2)
        {
            if (validFlags[i])
            {
                return i;
            }
        }

        NN_ABORT();
    }

    uintptr_t Allocate(size_t size)
    {
#if defined(USE_MEMORY_HEAP)
        return reinterpret_cast<uintptr_t>(malloc(size));
#else
        uintptr_t address;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateAddressRegion(&address, size));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryPages(address, size));
        return address;
#endif
    }

    void Free(uintptr_t address, size_t size)
    {
#if defined(USE_MEMORY_HEAP)
        NN_UNUSED(size);
        free(reinterpret_cast<void*>(address));
#else
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::FreeMemoryPages(address, size));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::FreeAddressRegion(address));
#endif
    }

    bool FragmentationTest(uintptr_t(*alloc)(size_t), void(*deallocate)(uintptr_t, size_t), size_t availableMemorySize)
    {
        // グローバル変数を初期化する
        for (int i = 0; i < AllocationCount; i++)
        {
            g_AllocationSizes[i] = (i + 1) * AllocationStepSize;
            g_Addresses[i] = 0;
            g_AddressValidFlags[i] = false;
        }

        size_t allocatedSize = 0;

        for (size_t i = 0; i < AllocationCount; i++)
        {
            while (allocatedSize + g_AllocationSizes[i] > availableMemorySize)
            {
                // 割り当てるサイズが最大値を超えたら、断片化しやすいパターンで解放する
                size_t index = Select(g_AddressValidFlags, i);

                NN_LOG("Deallocate %zu bytes (%p).\n", g_AllocationSizes[i], g_Addresses[index]);
                deallocate(g_Addresses[index], g_AllocationSizes[index]);

                g_AddressValidFlags[index] = false;
                allocatedSize -= g_AllocationSizes[index];
            }

            NN_LOG("Allocate %zu bytes ", g_AllocationSizes[i]);
            g_Addresses[i] = alloc(g_AllocationSizes[i]);
            NN_LOG("(%p)\n", g_Addresses[i]);

            if (g_Addresses[i] == 0)
            {
                // 途中で alloc に失敗したら false を返す
                return false;
            }

            g_AddressValidFlags[i] = true;
            allocatedSize += g_AllocationSizes[i];
        }

        // alloc にすべて成功したら true を返す
        return true;
    }

}

TEST(VirtualAddressMemoryTest, Fragmentation)
{
    // 仮想アドレスメモリの最大サイズを計算する
    nn::os::MemoryInfo info;
    nn::os::QueryMemoryInfo(&info);
#if defined(USE_MEMORY_HEAP)
    NN_UNUSED(MaxOsLibraryPageCount);
    size_t availableMemorySize = info.totalMemoryHeapSize;
#else
    size_t availableMemorySize =
        info.totalAvailableMemorySize - (info.totalUsedMemorySize + MaxOsLibraryPageCount * PageSize);
#endif
    NN_LOG("Available memory: %zu bytes.\n", availableMemorySize);

    ASSERT_LE(MaxAllocationSize, availableMemorySize);

    bool passed = FragmentationTest(Allocate, Free, availableMemorySize);
    ASSERT_EQ(passed, true);
}

}}} // namespace nnt::os::VirtualAddressMemory

#if !defined(USE_MEMORY_HEAP)
namespace
{
    const size_t HeapSize = 64 * 1024 * 1024UL;
    uint8_t g_Heap[HeapSize];
}

extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(reinterpret_cast<void*>(g_Heap), HeapSize);
}
#endif
