﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/os/os_VirtualAddressMemory.h>

#include <nnt.h>

#include "test_VirtualAddressMemoryCommon.h"

namespace nnt { namespace os { namespace VirtualAddressMemory {

void AddressRegionPerformanceTest(size_t size, size_t loopCount)
{
    NN_LOG("%s\n", __FUNCTION__);

    CommandProcessor processor;

    // Allocate
    CommandList allocateCommands;
    for (int i = 0; i < loopCount; i++)
    {
        allocateCommands.AddAllocateRegionCommand(i, size);
    }
    processor.Process(allocateCommands.begin(), allocateCommands.end());
    auto allocateTime = processor.GetProcessingTime();

    NN_LOG("nn::os::AllocateAddressRegion(%zu): %ld ns.\n", size, allocateTime.GetNanoSeconds() / loopCount);

    // Free
    CommandList freeCommands;
    for (int i = 0; i < loopCount; i++)
    {
        freeCommands.AddFreeRegionCommand(i);
    }
    processor.Process(freeCommands.begin(), freeCommands.end());
    auto freeTime = processor.GetProcessingTime();

    NN_LOG("nn::os::FreeAddressRegion(): %ld ns.\n", freeTime.GetNanoSeconds() / loopCount);
}

void MemoryPagePerformanceTest(size_t regionSize, size_t pageSize)
{
    NN_LOG("%s\n", __FUNCTION__);

    CommandProcessor processor;

    processor.ProcessAllocateAddressRegion(0, regionSize);

    const size_t count = regionSize / pageSize;

    // Allocate
    CommandList allocateCommands;
    for (int i = 0; i < count; i++)
    {
        allocateCommands.AddAllocatePagesCommand(0, i * pageSize, pageSize);
    }
    processor.Process(allocateCommands.begin(), allocateCommands.end());
    auto allocateTime = processor.GetProcessingTime();

    NN_LOG("nn::os::AllocateMemoryPages(%zu): %ld ns.\n", pageSize, allocateTime.GetNanoSeconds() / count);

    // Free
    CommandList freeCommands;
    for (int i = 0; i < count; i++)
    {
        freeCommands.AddFreePagesCommand(0, i * pageSize, pageSize);
    }
    processor.Process(freeCommands.begin(), freeCommands.end());
    auto freeTime = processor.GetProcessingTime();

    NN_LOG("nn::os::FreeMemoryPages(): %ld ns.\n", freeTime.GetNanoSeconds() / count);

    processor.ProcessFreeAddressRegion(0);
}

void CommandList::AddAllocateRegionCommand(int regionId, size_t size)
{
    Command command = { CommandType_AllocateAddressRegion, regionId, 0, size, nullptr, 0 };
    push_back(command);
}

void CommandList::AddFreeRegionCommand(int regionId)
{
    Command command = { CommandType_FreeAddressRegion, regionId, 0, 0, nullptr, 0 };
    push_back(command);
}

void CommandList::AddAllocatePagesCommand(int regionId, size_t offset, size_t size)
{
    Command command = { CommandType_AllocateMemoryPages, regionId, offset, size, nullptr, 0 };
    push_back(command);
}

void CommandList::AddFreePagesCommand(int regionId, size_t offset, size_t size)
{
    Command command = { CommandType_FreeMemoryPages, regionId, offset, size, nullptr, 0 };
    push_back(command);
}

void CommandList::AddFillCommand(int regionId, size_t offset, size_t size)
{
    Command command = { CommandType_FillMemory, regionId, offset, size, nullptr, 0 };
    push_back(command);
}

void CommandList::AddVerifyCommand(int regionId, size_t offset, size_t size)
{
    Command command = { CommandType_VerifyMemory, regionId, offset, size, nullptr, 0 };
    push_back(command);
}

void CommandList::AddSyncCommand(nn::os::BarrierType* pBarrier)
{
    Command command = { CommandType_Sync, 0, 0, 0, pBarrier, 0 };
    push_back(command);
}

void CommandList::AddSleepCommand(nn::TimeSpan timeSpan)
{
    Command command = { CommandType_Sleep, 0, 0, 0, nullptr, timeSpan };
    push_back(command);
}

void CommandProcessor::LoadRegionInfo(CommandProcessor& other)
{
    m_RegionAddressMap = other.m_RegionAddressMap;
}

void CommandProcessor::Process(Command& command)
{
    NN_ABORT_UNLESS(command.offset % nn::os::MemoryPageSize == 0);
    NN_ABORT_UNLESS(command.size % nn::os::MemoryPageSize == 0);

    switch (command.type)
    {
    case CommandType_AllocateAddressRegion:
        ProcessAllocateAddressRegion(command.regionId, command.size);
        break;
    case CommandType_FreeAddressRegion:
        ProcessFreeAddressRegion(command.regionId);
        break;
    case CommandType_AllocateMemoryPages:
        ProcessAllocateMemoryPages(command.regionId, command.offset, command.size);
        break;
    case CommandType_FreeMemoryPages:
        ProcessFreeMemoryPages(command.regionId, command.offset, command.size);
        break;
    case CommandType_FillMemory:
        ProcessFillMemory(command.regionId, command.offset, command.size);
        break;
    case CommandType_VerifyMemory:
        ProcessVerifyMemory(command.regionId, command.offset, command.size);
        break;
    case CommandType_Sync:
        ProcessSync(command.pBarrier);
        break;
    case CommandType_Sleep:
        ProcessSleep(command.timeSpan);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void CommandProcessor::ProcessAllocateAddressRegion(int regionId, size_t size)
{
    uintptr_t address;

    NNT_OS_DETAIL_LOG("AllocateAddressRegion regionId:%d size:%zu\n", regionId, size);
    NNT_ASSERT_RESULT_SUCCESS(nn::os::AllocateAddressRegion(&address, size));
    m_RegionAddressMap[regionId] = address;
}

void CommandProcessor::ProcessFreeAddressRegion(int regionId)
{
    NNT_OS_DETAIL_LOG("FreeAddressRegion regionId:%d\n", regionId);
    NNT_ASSERT_RESULT_SUCCESS(nn::os::FreeAddressRegion(m_RegionAddressMap[regionId]));
}

void CommandProcessor::ProcessAllocateMemoryPages(int regionId, size_t offset, size_t size)
{
    NNT_OS_DETAIL_LOG("AllocateMemoryPages regionId:%d offset:%zu size:%zu\n", regionId, offset, size);
    NNT_ASSERT_RESULT_SUCCESS(nn::os::AllocateMemoryPages(m_RegionAddressMap[regionId] + offset, size));
}

void CommandProcessor::ProcessFreeMemoryPages(int regionId, size_t offset, size_t size)
{
    NNT_OS_DETAIL_LOG("FreeMemoryPages regionId:%d offset:%zu size:%zu\n", regionId, offset, size);
    NNT_ASSERT_RESULT_SUCCESS(nn::os::FreeMemoryPages(m_RegionAddressMap[regionId] + offset, size));
}

void CommandProcessor::ProcessFillMemory(int regionId, size_t offset, size_t size)
{
    NNT_OS_DETAIL_LOG("Fill regionId:%d offset:%zu size:%zu\n", regionId, offset, size);
    Fill(m_RegionAddressMap[regionId] + offset, size);
}

void CommandProcessor::ProcessVerifyMemory(int regionId, size_t offset, size_t size)
{
    NNT_OS_DETAIL_LOG("Verify regionId:%d offset:%zu size:%zu\n", regionId, offset, size);
    m_IsVerificationPassed = Verify(m_RegionAddressMap[regionId] + offset, size);
}

void CommandProcessor::ProcessSync(nn::os::BarrierType* pBarrier)
{
    NNT_OS_DETAIL_LOG("Sync pBarrier:%p\n", pBarrier);
    nn::os::AwaitBarrier(pBarrier);
}

void CommandProcessor::ProcessSleep(nn::TimeSpan timeSpan)
{
    NNT_OS_DETAIL_LOG("Sleep %ld ns\n", timeSpan.GetNanoSeconds());
    nn::os::SleepThread(timeSpan);
}

void CommandProcessor::Fill(uintptr_t address, size_t size)
{
    NN_ABORT_UNLESS(address % sizeof(int32_t) == 0);
    NN_ABORT_UNLESS(size % sizeof(int32_t) == 0);
    NN_ABORT_UNLESS(size > 0);

    int32_t* p = reinterpret_cast<int32_t*>(address);
    size_t count = size / sizeof(int32_t);

    int32_t sum = 0;
    for (int i = 1; i < count; i++)
    {
        p[i] = static_cast<int32_t>(m_Mt.GenerateRandomU32());
        sum += p[i];
    }
    p[0] = -sum;
}

bool CommandProcessor::Verify(uintptr_t address, size_t size)
{
    NN_ABORT_UNLESS(address % sizeof(int32_t) == 0);
    NN_ABORT_UNLESS(size % sizeof(int32_t) == 0);
    NN_ABORT_UNLESS(size > 0);

    int32_t* p = reinterpret_cast<int32_t*>(address);
    size_t count = size / sizeof(int32_t);

    int32_t sum = 0;
    for (int i = 0; i < count; i++)
    {
        sum += p[i];
    }

    return sum == 0;
}

void ConfirmResourceUsage(size_t maxResourceUsage)
{
    auto usage = nn::os::GetVirtualAddressMemoryResourceUsage();
    NN_LOG("Resource usage: %zu / %zu\n", usage.usedSize, usage.assignedSize);
    EXPECT_LE(usage.usedSize, maxResourceUsage);
}

}}} // namespace nnt::os::VirtualAddressMemory
