﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <vector>
#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/os/os_VirtualAddressMemory.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_TinyMt.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_Result.h>
#include <nn/os/os_Random.h>
#include <nn/os/os_Thread.h>

#include <nnt.h>

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
#pragma warning( disable : 4146 )
#endif

#define NNT_OS_DETAIL_LOG(...)

namespace nn { namespace os {

//--------------------------------------------------------------------------
/**
 * @brief   仮想アドレス空間からアドレス領域を確保し、その領域に物理メモリを割り当てます。
 *
 * @param[out]  pOutAddress 確保したアドレス領域の先頭アドレスを格納する領域へのポインタ
 * @param[in]   size        確保するアドレス領域のサイズ
 *
 * @retresult
 *   @handleresult{nn::os::ResultOutOfVirtualAddressSpace,仮想アドレス空間不足によりアドレス領域の確保に失敗しました。}
 *   @handleresult{nn::os::ResultOutOfMemory,仮想アドレス空間管理に必要な物理メモリの確保に失敗しました。もしくは、割り当てる物理メモリの確保に失敗しました。}
 *   @handleresult{nn::os::ResultOutOfResource,物理メモリ操作に必要なリソースが不足しました。}
 * @endretresult
 *
 * @pre
 *  - nn::os::IsVirtualAddressMemoryEnabled() == true
 *  - pOutAddress が指す領域が書き込み可能
 *  - size > 0
 *  - size が nn::os::MemoryPageSize の整数倍である
 *
 * @post
 *  - サイズが size バイトのアドレス領域が確保され、その先頭アドレスが *pOutAddress に格納されている
 *  - *pOutAddress に格納されているアドレスが nn::os::AddressRegionAlignment でアライメントされている
 *  - *pOutAddress から size バイトの領域に物理メモリが割り当てられている
 *
 * @details
 *  size バイトの新たなアドレス領域を仮想アドレス空間上に確保し、その領域に物理メモリを割り当て、
 *  その領域の先頭アドレスを返します。
 *  確保されるアドレス領域の先頭アドレスは nn::os::AddressRegionAlignment 定数が示す値でアライメントされます。
 *
 *  以下の処理と同じ処理を行います。
 *      @code
 *      uintptr_t address;
 *      AllocateAddressRegion(&address, size);
 *      AllocateMemoryPages(address, size);
 *      @endcode
 *
 *  この関数で割り当てられた物理メモリの割り当てを解除するには nn::os::FreeMemoryPages を使用します。@n
 *  この関数で確保したアドレス領域を解放するには nn::os::FreeAddressRegion を使用します。
 *
 *  この関数はスレッドセーフです。
 *
 *  この関数は nn::os::IsVirtualAddressMemoryEnabled() が true を返す場合にのみ使用可能です。
 *  詳細は nn::os::IsVirtualAddressMemoryEnabled() を参照してください。
 */
nn::Result  AllocateMemory(uintptr_t* pOutAddress, size_t size) NN_NOEXCEPT;



}}

namespace nnt { namespace os { namespace VirtualAddressMemory {

//---------------------------------------------------------------------------
//  VirtualAddressMemory 機能のテスト
//---------------------------------------------------------------------------

// 64bit のみ対応
#if defined(NN_OS_CPU_ARM_AARCH64_ARMV8A) || defined(NN_BUILD_CONFIG_CPU_X64)

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
static size_t s_FirstUsageUsedSize;
TEST(VirtualAddressMemoryTest, FirstUsage)
{
    auto varu = nn::os::GetVirtualAddressMemoryResourceUsage();

    NN_LOG("%zu KB   %zu KB\n", varu.assignedSize / 1024, varu.usedSize / 1024);
    EXPECT_TRUE( varu.assignedSize == 2 * 1024 * 1024 );
    EXPECT_TRUE( varu.usedSize     >=      544 * 1024 );
    EXPECT_TRUE( varu.usedSize     <=      600 * 1024 );

    s_FirstUsageUsedSize = varu.usedSize;
}
#endif

//
// 単純なユースケースのテスト
//
TEST(VirtualAddressMemoryTest, BasicUsage)
{
    const int BlockNumMax = 128;
    uintptr_t allocAddr[BlockNumMax];
    size_t    allocSize[BlockNumMax];
    nn::os::MemoryInfo memInfo;
    nn::util::TinyMt mt;

    mt.Initialize(0);

    nn::os::QueryMemoryInfo(&memInfo);
    NN_LOG("Initial: Used memory size: 0x%zx\n", memInfo.totalUsedMemorySize);

    // アドレス空間を確保
    for (int i=0; i<BlockNumMax; ++i)
    {
        allocSize[i] = nn::os::MemoryPageSize * (mt.GenerateRandomU32() % 10 + 1);
        auto result = nn::os::AllocateAddressRegion(&allocAddr[i], allocSize[i]);
        NNT_EXPECT_RESULT_SUCCESS(result);
        EXPECT_TRUE(nn::util::is_aligned(allocAddr[i], nn::os::AddressRegionAlignment));
        NNT_OS_DETAIL_LOG("allocAddr[%d]=0x%zx\n", i, allocAddr[i]);
    }

    nn::os::QueryMemoryInfo(&memInfo);
    NN_LOG("After AllocateAddressRegion: Used memory size: 0x%zx\n", memInfo.totalUsedMemorySize);

    // 物理メモリを割り当て
    for (int i=0; i<BlockNumMax; ++i)
    {
        for (size_t j=0; j<allocSize[i] / nn::os::MemoryPageSize; ++j)
        {
            // 1ページずつ物理メモリを割り当ててみる
            auto result = nn::os::AllocateMemoryPages(allocAddr[i] + j * nn::os::MemoryPageSize, nn::os::MemoryPageSize);
            NNT_EXPECT_RESULT_SUCCESS(result);
        }
        NNT_OS_DETAIL_LOG("memset(0x%zx, 0x5a, %d)\n", allocAddr[i], allocSize[i]);

        // 物理メモリが割り当てられていなければ例外になるはず
        memset(reinterpret_cast<void*>(allocAddr[i]), 0x5a, allocSize[i]);
    }

    nn::os::QueryMemoryInfo(&memInfo);
    NN_LOG("After AllocateMemoryPages: Used memory size: 0x%zx\n", memInfo.totalUsedMemorySize);

    // 物理メモリを解放
    for (int i=0; i<BlockNumMax; ++i)
    {
        auto result = nn::os::FreeMemoryPages(allocAddr[i], allocSize[i]);
        NNT_EXPECT_RESULT_SUCCESS(result);
    }

    nn::os::QueryMemoryInfo(&memInfo);
    NN_LOG("After FreeMemoryPages: Used memory size: 0x%zx\n", memInfo.totalUsedMemorySize);

    // 空間を解放
    for (int i=0; i<BlockNumMax; ++i)
    {
        nn::os::FreeAddressRegion(allocAddr[i]);
    }

    nn::os::QueryMemoryInfo(&memInfo);
    NN_LOG("After FreeAddressRegion: Used memory size: 0x%zx\n", memInfo.totalUsedMemorySize);

    // アドレス空間と物理メモリを同時に確保
    for (int i=0; i<BlockNumMax; ++i)
    {
        auto result = nn::os::AllocateMemory(&allocAddr[i], allocSize[i]);
        NNT_EXPECT_RESULT_SUCCESS(result);
        EXPECT_TRUE(nn::util::is_aligned(allocAddr[i], nn::os::AddressRegionAlignment));
        NNT_OS_DETAIL_LOG("allocAddr[%d]=0x%zx\n", i, allocAddr[i]);
    }

    nn::os::QueryMemoryInfo(&memInfo);
    NN_LOG("After AllocateMemory: Used memory size: 0x%zx\n", memInfo.totalUsedMemorySize);

    // 空間を解放
    for (int i=0; i<BlockNumMax; ++i)
    {
        nn::os::FreeAddressRegion(allocAddr[i]);
    }

    nn::os::QueryMemoryInfo(&memInfo);
    NN_LOG("After FreeAddressRegion: Used memory size: 0x%zx\n", memInfo.totalUsedMemorySize);
}

//
// アドレス空間全体を確保できるか
//
TEST(VirtualAddressMemoryTest, LargeRegionAlloc)
{
    uintptr_t addr;

    // 64GB - 1GB - 1GB 空間を確保できるはず
    auto result = nn::os::AllocateAddressRegion(&addr,
        64ull * 1024 * 1024 * 1024 - 1 * 1024 * 1024 * 1024 - 1 * 1024 * 1024 * 1024);
    NNT_EXPECT_RESULT_SUCCESS(result);

    nn::os::FreeAddressRegion(addr);
}

namespace
{
    static const int AddressBufferCount = 16 * 1024 * 1024;
    static uintptr_t    s_Address[AddressBufferCount];

    void FreeAddressRegionBusyTest(bool busy1, bool busy2, bool busy3)
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        uintptr_t address;
        nn::os::TransferMemoryType tm1;
        nn::os::TransferMemoryType tm2;
        nn::os::TransferMemoryType tm3;

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemory(&address, nn::os::MemoryPageSize * 3) );

        if( busy1 )
        {
            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateTransferMemory(&tm1,
                    reinterpret_cast<void*>(address + nn::os::MemoryPageSize * 0),
                    nn::os::MemoryPageSize, nn::os::MemoryPermission_ReadWrite) );
        }
        if( busy2 )
        {
            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateTransferMemory(&tm2,
                    reinterpret_cast<void*>(address + nn::os::MemoryPageSize * 1),
                    nn::os::MemoryPageSize, nn::os::MemoryPermission_ReadWrite) );
        }
        if( busy3 )
        {
            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateTransferMemory(&tm3,
                    reinterpret_cast<void*>(address + nn::os::MemoryPageSize * 2),
                    nn::os::MemoryPageSize, nn::os::MemoryPermission_ReadWrite) );
        }

        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultBusy,
            nn::os::FreeAddressRegion(address) );

        if( busy3 )
        {
            nn::os::DestroyTransferMemory(&tm3);
        }
        if( busy2 )
        {
            nn::os::DestroyTransferMemory(&tm2);
        }
        if( busy1 )
        {
            nn::os::DestroyTransferMemory(&tm1);
        }

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeAddressRegion(address) );
#else
        NN_UNUSED(busy1);
        NN_UNUSED(busy2);
        NN_UNUSED(busy3);
#endif
    }

    void FreeMemoryPagesBusyTest(uintptr_t address, bool busy1, bool busy2, bool busy3)
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::os::TransferMemoryType tm1;
        nn::os::TransferMemoryType tm2;
        nn::os::TransferMemoryType tm3;

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address, nn::os::MemoryPageSize * 3) );

        if( busy1 )
        {
            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateTransferMemory(&tm1,
                    reinterpret_cast<void*>(address + nn::os::MemoryPageSize * 0),
                    nn::os::MemoryPageSize, nn::os::MemoryPermission_ReadWrite) );
        }
        if( busy2 )
        {
            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateTransferMemory(&tm2,
                    reinterpret_cast<void*>(address + nn::os::MemoryPageSize * 1),
                    nn::os::MemoryPageSize, nn::os::MemoryPermission_ReadWrite) );
        }
        if( busy3 )
        {
            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateTransferMemory(&tm3,
                    reinterpret_cast<void*>(address + nn::os::MemoryPageSize * 2),
                    nn::os::MemoryPageSize, nn::os::MemoryPermission_ReadWrite) );
        }

        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultBusy,
            nn::os::FreeMemoryPages(address, nn::os::MemoryPageSize * 3) );

        if( busy3 )
        {
            nn::os::DestroyTransferMemory(&tm3);
        }
        if( busy2 )
        {
            nn::os::DestroyTransferMemory(&tm2);
        }
        if( busy1 )
        {
            nn::os::DestroyTransferMemory(&tm1);
        }

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeMemoryPages(address, nn::os::MemoryPageSize * 3) );
#else
        NN_UNUSED(address);
        NN_UNUSED(busy1);
        NN_UNUSED(busy2);
        NN_UNUSED(busy3);
#endif
    }
}

TEST(VirtualAddressMemoryTest, OtherApiTest)
{
    //------------------------------------------
    // SetMemoryPermission
    {
        uintptr_t address;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemory(&address, nn::os::MemoryPageSize * 4) );

        nn::os::SetMemoryPermission(address + nn::os::MemoryPageSize * 0,
            nn::os::MemoryPageSize, nn::os::MemoryPermission_None );
        nn::os::SetMemoryPermission(address + nn::os::MemoryPageSize * 1,
            nn::os::MemoryPageSize, nn::os::MemoryPermission_ReadOnly );
        nn::os::SetMemoryPermission(address + nn::os::MemoryPageSize * 2,
            nn::os::MemoryPageSize, nn::os::MemoryPermission_ReadWrite );

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeAddressRegion(address) );
    }
}

TEST(VirtualAddressMemoryTest, BoundaryTest)
{
    uintptr_t   address;

    //------------------------------------------
    // AllocateAddressRegion
    {
        // ResultSuccess
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateAddressRegion(&address, nn::os::MemoryPageSize) );

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeAddressRegion(address) );

        // ResultOutOfVirtualAddressSpace
        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultOutOfVirtualAddressSpace,
            nn::os::AllocateAddressRegion(&address, 64ull * 1024 * 1024 * 1024) );


        // ResultOutOfMemory

        // 管理領域用のメモリは一度増えたら減らないので
        // 他のアドレス領域を大量に確保する系のテストより前に行わなければならない

        // ギリギリまでメモリ確保する
        nn::os::MemoryInfo mi;
        nn::os::QueryMemoryInfo(&mi);
        const size_t allocatableSize = mi.totalAvailableMemorySize - mi.totalUsedMemorySize;

        uintptr_t paddingAddress;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemory(&paddingAddress, allocatableSize) );

        // アドレス領域を大量に作ってメモリ不足を起こさせる
        int index;
        nn::Result result;
        for( index = 0; index < AddressBufferCount; ++index )
        {
            result = nn::os::AllocateAddressRegion(&address, nn::os::MemoryPageSize);
            if( result.IsSuccess() )
            {
                s_Address[index] = address;
            }
            else
            {
                break;
            }
        }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultOutOfMemory,
            result );
#else
        // Windows だと空間が先に枯渇する
        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultOutOfVirtualAddressSpace,
            result );
#endif

        for( int j = 0; j < index; j++ )
        {
            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::FreeAddressRegion(s_Address[j]) );
        }

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeAddressRegion(paddingAddress) );
    }

    //------------------------------------------
    // FreeAddressRegion
    {
        // ResultSuccess

        // ResultInvalidParameter
        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultInvalidParameter,
            nn::os::FreeAddressRegion(0) );

        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultInvalidParameter,
            nn::os::FreeAddressRegion(-nn::os::MemoryPageSize) );

        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultInvalidParameter,
            nn::os::FreeAddressRegion(~0ull) );

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateAddressRegion(&address, nn::os::MemoryPageSize) );

        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultInvalidParameter,
            nn::os::FreeAddressRegion(address + 1) );

        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultInvalidParameter,
            nn::os::FreeAddressRegion(address - 1) );

        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultInvalidParameter,
            nn::os::FreeAddressRegion(address + nn::os::MemoryPageSize) );

        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultInvalidParameter,
            nn::os::FreeAddressRegion(address - nn::os::MemoryPageSize) );

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeAddressRegion(address) );

        // ResultBusy
        FreeAddressRegionBusyTest(true,  false, false);
        FreeAddressRegionBusyTest(false, false, true );
        FreeAddressRegionBusyTest(false, true,  false);
        FreeAddressRegionBusyTest(true,  false, true);

        // ResultOutOfResource
        //
    }


    //------------------------------------------
    // AllocateMemory
    {
        // ResultSuccess

        // ResultOutOfVirtualAddressSpace
        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultOutOfVirtualAddressSpace,
            nn::os::AllocateMemory(&address, 64ull * 1024 * 1024 * 1024) );

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        // ResultOutOfMemory
        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultOutOfMemory,
            nn::os::AllocateMemory(&address, 8ull * 1024 * 1024 * 1024) );
#endif

/*
        // ResultOutOfResource
        // 4KB 単位で確保してメモリ不足を起こさせる
        bool isTestSuccess = false;
        int lastIndex = 0;
        for( int i = 0; i < AddressBufferCount; ++i )
        {
            auto result = nn::os::AllocateMemory(&address, nn::os::MemoryPageSize);
            if( result.IsSuccess() )
            {
                s_Address[i] = address;
            }
            else
            {
                NNT_EXPECT_RESULT_FAILURE(
                    nn::os::ResultOutOfResource,
                    result );

                isTestSuccess = true;
                break;
            }

            lastIndex = i;
        }

        for( int j = 0; j <= lastIndex; j++ )
        {
            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::FreeAddressRegion(s_Address[j]) );
        }

        EXPECT_TRUE(isTestSuccess);
*/
    }

    //------------------------------------------
    // AllocateMemoryPages
    {
        const size_t size = 32ull * 1024 * 1024 * 1024;

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateAddressRegion(&address, size) );

        const uintptr_t addressEnd = address + size;

        // ResultSuccess
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address, nn::os::MemoryPageSize) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeMemoryPages(address, nn::os::MemoryPageSize) );

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(addressEnd - nn::os::MemoryPageSize, nn::os::MemoryPageSize) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeMemoryPages(addressEnd - nn::os::MemoryPageSize, nn::os::MemoryPageSize) );

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address, nn::os::MemoryPageSize) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address, nn::os::MemoryPageSize * 2) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeMemoryPages(address, nn::os::MemoryPageSize * 2) );

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address + nn::os::MemoryPageSize, nn::os::MemoryPageSize) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address, nn::os::MemoryPageSize * 2) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeMemoryPages(address, nn::os::MemoryPageSize * 2) );

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address + nn::os::MemoryPageSize * 1, nn::os::MemoryPageSize) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address, nn::os::MemoryPageSize * 3) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeMemoryPages(address, nn::os::MemoryPageSize * 3) );

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address + nn::os::MemoryPageSize * 0, nn::os::MemoryPageSize) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address + nn::os::MemoryPageSize * 2, nn::os::MemoryPageSize) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address, nn::os::MemoryPageSize * 3) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeMemoryPages(address, nn::os::MemoryPageSize * 3) );


        // ResultInvalidParameter
        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultInvalidParameter,
            nn::os::AllocateMemoryPages(addressEnd, nn::os::MemoryPageSize) );

        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultInvalidParameter,
            nn::os::AllocateMemoryPages(addressEnd - nn::os::MemoryPageSize, nn::os::MemoryPageSize * 2) );

        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultInvalidParameter,
            nn::os::AllocateMemoryPages(address - nn::os::MemoryPageSize, nn::os::MemoryPageSize * 2) );

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        // ResultOutOfMemory
        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultOutOfMemory,
            nn::os::AllocateMemoryPages(address, size) );
#endif

/*
        // ResultOutOfResource
        // 4KB 単位で確保してメモリ不足を起こさせる
        bool isTestSuccess = false;
        int lastIndex = 0;
        for( int i = 0; i < AddressBufferCount; ++i )
        {
            const uintptr_t a = address + i * nn::os::MemoryPageSize * 2;
            auto result = nn::os::AllocateMemoryPages(a, nn::os::MemoryPageSize);
            if( result.IsFailure() )
            {
                NNT_EXPECT_RESULT_FAILURE(
                    nn::os::ResultOutOfResource,
                    result );

                isTestSuccess = true;
                break;
            }

            lastIndex = i;
        }

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeMemoryPages(address, size) );

        EXPECT_TRUE(isTestSuccess);
*/

        //
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeAddressRegion(address) );
    }


    //------------------------------------------
    // FreeMemoryPages
    {
        const size_t size = 32ull * 1024 * 1024 * 1024;

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateAddressRegion(&address, size) );

        const uintptr_t addressEnd = address + size;

        // ResultSuccess
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address, nn::os::MemoryPageSize) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeMemoryPages(address, nn::os::MemoryPageSize * 2) );

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address + nn::os::MemoryPageSize, nn::os::MemoryPageSize) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeMemoryPages(address, nn::os::MemoryPageSize * 2) );

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address + nn::os::MemoryPageSize, nn::os::MemoryPageSize) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeMemoryPages(address, nn::os::MemoryPageSize * 3) );

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address + nn::os::MemoryPageSize * 0, nn::os::MemoryPageSize) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(address + nn::os::MemoryPageSize * 2, nn::os::MemoryPageSize) );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeMemoryPages(address, nn::os::MemoryPageSize * 3) );


        // ResultInvalidParameter
        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultInvalidParameter,
            nn::os::FreeMemoryPages(addressEnd, nn::os::MemoryPageSize) );

        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultInvalidParameter,
            nn::os::FreeMemoryPages(addressEnd - nn::os::MemoryPageSize, nn::os::MemoryPageSize * 2) );

        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultInvalidParameter,
            nn::os::FreeMemoryPages(address - nn::os::MemoryPageSize, nn::os::MemoryPageSize * 2) );

        // ResultBusy
        FreeMemoryPagesBusyTest(address, true,  false, false);
        FreeMemoryPagesBusyTest(address, false, false, true );
        FreeMemoryPagesBusyTest(address, false, true,  false);
        FreeMemoryPagesBusyTest(address, true,  false, true);


        // ResultOutOfResource
        // 4KB 単位で確保してメモリ不足を起こさせる
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateMemoryPages(addressEnd - nn::os::MemoryPageSize * 512, nn::os::MemoryPageSize * 512) );

/*
        bool isTestSuccess = false;
        for( int i = 0; i < AddressBufferCount; ++i )
        {
            const uintptr_t a = address + i * nn::os::MemoryPageSize * 2;
            auto result = nn::os::AllocateMemoryPages(a, nn::os::MemoryPageSize);
            if( result.IsFailure() )
            {
                NNT_EXPECT_RESULT_FAILURE(
                    nn::os::ResultOutOfResource,
                    result );

                isTestSuccess = true;
                break;
            }
        }

        NNT_EXPECT_RESULT_FAILURE(
            nn::os::ResultOutOfResource,
            nn::os::FreeMemoryPages(addressEnd - nn::os::MemoryPageSize * 512, nn::os::MemoryPageSize) );
*/

        //
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeAddressRegion(address) );
    }
} // NOLINT(impl/function_size)

TEST(VirtualAddressMemoryTest, AlignmentTest)
{
    struct TestSet
    {
        size_t  size;
        size_t  alignment;
    };

    static const size_t k = 1024;
    static const size_t m = 1024 * k;
    static const size_t g = 1024 * m;
    static const TestSet tests[] =
    {
        // size             alignment
        {   4 * k,          64 * k },
        {  64 * k - 4 * k,  64 * k },
        {  64 * k,          64 * k },
        {   2 * m - 4 * k,  64 * k },
        {   2 * m,           2 * m },
        {   4 * m - 4 * k,   2 * m },
        {   4 * m,           4 * m },
        {  32 * m - 4 * k,   4 * m },
        {  32 * m,          32 * m },
        {   1 * g - 4 * k,  32 * m },
        {   1 * g,           1 * g },
    };

    uintptr_t* pAddressStore = s_Address;
    for( auto& test : tests )
    {
        for( int i = 0; i < 16; ++i )
        {
            uintptr_t address;

            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::AllocateAddressRegion(&address, test.size) );

            EXPECT_TRUE( (address % nn::os::AddressRegionAlignment) == 0 );
            EXPECT_TRUE( (address % test.alignment) == 0 );

            *pAddressStore++ = address;
        }
    }

    for( uintptr_t* pAddress = s_Address; pAddress < pAddressStore; pAddress++ )
    {
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeAddressRegion(*pAddress) );
    }

} // NOLINT(impl/function_size)

TEST(VirtualAddressMemoryTest, PatternTest)
{
    const int testCount = 1024 * 1024;
    NN_STATIC_ASSERT( testCount <= AddressBufferCount );

    // 1 つ Allocate 後、「1 つ Allocate、2 つ前を Free」を繰り返す
    {
        uintptr_t address1;
        uintptr_t address2;

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::AllocateAddressRegion(&address1, nn::os::MemoryPageSize) );

        for( int i = 0; i < testCount; ++i )
        {
            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::AllocateAddressRegion(&address2, nn::os::MemoryPageSize) );
            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::FreeAddressRegion(address1) );
            address1 = address2;
        }

        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::FreeAddressRegion(address2) );
    }

    //
}

TEST(VirtualAddressMemoryTest, RandomTest)
{
    const int testCount = 1024 * 1024;
    NN_STATIC_ASSERT( testCount <= AddressBufferCount );

    {
        std::vector<uintptr_t> list;
        nn::util::TinyMt tmt;
        uintptr_t address;

        // ランダムに 16～31 個のアドレス領域を作る
        {
            nn::Bit32 rand;
            nn::os::GenerateRandomBytes(&rand, sizeof(rand));
            tmt.Initialize(rand);

            int n = tmt.GenerateRandomN(16);
            for( int i = 0; i < 16 + n; ++i )
            {
                NNT_EXPECT_RESULT_SUCCESS(
                    nn::os::AllocateAddressRegion(&address, nn::os::MemoryPageSize) );
                list.push_back(address);
            }
        }

        for( int x = 0; x < testCount; ++x )
        {
            // ランダムにランダムな個数のアドレス領域を解放する
            {
                int n = tmt.GenerateRandomN(static_cast<uint16_t>(list.size() + 1));
                for( int i = 0; i < n; ++i )
                {
                    int index = tmt.GenerateRandomN(static_cast<uint16_t>(list.size()));
                    NNT_EXPECT_RESULT_SUCCESS(
                        nn::os::FreeAddressRegion(list[index]) );
                    list.erase(list.begin() + index);
                }
            }

            // ランダムに 0～15 個のアドレス領域を作る
            {
                int n = tmt.GenerateRandomN(16);
                for( int i = 0; i < n; ++i )
                {
                    NNT_EXPECT_RESULT_SUCCESS(
                        nn::os::AllocateAddressRegion(&address, nn::os::MemoryPageSize) );
                    EXPECT_TRUE(nn::util::is_aligned(address, nn::os::AddressRegionAlignment));
                    list.push_back(address);
                }
            }
        }

        // お片付け
        {
            while( !list.empty() )
            {
                NNT_EXPECT_RESULT_SUCCESS(
                    nn::os::FreeAddressRegion(list.back()) );
                list.pop_back();
            }
        }
    }
}

namespace
{
    const int ThreadCount = 4;

    void MultiThreadTestBody(void* pEvent)
    {
        nn::os::Event& startEvent = *reinterpret_cast<nn::os::Event*>(pEvent);

        static const int RegionCount = 64;
        static const int LoopCount = 256;
        uintptr_t addressArray[RegionCount];
        nn::util::TinyMt tmt;
        uintptr_t address;

        int regionCount;

        // ランダムに 16～31 個のアドレス領域を作る
        {
            nn::Bit32 rand;
            nn::os::GenerateRandomBytes(&rand, sizeof(rand));
            tmt.Initialize(rand);

            int n = tmt.GenerateRandomN(16);
            for( int i = 0; i < 16 + n; ++i )
            {
                NNT_EXPECT_RESULT_SUCCESS(
                    nn::os::AllocateAddressRegion(&address, nn::os::MemoryPageSize) );
                addressArray[i] = address;
            }

            regionCount = n;
        }

        startEvent.Wait();

        for( int loopCount = 0; loopCount < LoopCount; ++loopCount )
        {
            for( ; regionCount < RegionCount; ++regionCount )
            {
                NNT_EXPECT_RESULT_SUCCESS(
                    nn::os::AllocateAddressRegion(&address, nn::os::MemoryPageSize) );

                addressArray[regionCount] = address;
            }

            // RegionCount から RegionCount - 1 にする
            --regionCount;

            for( ; regionCount >= 0; --regionCount )
            {
                NNT_EXPECT_RESULT_SUCCESS(
                    nn::os::FreeAddressRegion(addressArray[regionCount]) );
            }

            // -1 から 0 にする
            ++regionCount;
        }
    }

    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_Stack[ThreadCount][4 * 1024];
}

TEST(VirtualAddressMemoryTest, MultiThreadTest)
{
    nn::os::Event startEvent(nn::os::EventClearMode_ManualClear);
    nn::os::ThreadType threads[ThreadCount];

    for( int i = 0; i < ThreadCount; ++i )
    {
        NNT_EXPECT_RESULT_SUCCESS(
            nn::os::CreateThread(
                &threads[i],
                MultiThreadTestBody,
                &startEvent,
                g_Stack[i],
                sizeof(g_Stack[i]),
                10,
                (i % 3) ) );
        nn::os::StartThread(&threads[i]);
    }

    startEvent.Signal();

    for( int i = 0; i < ThreadCount; ++i )
    {
        nn::os::WaitThread(&threads[i]);
        nn::os::DestroyThread(&threads[i]);
    }
}

//
// 細かい空間を大量に確保できるか
// ※ 空間管理用の NumberLineAllocaotr の管理領域用メモリが不足して止まる
//
uintptr_t smallAddr[0x1000000];

TEST(VirtualAddressMemoryTest, SmallRegionAlloc)
{
    int count = 0;
    nn::Result result = nn::ResultSuccess();

    // 4KB 空間を大量に確保する
    for (;;)
    {
        result = nn::os::AllocateAddressRegion(&smallAddr[count], nn::os::MemoryPageSize);
        if (result.IsFailure())
        {
            break;
        }
        ++count;
    }

    NN_LOG("Allocated address region: 4KB * %d = %llu\n", count, 4096ull * count);

    for (int i=0; i<count; ++i)
    {
        nn::os::FreeAddressRegion(smallAddr[i]);
    }
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
TEST(VirtualAddressMemoryTest, FinalUsage)
{
    auto varu = nn::os::GetVirtualAddressMemoryResourceUsage();

    NN_LOG("%zu KB   %zu KB  (first usage: %zu KB)\n", varu.assignedSize / 1024, varu.usedSize / 1024, s_FirstUsageUsedSize / 1024);
    EXPECT_TRUE( varu.assignedSize == 2 * 1024 * 1024 );
    EXPECT_TRUE( varu.usedSize     >= s_FirstUsageUsedSize);
    EXPECT_TRUE( varu.usedSize     <=      712 * 1024 );
}
#endif


#else // 32bit の場合は ResultNotSupported になることを確認

TEST(VirtualAddressMemoryTest, NotSupport32bit)
{
    uintptr_t addr = 0;
    nn::Result result;

    result = nn::os::AllocateAddressRegion(&addr, nn::os::MemoryPageSize);
    EXPECT_TRUE(result <= nn::os::ResultNotSupported());

    result = nn::os::AllocateMemoryPages(addr, nn::os::MemoryPageSize);
    EXPECT_TRUE(result <= nn::os::ResultNotSupported());

    result = nn::os::AllocateMemory(&addr, nn::os::MemoryPageSize);
    EXPECT_TRUE(result <= nn::os::ResultNotSupported());

    result = nn::os::FreeMemoryPages(addr, nn::os::MemoryPageSize);
    EXPECT_TRUE(result <= nn::os::ResultNotSupported());

    result = nn::os::FreeAddressRegion(addr);
    EXPECT_TRUE(result <= nn::os::ResultNotSupported());
}

#endif

}}} // namespace nnt::os::VirtualAddressMemory
