﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Thread 専用のテストヘルパー
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/os/os_SdkThread.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "test_ThreadHelper.h"

#include <nnt/nntest.h>

//---------------------------------------------------------------------------

namespace nnt { namespace os { namespace thread {

int             g_seq;
ThreadHelper    g_ThrHelper;

NN_ALIGNAS( 4096 ) char g_StackOfThread1[ 16384 ];
NN_ALIGNAS( 4096 ) char g_StackOfThread2[ 16384 ];


//---------------------------------------------------------------------------
// スレッド１の生成
void ThreadHelper::CreateThread1( TestThreadAction state )
{
    SEQ_NONE();
    NNT_OS_LOG("Create Thread1:");

    NN_UNUSED( state );

    nn::Result err = nn::os::CreateThread(
                        &m_Thread1,                     // Thread オブジェクト
                        &ThreadFunc1,                   // エントリ関数
                        reinterpret_cast<void*>(0x1111),// 引数
                        g_StackOfThread1,               // スタック
                        16384,                          // スタックサイズ
                        priThread1);                    // 優先度

    CheckBool( err.IsSuccess() );
}

// スレッド２の生成
void ThreadHelper::CreateThread2( TestThreadAction state )
{
    SEQ_NONE();
    NNT_OS_LOG("Create Thread2:");

    NN_UNUSED( state );

    nn::Result err = nn::os::CreateThread(
                      &m_Thread2,                     // Thread オブジェクト
                      &ThreadFunc2,                   // エントリ関数
                      reinterpret_cast<void*>(0x2222),// 引数
                      g_StackOfThread2,               // スタック
                      16384,                          // スタックサイズ
                      priThread2);                    // 優先度

    CheckBool( err.IsSuccess() );
}

//---------------------------------------------------------------------------
// スレッド１の破棄
void ThreadHelper::DestroyThread1( TestThreadAction state )
{
    NN_UNUSED( state );

    SEQ_NONE();
    NNT_OS_LOG("Destroy Thread1:");
    nn::os::DestroyThread( &m_Thread1 );

    CheckBool( true );
}

// スレッド２の破棄
void ThreadHelper::DestroyThread2( TestThreadAction state )
{
    NN_UNUSED( state );

    SEQ_NONE();
    NNT_OS_LOG("Destroy Thread2:");
    nn::os::DestroyThread( &m_Thread2 );

    CheckBool( true );
}


//---------------------------------------------------------------------------
// Thread オブジェクトの Initialize
void ThreadHelper::InitializeThreadParameter(int priority)
{
    SEQ_NONE();
    NNT_OS_LOG("Initialize Thread1 Priority=%d", priority);

    nn::os::ChangeThreadPriority( &m_Thread1,   // Thread1 オブジェクト
                                  priority );   // 優先度
    CheckBool( true );
}

//---------------------------------------------------------------------------
// Thread オブジェクトのクリーンナップ（待ちスレッドを全解除）
void ThreadHelper::Cleanup()
{
    // Cleanup 後の状態へ
    m_isCleanuped = true;

    // ２つのスレッドが終了するまで待つ
    if (m_Thread2._state == nn::os::ThreadType::State_Started)
    {
        nn::os::WaitThread( &m_Thread2 );
    }
    if (m_Thread1._state == nn::os::ThreadType::State_Started)
    {
        nn::os::WaitThread( &m_Thread1 );
    }
}

//---------------------------------------------------------------------------
//  スレッド実行の開始
void ThreadHelper::StartThread1( TestThreadAction state )
{
    SEQ_NONE();
    if (state != THR_NOTUSE)
    {
        NNT_OS_LOG("Start Thread1:");
        nn::os::StartThread( &m_Thread1 );     // Thread オブジェクト
        CheckBool( true );
    }
    else
    {
        NNT_OS_LOG("Thread1 is not used\n");
    }
}

void ThreadHelper::StartThread2( TestThreadAction state )
{
    SEQ_NONE();
    if (state != THR_NOTUSE)
    {
        NNT_OS_LOG("Start Thread2:");
        nn::os::StartThread( &m_Thread2 );     // Thread オブジェクト
        CheckBool( true );
    }
    else
    {
        NNT_OS_LOG("Thread2 is not used\n");
    }
}

//---------------------------------------------------------------------------
// スレッドの内部状態のチェック（非シグナル状態にしてリターン）
void    ThreadHelper::ThreadCheck( int expectPriority )
{
    // Thread1 オブジェクトの内部状態を参照
    CheckParam( nn::os::GetThreadCurrentPriority(&m_Thread1), expectPriority );
}

//---------------------------------------------------------------------------
// スレッド状態のチェック
void ThreadHelper::ThreadStateCheck( const nn::os::ThreadType& pThread,
                                     TestThreadAction          action,
                                     TestThreadLastState       expect,
                                     bool                      apiResult,
                                     bool                      apiExpect )
{
    NN_UNUSED( action );

    // 現在のスレッド状態を表示
    switch (pThread._state)
    {
        case nn::os::ThreadType::State_Initialized:
            {
                NNT_OS_LOG(" INITIALIZED");
            }
            break;

        case nn::os::ThreadType::State_Started:
            {
                NNT_OS_LOG(" STARTED");
            }
            break;

        case nn::os::ThreadType::State_Exited:
            {
                NNT_OS_LOG(" EXITED");
            }
            break;

        case nn::os::ThreadType::State_NotInitialized:
            {
                NNT_OS_LOG(" NotInitialized");
            }
            break;

        default:
            {
                NNT_OS_LOG(" UNKNOWN");
            }
            break;
    }


    // スレッドの事後期待値を検査
    switch (expect)
    {
        // スレッド未使用
        case THR_STATE_NONE:
            {
                CheckParam( pThread._state, nn::os::ThreadType::State_Initialized );
            }
            break;

        // EXITED
        case THR_STATE_EXITED:
        case THR_STATE_WAITING2:
            {
                CheckParam( pThread._state, nn::os::ThreadType::State_Exited );
                NNT_OS_LOG("(-----): API return value:");
                CheckParam( apiResult, apiExpect );
            }
            break;

        // WAITING（待ちを継続）
        case THR_STATE_WAITING:
            {
                if (g_ThrHelper.GetThreadExpectState1() == THR_STATE_WAITING2)
                {
                    NNT_OS_LOG(" ... Not Checked\n");
                }
                else
                {
                    CheckParam( pThread._state, nn::os::ThreadType::State_Started );
                }
            }
            break;

        // タイムアウトを検知して終了
        case THR_STATE_SLEEP:
            {
                CheckParam( pThread._state, nn::os::ThreadType::State_Exited );
                NNT_OS_LOG("(-----): API return value:");
                CheckParam( apiResult, apiExpect );
            }
            break;

        // 想定外のエラー（テスト不具合）
        default:
            {
                NN_ASSERT( false );
            }
            break;
    }
}


//---------------------------------------------------------------------------
//  OS-API 発行
//---------------------------------------------------------------------------

bool    ThreadHelper::CallThreadApi( TestThreadAction   action,
                                     int                seq,
                                     bool               expectResult)
{
    bool    result = true;

    NN_UNUSED( expectResult );

    SEQ_CHECK( seq );
    NNT_OS_LOG("Call ");

    // スレッド事前条件の検査
    switch (action)
    {
        // NOTUSE 系（スレッドを使用しない）
        case THR_NOTUSE:
            {
                NN_ASSERT( false );
            }
            break;

        // WaitThread() 待ち状態（無期限）
        case THR_WAIT_WAIT:
            {
                NNT_OS_LOG("WaitThread() goes WAIT\n");
                nn::os::WaitThread( &m_Thread1 );
            }
            break;

        // SleepThread() 待ち状態（時限あり）
        case THR_WAIT_SLEEP:
            {
                NNT_OS_LOG("SleepThread() goes WAIT\n");
                nn::os::SleepThread( nn::TimeSpan::FromNanoSeconds( NNT_CALIBRATE_RATE() * 10 ) );
                result = false;
            }
            break;


        // WaitThread() の発行
        case THR_CALL_WAIT:
            {
                NNT_OS_LOG("WaitThread()\n");
                nn::os::WaitThread( &m_Thread1 );
            }
            break;

        // SetThreadName() の発行
        case THR_CALL_SET_NAME:
            {
                const char* name = "HogeThread";
                NNT_OS_LOG("SetThreadName()\n");
                nn::os::SetThreadName( &m_Thread1, name );
            }
            break;

        // GetThreadName() の発行（デフォルト名と比較）
        case THR_CALL_GET_NAME_DFLT:
            {
                NNT_OS_LOG("GetThreadNamePointer():");
                const char* name = nn::os::GetThreadNamePointer( &m_Thread1 );
                NNT_OS_LOG("name=\x22%s\x22\n", name);
                result = (std::strncmp(name, "Thread", 6) == 0) ? true : false;
            }
            break;

        // GetThreadName() の発行（SET_NAME したものと比較）
        case THR_CALL_GET_NAME_SET:
            {
                NNT_OS_LOG("GetThreadNamePointer():");
                const char* name = nn::os::GetThreadNamePointer( &m_Thread1 );
                NNT_OS_LOG("name=\x22%s\x22\n", name);
                result = (std::strcmp(name, "HogeThread") == 0) ? true : false;
            }
            break;


        // ChangeThreadPriority( -1 ) の発行
        case THR_CALL_CHG_PRI_UDR:
            {
                NNT_OS_LOG("ChangeThreadPriority(pri=-1)\n");
                nn::os::ChangeThreadPriority( &m_Thread1, -1 );
                result = false;
            }
            break;

        // ChangeThreadPriority( 0 ) の発行
        case THR_CALL_CHG_PRI_HIGH:
            {
                NNT_OS_LOG("ChangeThreadPriority(pri=0)\n");
                nn::os::ChangeThreadPriority( &m_Thread1, 0 );
            }
            break;

        // ChangeThreadPriority( 31 ) の発行
        case THR_CALL_CHG_PRI_LOW:
            {
                NNT_OS_LOG("ChangeThreadPriority(pri=MAX)\n");
                nn::os::ChangeThreadPriority( &m_Thread1,
                                              nn::os::LowestThreadPriority );
            }
            break;

        // ChangeThreadPriority( 32 ) の発行
        case THR_CALL_CHG_PRI_OVR:
            {
                NNT_OS_LOG("ChangeThreadPriority(pri=MAX+1)\n");
                nn::os::ChangeThreadPriority( &m_Thread1,
                                              nn::os::LowestThreadPriority + 1);
                result = false;
            }
            break;

        // ChangeThreadPriority( HighestSystemThreadPriority() ) の発行
        case THR_CALL_CHG_PRI_SYS_HIGH:
            {
                int pri = nn::os::HighestSystemThreadPriority;
                NNT_OS_LOG("ChangeThreadPriority(pri=%d)\n", pri);
                nn::os::ChangeThreadPriority( &m_Thread1, pri );
            }
            break;

        // ChangeThreadPriority( LowestSystemThreadPriority ) の発行
        case THR_CALL_CHG_PRI_SYS_LOW:
            {
                int pri = nn::os::LowestSystemThreadPriority;
                NNT_OS_LOG("ChangeThreadPriority(pri=%d)\n", pri);
                nn::os::ChangeThreadPriority( &m_Thread1, pri );
            }
            break;

        // GetThreadCurrentPriority() の発行
        case THR_CALL_GET_PRI:
            {
                NNT_OS_LOG("GetThreadCurrentPriority():");
                int basePri = nn::os::GetThreadPriority( &m_Thread1 );
                int pri     = nn::os::GetThreadCurrentPriority( &m_Thread1 );
                NN_UNUSED(pri);
                NNT_OS_LOG("org=%d pri=%d\n", basePri, pri);
                if ( nn::os::GetCurrentThread() == &m_Thread1 )
                {
                    EXPECT_EQ( priThread1, basePri );
                }
                else
                {
                    EXPECT_EQ( priThread2, basePri );
                }
            }
            break;

        // YieldThread() の発行
        case THR_CALL_YLD:
            {
                NNT_OS_LOG("YieldThread()\n");
                nn::os::YieldThread();
            }
            break;


        // 想定外のエラー（テスト不具合）
        default:
            {
                NN_ASSERT( false );
            }
            break;
    }

    return result;
}   // NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  テスト用スレッド関数の本体
//---------------------------------------------------------------------------
void    ThreadFunc1(void* exinf)
{
    TestThreadAction    action      = g_ThrHelper.GetThreadAction1();
    TestThreadLastState expectState = g_ThrHelper.GetThreadExpectState1();
    bool                expect      = g_ThrHelper.GetThreadExpect1();
    bool                result;

    NN_UNUSED( exinf );

    SEQ_SYNC_WAIT( NNT_POINT_THREAD1_START );

    // スレッド起動
    SEQ_CHECK( g_seq + 1 );
    NNT_OS_LOG("Begin of thread1:");
    CheckBool( true );

    // Thread1 から API を発行
    g_ThrHelper.SetApiResult1( false );
    result = g_ThrHelper.CallThreadApi( action, g_seq + 2, expect );
    g_ThrHelper.SetApiResult1( result );

    // Cleanuped ならスレッド終了（THR_STATE_WAITING だった場合など）
    if ( g_ThrHelper.IsCleanuped() )
    {
        NNT_OS_LOG("(-----): Thread1: Canceled\n");
        return;
    }

    SEQ_SYNC_ACK( NNT_POINT_THREAD1_CALLED );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD1_CALLED );

    // スレッドの事後状態別の後処理
    switch (expectState)
    {
        // API から普通に戻ってきた場合
        case THR_STATE_EXITED:
            {
                SEQ_CHECK( g_seq + 5 );
                NNT_OS_LOG("Thread1: return from API:");
                CheckParam( result, expect );
                SEQ_SYNC_ACK( NNT_POINT_THREAD1_SLEEP );
            }
            break;

        case THR_STATE_SLEEP:
            SEQ_SYNC_ACK( NNT_POINT_THREAD1_SLEEP );
            SEQ_SYNC_WAIT( NNT_POINT_THREAD1_SLEEP );
            break;

        case THR_STATE_WAITING2:
            {
                SEQ_NONE();
                NNT_OS_LOG("Thread1: return from API:");
                CheckParam( result, expect );

                SEQ_NONE();
                NNT_OS_LOG("Wait thread2's post transaction and End of Thread1\n");

                return;
            }
            break;

        // 想定外のエラー（テスト不具合）
        case THR_STATE_WAITING:
        default:
            CheckBool( false );
            break;
    }

    // スレッド終了
    SEQ_CHECK( expectState == THR_STATE_SLEEP ? g_seq + 9 : g_seq + 6 );
    NNT_OS_LOG("End of thread1:");
    CheckBool( true );

    SEQ_SYNC_ACK( NNT_POINT_THREAD1_END );
}


void    ThreadFunc2(void* exinf)
{
    TestThreadAction    action      = g_ThrHelper.GetThreadAction2();
    TestThreadLastState expectState = g_ThrHelper.GetThreadExpectState2();
    bool                expect      = g_ThrHelper.GetThreadExpect2();
    bool                result;

    NN_UNUSED( exinf );

    SEQ_SYNC_WAIT( NNT_POINT_THREAD2_START );

    // スレッド起動
    SEQ_CHECK( g_seq + 3 );
    NNT_OS_LOG("Begin of thread2:");
    CheckBool( true );

    // Thread2 から API を発行
    g_ThrHelper.SetApiResult2( false );
    result = g_ThrHelper.CallThreadApi( action, g_seq + 4, expect );
    g_ThrHelper.SetApiResult2( result );

    // Cleanuped ならスレッド終了（THR_STATE_WAITING だった場合など）
    if ( g_ThrHelper.IsCleanuped() )
    {
        NNT_OS_LOG("(-----): Thread2: Canceled\n");
        return;
    }

    SEQ_SYNC_ACK( NNT_POINT_THREAD2_CALLED );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD2_CALLED );

    // スレッドの事後状態別の後処理
    switch (expectState)
    {
        // API から普通に戻ってきた場合
        case THR_STATE_EXITED:
            {
                SEQ_CHECK( g_seq + 7 );
                NNT_OS_LOG("Thread2: return from API:");
                CheckParam( result, expect );
                SEQ_SYNC_ACK( NNT_POINT_THREAD2_SLEEP );
            }
            break;

        case THR_STATE_SLEEP:
            SEQ_SYNC_ACK( NNT_POINT_THREAD2_SLEEP );
            SEQ_SYNC_WAIT( NNT_POINT_THREAD2_SLEEP );
            break;

        case THR_STATE_WAITING:
            if ( g_ThrHelper.GetThreadExpectState1() == THR_STATE_WAITING2 )
            {
                break;
            }
            NN_FALL_THROUGH;

        // 想定外のエラー（テスト不具合）
        default:
            CheckBool( false );
            break;
    }

    // スレッド終了
    SEQ_CHECK( expectState == THR_STATE_SLEEP ? g_seq + 10 : g_seq + 8 );
    NNT_OS_LOG("End of thread2:");
    CheckBool( true );

    SEQ_SYNC_ACK( NNT_POINT_THREAD2_END );
}

//---------------------------------------------------------------------------

int    ThreadHelper::GetLastSeq( TestThreadLastState th1ex,
                                 TestThreadLastState th2ex )
{
    static const int lastSeqTable[THR_STATE_MAX][THR_STATE_MAX] =
    {
    //  th2: NONE,WAITING, EXITED,  SLEEP,WAITING2
        {       0,      0,      8,      10,     0,  },  // NONE
        {       2,      4,      8,      10,     4,  },  // WAITING
        {       6,      6,      8,      10,     6,  },  // EXITED
        {       9,      9,      9,      10,     9,  },  // SLEEP
        {       0,      8,      0,      0,      0,  },  // WAITING2
    };                                                  // th1

    return g_seq + lastSeqTable[th1ex][th2ex];
}

//---------------------------------------------------------------------------
//  doTest() 関数
//---------------------------------------------------------------------------

void    ThreadHelper::doTest(int                    priority,
                             TestThreadAction       th1,
                             TestThreadAction       th2,
                             bool                   rslt1,
                             bool                   rslt2,
                             int                    expectPriority,
                             TestThreadLastState    th1ex,
                             TestThreadLastState    th2ex,
                             const char*            functionName,
                             int                    line)
{
    // シーケンス番号を次の 100番台 へ進める
    g_seq = g_Sequence.Get();
    g_seq = ((g_seq + 100) / 100) * 100;

    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    // テスト用 API の発行元情報を退避
    g_Sequence.SetFunctionName( functionName );
    g_Sequence.SetLineNumber( line );

    // Cleanup 前の状態へ
    m_isCleanuped = false;
    g_ThrHelper.SetApiResult1( true );
    g_ThrHelper.SetApiResult2( true );

    // スレッドの生成
    g_seq = (g_seq / 100) * 100;
    SEQ_SET( g_seq );
    NNT_OS_LOG("Start Test\n");
    CreateThread1( th1 );
    CreateThread2( th2 );

    // スレッドの状態を構築
    SetThreadInfo1( th1, th1ex, rslt1 );
    SetThreadInfo2( th2, th2ex, rslt2 );

    // 順番にテストシーケンスを指示する
    SEQ_SYNC_INIT();

    InitializeThreadParameter( priority );
    StartThread1( th1 );
    StartThread2( th2 );
    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_START );
    SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD1_CALLED );

    if (th2ex != THR_STATE_NONE)
    {
        SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_START );
        SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD2_CALLED );
    }

    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_CALLED );
    SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD1_SLEEP );

    if (th2ex != THR_STATE_NONE)
    {
        SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_CALLED );
        SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD2_SLEEP );
    }

    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_SLEEP );
    SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD1_END );

    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_SLEEP );
    SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD2_END );

    // スレッド１の状態を検査
    SEQ_WAIT_UNTIL( GetLastSeq(th1ex, th2ex), g_seq + 11 );
    NNT_OS_LOG("Thread1 state check:");
    ThreadStateCheck( m_Thread1, th1, th1ex, GetApiResult1(), rslt1 );

    // スレッド２の状態を検査
    SEQ_NONE();
    NNT_OS_LOG("Thread2 state check:");
    ThreadStateCheck( m_Thread2, th2, th2ex, GetApiResult2(), rslt2 );

    // スレッドの状態を検査
    SEQ_NONE();
    NNT_OS_LOG("Thread state check:");
    ThreadCheck( expectPriority );

    // スレッド状態を初期状態へ
    SEQ_NONE();
    NNT_OS_LOG("Cleanup Thread and Threads\n");
    Cleanup();

    // スレッドオブジェクトの Finalize
    DestroyThread1( th1 );
    DestroyThread2( th2 );

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

}}} // namespace nnt::os::thread

