﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  InterProcessEvent としての SystemEvent 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "../Common/test_NamedPipe.h"
#include "test_SystemEventAsInterProcessEventServer.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

#define AUTO_CLEAR    nn::os::EventClearMode_AutoClear
#define MANUAL_CLEAR  nn::os::EventClearMode_ManualClear

namespace nnt { namespace os { namespace systemEventAsInterProcessEvent {

//---------------------------------------------------------------------------
// SystemEvent 受理（AutoReset）
//
//  システムイベントを生成し、WaitSystemEvent の動作を確認する。
//  Pipe 機能を使い、システムイベントの WritableHandle をサーバに渡し、
//  そこから SignalSystemEvent() してもらう。
//
TEST(SystemEventAsInterProcessEvent, test_WaitSystemEventAsInterProcessEventAutoReset)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // サーバの起動
    nnt::os::systemEventAsInterProcessEvent::StartTestServer();

    // クライアントからサーバへの接続
    detail::NamedPipeClient clientPipe( NNT_OS_TEST_PORTNAME );
    nn::os::SystemEventType systemEvent;

    NNT_OS_LOG(NN_TEXT("<< AutoReset の SystemEvent に対する Wait のテスト >>\n"));
    NNT_OS_LOG(NN_TEXT("<< 事前に serverPipe に通知して 50msec 後に Signal してもらう。 >>\n"));

    // システムイベントの初期化
    auto result = nn::os::CreateSystemEvent( &systemEvent, AUTO_CLEAR, true );
    EXPECT_TRUE( result.IsSuccess() );

    // 初期化直後が非シグナル状態化をチェック
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("CreateSystemEvent() 直後は非シグナル状態か"));
    bool ret = TryWaitSystemEvent( &systemEvent );
    CheckBool( ret == false );

    // 通知後にすぐに待機
    auto handle = nn::os::GetWritableHandleOfSystemEvent( &systemEvent );
    clientPipe.SendHandle( detail::RequireSignal, handle );
    nn::os::WaitSystemEvent( &systemEvent );

    // systemEvent が正しく AUTO_CLEAR されているかチェックする。
    // TryWaitSystemEvent() で false なら期待通りクリアされている。
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("WaitSystemEvent() 後に自動クリアされているか"));
    ret = TryWaitSystemEvent( &systemEvent );
    CheckBool( ret == false );

    // システムイベントの削除
    nn::os::DestroySystemEvent( &systemEvent );

    // サーバの終了
    nnt::os::systemEventAsInterProcessEvent::StopTestServer(clientPipe);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// SystemEvent 受理（ManualReset）
//
//  システムイベントを生成し、WaitSystemEvent の動作を確認する。
//  Pipe 機能を使い、システムイベントの WritableHandle をサーバに渡し、
//  そこから SignalSystemEvent() してもらう。
//
//
TEST(SystemEventAsInterProcessEvent, test_WaitSystemEventAsInterProcessEventManualReset)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // サーバの起動
    nnt::os::systemEventAsInterProcessEvent::StartTestServer();

    // クライアントからサーバへの接続
    detail::NamedPipeClient clientPipe( NNT_OS_TEST_PORTNAME );
    nn::os::SystemEventType systemEvent;

    NNT_OS_LOG(NN_TEXT("<< ManualReset の SystemEvent に対する Wait のテスト >>\n"));
    NNT_OS_LOG(NN_TEXT("<< 事前に serverPipe に通知して 50msec 後に Signal してもらう。 >>\n"));

    // システムイベントの初期化
    auto result = nn::os::CreateSystemEvent( &systemEvent, MANUAL_CLEAR, true );
    EXPECT_TRUE( result.IsSuccess() );

    // 初期化直後が非シグナル状態化をチェック
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("CreateSystemEvent() 直後は非シグナル状態か"));
    bool ret = TryWaitSystemEvent( &systemEvent );
    CheckBool( ret == false );

    // 通知後にすぐに待機
    auto handle = nn::os::GetWritableHandleOfSystemEvent( &systemEvent );
    clientPipe.SendHandle( detail::RequireSignal, handle );
    nn::os::WaitSystemEvent( &systemEvent );

    // systemEvent のシグナル状態が維持されているかをチェックする。
    // TryWaitSystemEvent() で true なら期待通り維持されている。
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("WaitSystemEvent() 後にシグナル状態が維持されているか"));
    ret = TryWaitSystemEvent( &systemEvent );
    CheckBool( ret == true );

    // システムイベントの削除
    nn::os::DestroySystemEvent( &systemEvent );

    // サーバの終了
    nnt::os::systemEventAsInterProcessEvent::StopTestServer(clientPipe);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// TryWaitSystemEvent 動作テスト
//
//  システムイベントを生成し、TryWaitSystemEvent の動作を確認する。
//  Pipe 機能を使い、システムイベントの WritableHandle をサーバに渡し、
//  そこから SignalSystemEvent() してもらう。
//  シグナル化された同期イベントに対して TryWait して true を確認した後、
//  非シグナル状態の同期イベントに対しても TryWait して false を確認する。
//  なお、本テストは ClearSystemEvent() のテストにもなっている。
//
TEST(SystemEventAsInterProcessEvent, test_TryWaitSystemEventAsInterProcessEvent)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // サーバの起動
    nnt::os::systemEventAsInterProcessEvent::StartTestServer();

    // クライアントからサーバへの接続
    detail::NamedPipeClient clientPipe( NNT_OS_TEST_PORTNAME );
    nn::os::SystemEventType systemEvent;

    NNT_OS_LOG(NN_TEXT("<< ManualReset の SystemEvent に対する TryWait のテスト >>\n"));
    NNT_OS_LOG(NN_TEXT("<< 事前に serverPipe に通知して 50msec 後に Signal してもらう。 >>\n"));

    // システムイベントの初期化
    auto result = nn::os::CreateSystemEvent( &systemEvent, MANUAL_CLEAR, true );
    EXPECT_TRUE( result.IsSuccess() );

    nn::os::ClearSystemEvent( &systemEvent );

    // 通知後にすぐに TryWait を発行

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("非シグナル状態中に TryWaitSystemEvent() でポーリングする。\n"));
    NNT_OS_LOG(NN_TEXT("返値が false から true になったのを確認してループを抜ける。"));

    auto handle = nn::os::GetWritableHandleOfSystemEvent( &systemEvent );
    clientPipe.SendHandle( detail::RequireSignal, handle );
    int count = 0;
    for (;;)
    {
        bool ret = nn::os::TryWaitSystemEvent( &systemEvent );
        if (ret != false)
        {
            break;
        }
        ++count;
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
    }

    CheckBool( count > 0);

    // その後すぐに TryWait を発行

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("シグナル状態中に TryWaitSystemEvent() で true を確認"));

    bool ret = nn::os::TryWaitSystemEvent( &systemEvent );
    CheckBool( ret == true );

    // 一旦クリアする

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("ClearSystemEvent() を発行\n"));
    nn::os::ClearSystemEvent( &systemEvent );

    // すぐに TryWait を発行

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("非シグナル状態中に TryWaitSystemEvent() で false を確認"));
    ret = nn::os::TryWaitSystemEvent( &systemEvent );
    CheckBool( ret == false );

    // システムイベントの削除
    nn::os::DestroySystemEvent( &systemEvent );

    // サーバの終了
    nnt::os::systemEventAsInterProcessEvent::StopTestServer(clientPipe);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// TimedWaitSystemEvent 動作テスト
//
//  システムイベントを生成し、TimedWaitSystemEvent の動作を
//  確認する。Pipe 機能を使い、システムイベントの WritableHandle を
//  サーバに渡し、50msec 後に SignalSystemEvent() してもらう。
//
//  タイムアウト 1msec で、うまくタイムアウトできることの確認。
//  タイムアウト 2sec  で、それまでにシグナルが受理されることの確認。
//
TEST(SystemEventAsInterProcessEvent, test_TimedWaitSystemEventAsInterProcessEvent)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // サーバの起動
    nnt::os::systemEventAsInterProcessEvent::StartTestServer();

    // クライアントからサーバへの接続
    detail::NamedPipeClient clientPipe( NNT_OS_TEST_PORTNAME );
    nn::os::SystemEventType systemEvent;

    NNT_OS_LOG(NN_TEXT("<< まず最初に、単純にタイムアウトが成立するかを確認 >>\n"));
    NNT_OS_LOG(NN_TEXT("<< その後、タイムアウト前に起床されるかを確認 >>\n"));

    // システムイベントの初期化
    auto result = nn::os::CreateSystemEvent( &systemEvent, MANUAL_CLEAR, true );
    EXPECT_TRUE( result.IsSuccess() );

    nn::os::ClearSystemEvent( &systemEvent );

    // 通知後にすぐに TimedWait でタイムアウトが成立するのを確認する

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("すぐにタイムアウトが成立するか"));

    auto handle = nn::os::GetWritableHandleOfSystemEvent( &systemEvent );
    clientPipe.SendHandle( detail::RequireSignal, handle );
    bool ret = nn::os::TimedWaitSystemEvent( &systemEvent, nn::TimeSpan::FromMilliSeconds(1) );
    CheckBool( ret == false );

    // 一旦、シグナル化されるまで待ち、クリアしておく
    nn::os::WaitSystemEvent( &systemEvent );
    nn::os::ClearSystemEvent( &systemEvent );


    // 通知後にすぐに TimedWait でタイムアウト前に起床されるのを確認する

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("タイムアウト前に起床されるか"));

    handle = nn::os::GetWritableHandleOfSystemEvent( &systemEvent );
    clientPipe.SendHandle( detail::RequireSignal, handle );
    ret = nn::os::TimedWaitSystemEvent( &systemEvent, nn::TimeSpan::FromMilliSeconds(5000) );
    CheckBool( ret == true );

    // システムイベントの削除
    nn::os::DestroySystemEvent( &systemEvent );

    // サーバの終了
    nnt::os::systemEventAsInterProcessEvent::StopTestServer(clientPipe);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// 多重待ちによる SystemEvent 受理（AutoReset）
//
//  システムイベントを生成し、WaitSystemEvent の動作を確認する。
//  Pipe 機能を使い、システムイベントの WritableHandle をサーバに渡し、
//  そこから SignalSystemEvent() してもらう。
//
TEST(SystemEventAsInterProcessEvent, test_WaitAny_SystemEventAsInterProcessEventAutoReset)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // サーバの起動
    nnt::os::systemEventAsInterProcessEvent::StartTestServer();

    // クライアントからサーバへの接続
    detail::NamedPipeClient         clientPipe( NNT_OS_TEST_PORTNAME );
    nn::os::SystemEventType         systemEvent;
    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     holder;

    NNT_OS_LOG(NN_TEXT("<< AutoReset の SystemEvent に対する WaitAny のテスト >>\n"));
    NNT_OS_LOG(NN_TEXT("<< 事前に serverPipe に通知して 50msec 後に Signal してもらう。 >>\n"));

    // システムイベントの初期化
    auto result = nn::os::CreateSystemEvent( &systemEvent, AUTO_CLEAR, true );
    EXPECT_TRUE( result.IsSuccess() );

    nn::os::InitializeMultiWait( &multiWait );
    nn::os::InitializeMultiWaitHolder( &holder, &systemEvent );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder );

    // 初期化直後が非シグナル状態化をチェック
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("CreateSystemEvent() 直後は非シグナル状態か"));
    bool ret = TryWaitSystemEvent( &systemEvent );
    CheckBool( ret == false );

    // 通知後にすぐに多重待ち
    auto handle = nn::os::GetWritableHandleOfSystemEvent( &systemEvent );
    clientPipe.SendHandle( detail::RequireSignal, handle );
    nn::os::WaitAny( &multiWait );

    // systemEvent が WaitAny() 時は AutoClear されないことをチェックする。
    // TryWaitSystemEvent() で true ならクリアされていない。
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("AutoClear でも WaitAny() 後は自動クリアされない"));
    ret = TryWaitSystemEvent( &systemEvent );
    CheckBool( ret == true );

    // システムイベントの削除
    nn::os::UnlinkMultiWaitHolder( &holder );
    nn::os::FinalizeMultiWaitHolder( &holder );
    nn::os::DestroySystemEvent( &systemEvent );

    // サーバの終了
    nnt::os::systemEventAsInterProcessEvent::StopTestServer(clientPipe);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// 多重待ちによる SystemEvent 受理（ManualReset）
//
//  システムイベントを生成し、WaitSystemEvent の動作を確認する。
//  Pipe 機能を使い、システムイベントの WritableHandle をサーバに渡し、
//  そこから SignalSystemEvent() してもらう。
//
//
TEST(SystemEventAsInterProcessEvent, test_WaitAny_SystemEventAsInterProcessEventManualReset)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // サーバの起動
    nnt::os::systemEventAsInterProcessEvent::StartTestServer();

    // クライアントからサーバへの接続
    detail::NamedPipeClient         clientPipe( NNT_OS_TEST_PORTNAME );
    nn::os::SystemEventType         systemEvent;
    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     holder;

    NNT_OS_LOG(NN_TEXT("<< ManualReset の SystemEvent に対する WaitAny のテスト >>\n"));
    NNT_OS_LOG(NN_TEXT("<< 事前に serverPipe に通知して 50msec 後に Signal してもらう。 >>\n"));

    // システムイベントの初期化
    auto result = nn::os::CreateSystemEvent( &systemEvent, MANUAL_CLEAR, true );
    EXPECT_TRUE( result.IsSuccess() );

    nn::os::InitializeMultiWait( &multiWait );
    nn::os::InitializeMultiWaitHolder( &holder, &systemEvent );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder );

    // 初期化直後が非シグナル状態化をチェック
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("CreateSystemEvent() 直後は非シグナル状態か"));
    bool ret = TryWaitSystemEvent( &systemEvent );
    CheckBool( ret == false );

    // 通知後にすぐに待機
    auto handle = nn::os::GetWritableHandleOfSystemEvent( &systemEvent );
    clientPipe.SendHandle( detail::RequireSignal, handle );
    nn::os::WaitAny( &multiWait );

    // systemEvent のシグナル状態が維持されているかをチェックする。
    // TryWaitSystemEvent() で true なら期待通り維持されている。
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("WaitAny() 後にシグナル状態が維持されているか"));
    ret = TryWaitSystemEvent( &systemEvent );
    CheckBool( ret == true );

    // システムイベントの削除
    nn::os::UnlinkMultiWaitHolder( &holder );
    nn::os::FinalizeMultiWaitHolder( &holder );
    nn::os::DestroySystemEvent( &systemEvent );

    // サーバの終了
    nnt::os::systemEventAsInterProcessEvent::StopTestServer(clientPipe);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// 多重待ちによる TryWaitSystemEvent 動作テスト
//
//  システムイベントを生成し、TryWaitSystemEvent の動作を確認する。
//  Pipe 機能を使い、システムイベントの WritableHandle をサーバに渡し、
//  そこから SignalSystemEvent() してもらう。
//  シグナル化された同期イベントに対して TryWait して true を確認した後、
//  非シグナル状態の同期イベントに対しても TryWait して false を確認する。
//  なお、本テストは ClearSystemEvent() のテストにもなっている。
//
TEST(SystemEventAsInterProcessEvent, test_TryWaitAny_SystemEventAsInterProcessEvent)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // サーバの起動
    nnt::os::systemEventAsInterProcessEvent::StartTestServer();

    // クライアントからサーバへの接続
    detail::NamedPipeClient         clientPipe( NNT_OS_TEST_PORTNAME );
    nn::os::SystemEventType         systemEvent;
    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     holder;

    NNT_OS_LOG(NN_TEXT("<< ManualReset の SystemEvent に対する TryWaitAny のテスト >>\n"));
    NNT_OS_LOG(NN_TEXT("<< 事前に serverPipe に通知して 50msec 後に Signal してもらう。 >>\n"));

    // システムイベントの初期化
    auto result = nn::os::CreateSystemEvent( &systemEvent, MANUAL_CLEAR, true );
    EXPECT_TRUE( result.IsSuccess() );

    nn::os::InitializeMultiWait( &multiWait );
    nn::os::InitializeMultiWaitHolder( &holder, &systemEvent );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder );

    // 通知後にすぐに TryWait を発行

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("非シグナル状態中に TryWaitAny() でポーリングする。\n"));
    NNT_OS_LOG(NN_TEXT("返値が false から true になったのを確認してループを抜ける。"));

    auto handle = nn::os::GetWritableHandleOfSystemEvent( &systemEvent );
    clientPipe.SendHandle( detail::RequireSignal, handle );
    int count = 0;
    for (;;)
    {
        nn::os::MultiWaitHolderType* ret = nn::os::TryWaitAny( &multiWait );
        if (ret != NULL)
        {
            break;
        }
        ++count;
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
    }

    CheckBool( count > 0);

    // その後すぐに TryWait を発行

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("シグナル状態中に TryWaitAny() で true を確認"));

    nn::os::MultiWaitHolderType* ret = nn::os::TryWaitAny( &multiWait );
    CheckBool( ret == &holder );

    // 一旦クリアする

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("ClearSystemEvent() を発行\n"));
    nn::os::ClearSystemEvent( &systemEvent );

    // すぐに TryWait を発行

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("非シグナル状態中に TryWaitAny() で false を確認"));
    ret = nn::os::TryWaitAny( &multiWait );
    CheckBool( ret == NULL );

    // システムイベントの削除
    nn::os::UnlinkMultiWaitHolder( &holder );
    nn::os::FinalizeMultiWaitHolder( &holder );
    nn::os::DestroySystemEvent( &systemEvent );

    // サーバの終了
    nnt::os::systemEventAsInterProcessEvent::StopTestServer(clientPipe);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// 多重待ちによる TimedWaitSystemEvent 動作テスト
//
//  システムイベントを生成し、TimedWaitSystemEvent の動作を
//  確認する。Pipe 機能を使い、システムイベントの WritableHandle を
//  サーバに渡し、50msec 後に SignalSystemEvent() してもらう。
//
//  タイムアウト 1msec で、うまくタイムアウトできることの確認。
//  タイムアウト 2sec  で、それまでにシグナルが受理されることの確認。
//
TEST(SystemEventAsInterProcessEvent, test_TimedWaitAny_SystemEventAsInterProcessEvent)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // サーバの起動
    nnt::os::systemEventAsInterProcessEvent::StartTestServer();

    // クライアントからサーバへの接続
    detail::NamedPipeClient         clientPipe( NNT_OS_TEST_PORTNAME );
    nn::os::SystemEventType         systemEvent;
    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     holder;

    NNT_OS_LOG(NN_TEXT("<< まず最初に、単純にタイムアウトが成立するかを確認 >>\n"));
    NNT_OS_LOG(NN_TEXT("<< その後、タイムアウト前に起床されるかを確認 >>\n"));

    // システムイベントの初期化
    auto result = nn::os::CreateSystemEvent( &systemEvent, MANUAL_CLEAR, true );
    EXPECT_TRUE( result.IsSuccess() );

    nn::os::InitializeMultiWait( &multiWait );
    nn::os::InitializeMultiWaitHolder( &holder, &systemEvent );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder );

    // 通知後にすぐに TimedWait でタイムアウトが成立するのを確認する

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("すぐにタイムアウトが成立するか"));

    auto handle = nn::os::GetWritableHandleOfSystemEvent( &systemEvent );
    clientPipe.SendHandle( detail::RequireSignal, handle );
    nn::os::MultiWaitHolderType* ret = nn::os::TimedWaitAny( &multiWait, nn::TimeSpan::FromMilliSeconds(1) );
    CheckBool( ret == NULL );

    // 一旦、シグナル化されるまで待ち、クリアしておく
    nn::os::WaitAny( &multiWait );
    nn::os::ClearSystemEvent( &systemEvent );


    // 通知後にすぐに TimedWait でタイムアウト前に起床されるのを確認する

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("タイムアウト前に起床されるか"));

    handle = nn::os::GetWritableHandleOfSystemEvent( &systemEvent );
    clientPipe.SendHandle( detail::RequireSignal, handle );
    ret = nn::os::TimedWaitAny( &multiWait, nn::TimeSpan::FromMilliSeconds(5000) );
    CheckBool( ret == &holder );

    // システムイベントの削除
    nn::os::UnlinkMultiWaitHolder( &holder );
    nn::os::FinalizeMultiWaitHolder( &holder );
    nn::os::DestroySystemEvent( &systemEvent );

    // サーバの終了
    nnt::os::systemEventAsInterProcessEvent::StopTestServer(clientPipe);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  SystemEvent クラスの InterProcessEvent としてのテスト
//
//  ここでは、SystemEvent クラスが OS-API をラッピングしたものである
//  という前提で各メソッドの簡単な動作確認のみを行なうテストである。
//
//  SystemEvent クラスを使ってプロセス間同期イベントのインスタンスを生成し、
//  その Readable/Writable ハンドルを、それぞれ別の SystemEvent インスタンス
//  として初期化する。あとはそれらのインスタンスを使って同期の確認を行なう。
//
//  このテストは InterpProcessEvent で行なっているものと内容的に等価である。
//
TEST(SystemEventClass, test_SystemEventClassTestManualClear)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // サーバの起動
    nnt::os::systemEventAsInterProcessEvent::StartTestServer();

    // クライアントからサーバへの接続
    detail::NamedPipeClient clientPipe( NNT_OS_TEST_PORTNAME );
    const nn::TimeSpan  shortTimeout = nn::TimeSpan::FromMicroSeconds(1);
    const nn::TimeSpan  longTimeout  = nn::TimeSpan::FromSeconds(10);

    // SystemEvent クラスのインスタンスの生成
    nn::os::SystemEvent systemEvent( MANUAL_CLEAR, true );
    auto readableHandle = systemEvent.DetachReadableHandle();
    auto writableHandle = systemEvent.DetachWritableHandle();

    nn::os::SystemEvent readSystemEvent;
    nn::os::SystemEvent writeSystemEvent;

    readSystemEvent.AttachReadableHandle( readableHandle, true, MANUAL_CLEAR );
    writeSystemEvent.AttachWritableHandle( writableHandle, true, MANUAL_CLEAR );

    // ここからがテスト

    // Signal のテスト
    NNT_OS_LOG(NN_TEXT("main: writeSystemEvent.Signal() に成功する\n"));
    writeSystemEvent.Signal();

    // Wait のテスト
    NNT_OS_LOG(NN_TEXT("main: readSystemEvent.Wait() に成功する\n"));
    readSystemEvent.Wait();

    // TryWait のテスト
    NNT_OS_LOG(NN_TEXT("main: readSystemEvent.TryWait() が true を返す"));
    CheckBool( readSystemEvent.TryWait() == true );

    // TimedWait のテスト
    NNT_OS_LOG(NN_TEXT("main: readSystemEvent.TimedWait(10sec) が true を返す"));
    CheckBool( readSystemEvent.TimedWait(longTimeout) == true );

    // クリアの確認
    NNT_OS_LOG(NN_TEXT("main: writeSystemEvent.Clear() に成功する\n"));
    readSystemEvent.Clear();

    // TryWait のテスト
    NNT_OS_LOG(NN_TEXT("main: readSystemEvent.TryWait() が false を返す"));
    CheckBool( readSystemEvent.TryWait() == false );

    // TimedWait のテスト
    NNT_OS_LOG(NN_TEXT("main: readSystemEvent.TimedWait(1usec) が false を返す"));
    CheckBool( readSystemEvent.TimedWait(shortTimeout) == false );

    // Wait の待ち解除テスト（サーバ側で Signal してもらう）
    clientPipe.SendHandle( detail::RequireSignal, writableHandle );
    NNT_OS_LOG(NN_TEXT("main: readSystemEvent.Wait() に成功する\n"));
    readSystemEvent.Wait();

    // ハンドルのクローズ
    NNT_OS_LOG(NN_TEXT("main: nn::os::CloseNativeHandle() に成功する"));
    readableHandle = readSystemEvent.DetachReadableHandle();
    writableHandle = writeSystemEvent.DetachWritableHandle();
    nn::os::CloseNativeHandle( readableHandle );
    nn::os::CloseNativeHandle( writableHandle );
    CheckBool( true );

    // サーバの終了
    nnt::os::systemEventAsInterProcessEvent::StopTestServer(clientPipe);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  SystemEvent クラスの型変換関数のテスト
//  ここでは、SystemEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  ここでは、GetBase() および operator SystemEventType&() の動作テストを行なう。
//
TEST(SystemEventClass, test_SystemEventClassTestTypeExchange)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        NNT_OS_LOG(NN_TEXT("SystemEvent インスタンスの生成（AutoClear）\n"));
        nn::os::SystemEvent systemEvent( MANUAL_CLEAR, true );

        // ここからがテスト
        NNT_OS_LOG(NN_TEXT("systemEvent.GetBase() で SystemEventType オブジェクトを取得"));
        nn::os::SystemEventType*  systemEventType = systemEvent.GetBase();
        CheckBool( systemEventType != NULL );

        // クラスとオブジェクトの API で readableHandle が同じものかを確認
        NNT_OS_LOG(NN_TEXT("クラスとオブジェクトで同じ readableHandle を取得できるか"));
        auto readableHandle = nn::os::GetReadableHandleOfSystemEvent(systemEventType);
        CheckBool( systemEvent.GetReadableHandle() == readableHandle );

        // クラスとオブジェクトの API で writableHandle が同じものかを確認
        NNT_OS_LOG(NN_TEXT("クラスとオブジェクトで同じ writableHandle を取得できるか"));
        auto writableHandle = nn::os::GetWritableHandleOfSystemEvent(systemEventType);
        CheckBool( systemEvent.GetWritableHandle() == writableHandle );

        // operator SystemEventType&() の確認
        NNT_OS_LOG(NN_TEXT("operator SystemEventType&() を使ってオブジェクトの参照を取得"));
        nn::os::SystemEventType& systemEventTypeRefer = systemEvent;
        CheckBool( &systemEventTypeRefer == systemEventType );

        // operator const SystemEventType&() の確認
        NNT_OS_LOG(NN_TEXT("operator const SystemEventType&() を使ってオブジェクトの参照を取得"));
        const nn::os::SystemEventType& systemEventTypeReferConst = systemEvent;
        CheckBool( &systemEventTypeReferConst == systemEventType );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
//  SIGLO-31003 複数のスレッドで WaitSystemEvent を行なうテスト。
//  AutoClear な場合の OS ライブラリ内部での ResetSignal() 実装修正用の
//  リグレッション用のテスト。
//
//  テストロジック：
//  1) Core0 で AutoClear な SystemEvent を作成
//  2) Core1 で 1) の SystemEvent を Wait()
//  3) Core2 で 1) の SystemEvent を Wait()
//  4) Core0 で 1) の SystemEvent を Signal() して、2) と 3) のスレッドの
//     片方（どちらかは問わない）だけが起床していることを確認する。
//
NN_OS_ALIGNAS_THREAD_STACK  char g_ThreadStack1[0x4000];
NN_OS_ALIGNAS_THREAD_STACK  char g_ThreadStack2[0x4000];
nn::os::Semaphore                g_Semaphore(0, 10);
nn::os::Barrier                  g_Barrier(3);

void SystemEventSIGLO31003_Thread1(void* arg)
{
    auto& event = *reinterpret_cast<nn::os::SystemEvent*>(arg);

    event.Wait();
    g_Semaphore.Release();
    g_Barrier.Await();

    event.TimedWait( nn::TimeSpan::FromHours(1) );
    g_Semaphore.Release();
}

void SystemEventSIGLO31003_Thread2(void* arg)
{
    auto& event = *reinterpret_cast<nn::os::SystemEvent*>(arg);

    event.Wait();
    g_Semaphore.Release();
    g_Barrier.Await();

    event.TimedWait( nn::TimeSpan::FromHours(1) );
    g_Semaphore.Release();
}

TEST(SystemEventSIGLO31003, test_SystemEventSIGLO31003_Main)
{
    // 搭載コア数を算出（最下位の 1 のビット数をカウントしてコア数とする）
    nn::Bit64 coreMask = nn::os::GetThreadAvailableCoreMask();
              coreMask = nn::util::maskt1<nn::Bit64>(coreMask);
    int       coreNums = nn::util::cntt0<nn::Bit64>(~coreMask);
    NNT_OS_LOG("Core nums = %d\n", coreNums);
    if (coreNums < 3)
    {
        // CPU コア数が 3 未満ならテストをスキップ
        NNT_OS_LOG("Skipped this test because core nums are less than 3.\n");
        SUCCEED();
        return;
    }

    nn::os::SystemEvent event(nn::os::EventClearMode_AutoClear, true);

    int pri = nn::os::GetThreadCurrentPriority( nn::os::GetCurrentThread() );

    nn::os::ThreadType  thread1;
    nn::os::ThreadType  thread2;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread(
                                        &thread1,
                                        SystemEventSIGLO31003_Thread1,
                                        &event,
                                        g_ThreadStack1,
                                        sizeof(g_ThreadStack1),
                                        pri,
                                        1) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread(
                                        &thread2,
                                        SystemEventSIGLO31003_Thread2,
                                        &event,
                                        g_ThreadStack2,
                                        sizeof(g_ThreadStack2),
                                        pri,
                                        2) );
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );

    // thread1, thread2 が SystemEvent を Wait() してから Signal() する
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
    event.Signal();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(20) );

    // Wait が解除されるのに十分な時間を経てから
    // g_Semaphore のカウンタを見て、解除された Wait() の数を検査する
    EXPECT_EQ(1, g_Semaphore.GetCurrentCount());

    // 次のテストに移行するため、もう１つの Wait も解除。
    // g_Semaphore カウンタも 0 にしておく。
    event.Signal();
    g_Semaphore.Acquire();
    g_Semaphore.Acquire();
    EXPECT_EQ(0, g_Semaphore.GetCurrentCount());
    g_Barrier.Await();


    // thread1, thread2 が SystemEvent を TimedWait() してから Signal() する
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
    event.Signal();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(20) );

    // Wait が解除されるのに十分な時間を経てから
    // g_Semaphore のカウンタを見て、解除された Wait() の数を検査する
    EXPECT_EQ(1, g_Semaphore.GetCurrentCount());

    // テストを終了するための後処理
    event.Signal();

    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );
}


}}} // namespace nnt::os::systemEventAsInterProcessEvent

//-----------------------------------------------------------------------------
//  Test Main 関数
//-----------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of SystemEvent as InterProcessEvent APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of SystemEvent as InterProcessEvent APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
