﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"
#include "../Common/test_Helper.h"

#include <nn/os/os_Config.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include "test_DelayedSystemEvent.h"

namespace nnt { namespace os { namespace detail {

//-----------------------------------------------------------------------------

namespace {

NN_ALIGNAS(4096) char   g_ServerThreadStack[8192];

const int   queueSize = 16;
uintptr_t   g_QueueBuffer[ queueSize ];

//-----------------------------------------------------------------------------
//  サーバ用スレッドのラッパー
//
void DelayedSystemEventServerThread(void* args)
{
    auto* p = reinterpret_cast<DelayedSystemEventServer*>( args );

    NNT_OS_LOG("(-----): Start Server\n");

    p->ServerThreadImpl();

    NNT_OS_LOG("(-----): End Server\n");
}

}   // namespace


//-----------------------------------------------------------------------------
//  サーバスレッドの本体
void DelayedSystemEventServer::ServerThreadImpl()
NN_NOEXCEPT
{
    for (;;)
    {
        uintptr_t   cmd;
        uintptr_t   msg;

        nn::os::ReceiveMessageQueue(&cmd, &m_Queue);
        switch(cmd)
        {
        case Command_DelayedSignal:
                nn::os::ReceiveMessageQueue(&msg, &m_Queue);

                nn::os::SleepThread( nn::TimeSpan::FromNanoSeconds(msg) );
                nn::os::SignalSystemEvent( m_SystemEvent );
                break;

        case Command_Shutdown:
                return;

        default:
                continue;
        }
    }
}


//-----------------------------------------------------------------------------
//  コンストラクタ。サーバ用スレッドを起動する。
//
DelayedSystemEventServer::DelayedSystemEventServer(nn::os::SystemEventType* event)
NN_NOEXCEPT : m_SystemEvent(event)
{
    // メッセージキューを初期化
    nn::os::InitializeMessageQueue( &m_Queue, g_QueueBuffer, queueSize );

    // サーバスレッドを生成
    nn::os::CreateThread(&m_ServerThread, DelayedSystemEventServerThread, this, g_ServerThreadStack, sizeof(g_ServerThreadStack), nn::os::DefaultThreadPriority - 1);

    // サーバを起動
    nn::os::StartThread(&m_ServerThread);
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(300) );
}

//-----------------------------------------------------------------------------
//  デストラクタ。サーバ用スレッドを削除する。
//
DelayedSystemEventServer::~DelayedSystemEventServer()
NN_NOEXCEPT
{
    // サーバへ終了要求を出す
    nn::os::SendMessageQueue( &m_Queue, Command_Shutdown );

    // サーバが終了するのを待って削除
    nn::os::WaitThread(&m_ServerThread);
    nn::os::DestroyThread(&m_ServerThread);
}


//-----------------------------------------------------------------------------

}}} // namespace nnt::os::detail


