﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include <nnt/nntest.h>

#include "../Common/test_Thread.h"

namespace nnt { namespace os { namespace readWriteLock {

//-----------------------------------------------------------------------------
//  Readers-Writer ロック機能のテスト
//
//  このソースでは、コア API の機能テストを行なう。
//-----------------------------------------------------------------------------

namespace {

    int g_Sequence;

    inline void CheckReaderWriterLockHeld(nn::os::ReaderWriterLockType* rwlock, bool readLocked, bool writeLocked, bool isOwner)
    {
        EXPECT_EQ( readLocked,  IsReadLockHeld(rwlock) );
        EXPECT_EQ( writeLocked, IsWriteLockHeldByCurrentThread(rwlock) );
        EXPECT_EQ( isOwner,     IsReaderWriterLockOwnerThread(rwlock) );
    }

    //----------

    nnt::os::StackImpl<0x4000> g_ThreadStack1;
    nnt::os::StackImpl<0x4000> g_ThreadStack2;

}   // namespace


//-----------------------------------------------------------------------------
// 基本的なコア API の呼出し
//-----------------------------------------------------------------------------

TEST(ReaderWriterLockTest, ReadLock)
{
    // 初期化
    nn::os::ReaderWriterLockType   rwlock;
    nn::os::InitializeReaderWriterLock( &rwlock );
    CheckReaderWriterLockHeld( &rwlock, false, false, false );

    // Read ロックの基本テスト
    nn::os::AcquireReadLock( &rwlock );
    CheckReaderWriterLockHeld( &rwlock, true, false, false );
    {
        // 再帰ロック
        nn::os::AcquireReadLock( &rwlock );
        CheckReaderWriterLockHeld( &rwlock, true, false, false );

        // Read ロックのポーリング
        auto ret = nn::os::TryAcquireReadLock( &rwlock );
        EXPECT_TRUE( ret == true );

        CheckReaderWriterLockHeld( &rwlock, true, false, false );
        nn::os::ReleaseReadLock( &rwlock );

        // Write ロックのポーリング
        ret = nn::os::TryAcquireWriteLock( &rwlock );
        EXPECT_TRUE( ret == false );
        CheckReaderWriterLockHeld( &rwlock, true, false, false );

        // 再帰ロック解除
        nn::os::ReleaseReadLock( &rwlock );
        CheckReaderWriterLockHeld( &rwlock, true, false, false );
    }
    nn::os::ReleaseReadLock( &rwlock );
    CheckReaderWriterLockHeld( &rwlock, false, false, false );

    // Write ロックの基本テスト
    nn::os::AcquireWriteLock( &rwlock );
    CheckReaderWriterLockHeld( &rwlock, false, true, true );
    {
        // 再帰ロック
        nn::os::AcquireWriteLock( &rwlock );
        CheckReaderWriterLockHeld( &rwlock, false, true, true );

        // Write ロックのポーリング
        auto ret = nn::os::TryAcquireWriteLock( &rwlock );
        EXPECT_TRUE( ret == true );

        CheckReaderWriterLockHeld( &rwlock, false, true, true );
        nn::os::ReleaseWriteLock( &rwlock );

        // Write ロック中の Read ロック
        nn::os::AcquireReadLock( &rwlock );
        CheckReaderWriterLockHeld( &rwlock, true, true, true );

        nn::os::AcquireReadLock( &rwlock );
        CheckReaderWriterLockHeld( &rwlock, true, true, true );

        nn::os::ReleaseReadLock( &rwlock );
        nn::os::ReleaseReadLock( &rwlock );
        CheckReaderWriterLockHeld( &rwlock, false, true, true );

        // Write ロック中の Read ロックのポーリング
        ret = nn::os::TryAcquireReadLock( &rwlock );
        EXPECT_TRUE( ret == true );
        CheckReaderWriterLockHeld( &rwlock, true, true, true );

        ret = nn::os::TryAcquireReadLock( &rwlock );
        EXPECT_TRUE( ret == true );
        CheckReaderWriterLockHeld( &rwlock, true, true, true );

        nn::os::ReleaseReadLock( &rwlock );
        nn::os::ReleaseReadLock( &rwlock );
        CheckReaderWriterLockHeld( &rwlock, false, true, true );

        // 再帰ロック解除
        nn::os::ReleaseWriteLock( &rwlock );
        CheckReaderWriterLockHeld( &rwlock, false, true, true );
    }
    nn::os::ReleaseWriteLock( &rwlock );
    CheckReaderWriterLockHeld( &rwlock, false, false, false );

    // Write/Read ロックの混在テスト
    // WriteLock -> ReadLock -> WriteUnlock -> ReadUnlock

    // Write ロック
    nn::os::AcquireWriteLock( &rwlock );
    CheckReaderWriterLockHeld( &rwlock, false, true, true );
    {
        // Read ロック
        nn::os::AcquireReadLock( &rwlock );
        CheckReaderWriterLockHeld( &rwlock, true, true, true );
        nn::os::ReleaseReadLock( &rwlock );
        CheckReaderWriterLockHeld( &rwlock, false, true, true );

        // Read ロックポーリング
        auto ret = nn::os::TryAcquireReadLock( &rwlock );
        EXPECT_TRUE( ret == true );
        CheckReaderWriterLockHeld( &rwlock, true, true, true );

        // Write ロック解除
        nn::os::ReleaseWriteLock( &rwlock );
        CheckReaderWriterLockHeld( &rwlock, true, false, true );
    }
    // Read ロック解除
    nn::os::ReleaseReadLock( &rwlock );
    CheckReaderWriterLockHeld( &rwlock, false, false, false );

    // 破棄
    nn::os::FinalizeReaderWriterLock( &rwlock );
}


//-----------------------------------------------------------------------------
// ReadLock 中の ReadLock 取得／解除と WriteLock 待機のテスト
//
//  1) メインスレッド側で ReadLock 取得
//  2) サブスレッド側で WriteLock 取得待ち
//  3) メインスレッド側で ReadLock 解除
//  4) サブスレッド側で WriteLock 取得
//  5) サブスレッド側で WriteLock 解除
//-----------------------------------------------------------------------------

void ReaderWriterLockTest_SubThread(void* arg)
{
    // ここに来た時点で Read ロック済み
    auto* rwlock = reinterpret_cast<nn::os::ReaderWriterLockType*>(arg);

    {
        // Read ロックのポーリング
        auto ret = nn::os::TryAcquireReadLock( rwlock );
        EXPECT_TRUE( ret == true );

        // Read ロック
        nn::os::AcquireReadLock( rwlock );

        // Read ロック解除×２
        nn::os::ReleaseReadLock( rwlock );
        nn::os::ReleaseReadLock( rwlock );
    }

    // Write ロックのポーリング
    auto ret = nn::os::TryAcquireWriteLock( rwlock );
    EXPECT_TRUE( ret == false );

    // Write ロック取得待ち → 一旦メインスレッドへ戻る
    EXPECT_TRUE( ++g_Sequence == 1 );
    nn::os::AcquireWriteLock( rwlock );

    // Write ロック取得
    // SIGLO-66053 の優先度継承により、シーケンス順が変わるのでコメントアウト
    ++g_Sequence;
    //EXPECT_TRUE( g_Sequence == 3 );

    // Write ロック解除
    nn::os::ReleaseWriteLock( rwlock );
}

TEST(ReaderWriterLockTest, WriteLockDuringReadLocked)
{
    // 初期化
    nn::os::ReaderWriterLockType   rwlock;
    nn::os::InitializeReaderWriterLock( &rwlock );

    nnt::os::Thread thread(ReaderWriterLockTest_SubThread, &rwlock, g_ThreadStack1, 0);

    // 初期化後の状態チェック
    CheckReaderWriterLockHeld( &rwlock, false, false, false );

    // Read ロック
    g_Sequence = 0;
    nn::os::AcquireReadLock( &rwlock );
    thread.Start();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(30) );

    EXPECT_TRUE( ++g_Sequence == 2 );

    nn::os::ReleaseReadLock( &rwlock );
    thread.Wait();

    // SIGLO-66053 の優先度継承により、シーケンス順が変わるのでコメントアウト
    ++g_Sequence;
    //EXPECT_TRUE( g_Sequence == 4 );

    // 破棄直前の状態チェック
    CheckReaderWriterLockHeld( &rwlock, false, false, false );

    // 破棄
    nn::os::FinalizeReaderWriterLock( &rwlock );
}


//-----------------------------------------------------------------------------
// WriteLock 中の ReadLock 待機と WriteLock 待機のテスト
//
//  1) メインスレッド側で WriteLock 取得
//  2) サブスレッド１側で ReadLock 取得待ち
//  3) サブスレッド２側で WriteLock 取得待ち
//  4) メインスレッド側で WriteLock 解除
//
//  5) サブスレッド２側で WriteLock 取得 ← WriteLock が先にロックを取れる
//  6) サブスレッド２側で WriteLock 解除
//
//  7) サブスレッド１側で ReadLock 取得  ← ReadLock は WriteLock 待ちが
//                                          いなくなったら取れる
//  8) サブスレッド１側で ReadLock 解除
//-----------------------------------------------------------------------------

void ReaderWriterLockTest_SubThread1(void* arg)
{
    // ここに来た時点で Read ロック済み
    auto* rwlock = reinterpret_cast<nn::os::ReaderWriterLockType*>(arg);

    // Read ロックのポーリング
    auto ret = nn::os::TryAcquireReadLock( rwlock );
    EXPECT_TRUE( ret == false );

    // Read ロック取得待ち → 一旦メインスレッドへ戻る
    EXPECT_TRUE( ++g_Sequence == 1 );
    nn::os::AcquireReadLock( rwlock );

    // Read ロック取得
    // SIGLO-66053 の優先度継承により、シーケンス順が変わるのでコメントアウト
    ++g_Sequence;
    //EXPECT_TRUE( g_Sequence == 5 );

    // Read ロック解除
    nn::os::ReleaseReadLock( rwlock );
}

void ReaderWriterLockTest_SubThread2(void* arg)
{
    // ここに来た時点で Read ロック済み
    auto* rwlock = reinterpret_cast<nn::os::ReaderWriterLockType*>(arg);

    // Write ロックのポーリング
    auto ret = nn::os::TryAcquireWriteLock( rwlock );
    EXPECT_TRUE( ret == false );

    // Write ロック取得待ち → 一旦メインスレッドへ戻る
    EXPECT_TRUE( ++g_Sequence == 2 );
    nn::os::AcquireWriteLock( rwlock );

    // Write ロック取得
    // SIGLO-66053 の優先度継承により、シーケンス順が変わるのでコメントアウト
    ++g_Sequence;
    //EXPECT_TRUE( g_Sequence == 4 );

    // Write ロック解除
    nn::os::ReleaseWriteLock( rwlock );
}

TEST(ReaderWriterLockTest, ReadLockAndWriteLockDuringWriteLocked)
{
    // 初期化
    nn::os::ReaderWriterLockType   rwlock;
    nn::os::InitializeReaderWriterLock( &rwlock );

    nnt::os::Thread thread1(ReaderWriterLockTest_SubThread1, &rwlock, g_ThreadStack1, 0);
    nnt::os::Thread thread2(ReaderWriterLockTest_SubThread2, &rwlock, g_ThreadStack2, 0);

    // 初期化後の状態チェック
    CheckReaderWriterLockHeld( &rwlock, false, false, false );

    // Write ロック
    g_Sequence = 0;
    nn::os::AcquireWriteLock( &rwlock );
    CheckReaderWriterLockHeld( &rwlock, false, true, true );

    thread1.Start();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(30) );

    thread2.Start();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(30) );

    // Write ロック解除
    EXPECT_TRUE( ++g_Sequence == 3 );
    nn::os::ReleaseWriteLock( &rwlock );
    thread1.Wait();
    thread2.Wait();

    EXPECT_TRUE( ++g_Sequence == 6 );

    // 破棄直前の状態チェック
    CheckReaderWriterLockHeld( &rwlock, false, false, false );

    // 破棄
    nn::os::FinalizeReaderWriterLock( &rwlock );
}

//-----------------------------------------------------------------------------
// WriteLock 中の ReadLock 待機と WriteLock 待機のテストその２
// ロックオーナーを維持した ReadLock 状態からのロック解除
//
//  1) メインスレッド側で WriteLock 取得
//     メインスレッド側で ReadLock 取得
//     メインスレッド側で WriteLock 解除
//
//  2) サブスレッド１側で ReadLock 取得待ち
//  3) サブスレッド２側で WriteLock 取得待ち
//
//  4) メインスレッド側で ReadLock 解除
//
//  5) サブスレッド２側で WriteLock 取得 ← WriteLock が先にロックを取れる
//  6) サブスレッド２側で WriteLock 解除
//
//  7) サブスレッド１側で ReadLock 取得  ← ReadLock は WriteLock 待ちが
//                                          いなくなったら取れる
//  8) サブスレッド１側で ReadLock 解除
//-----------------------------------------------------------------------------

void ReaderWriterLockTest2_SubThread1(void* arg)
{
    // ここに来た時点で Read ロック済み
    auto* rwlock = reinterpret_cast<nn::os::ReaderWriterLockType*>(arg);

    // Read ロックのポーリング
    auto ret = nn::os::TryAcquireReadLock( rwlock );
    EXPECT_TRUE( ret == false );

    // Read ロック取得待ち → 一旦メインスレッドへ戻る
    EXPECT_TRUE( ++g_Sequence == 1 );
    nn::os::AcquireReadLock( rwlock );

    // Read ロック取得
    // SIGLO-66053 の優先度継承により、シーケンス順が変わるのでコメントアウト
    ++g_Sequence;
    //EXPECT_TRUE( g_Sequence == 5 );

    // Read ロック解除
    nn::os::ReleaseReadLock( rwlock );
}

void ReaderWriterLockTest2_SubThread2(void* arg)
{
    // ここに来た時点で Read ロック済み
    auto* rwlock = reinterpret_cast<nn::os::ReaderWriterLockType*>(arg);

    // Write ロックのポーリング
    auto ret = nn::os::TryAcquireWriteLock( rwlock );
    EXPECT_TRUE( ret == false );

    // Write ロック取得待ち → 一旦メインスレッドへ戻る
    EXPECT_TRUE( ++g_Sequence == 2 );
    nn::os::AcquireWriteLock( rwlock );

    // Write ロック取得
    // SIGLO-66053 の優先度継承により、シーケンス順が変わるのでコメントアウト
    ++g_Sequence;
    //EXPECT_TRUE( g_Sequence == 4 );

    // Write ロック解除
    nn::os::ReleaseWriteLock( rwlock );
}

TEST(ReaderWriterLockTest, ReadLockAndWriteLockDuringOwnedReadLocked)
{
    // 初期化
    nn::os::ReaderWriterLockType   rwlock;
    nn::os::InitializeReaderWriterLock( &rwlock );

    nnt::os::Thread thread1(ReaderWriterLockTest2_SubThread1, &rwlock, g_ThreadStack1, 0);
    nnt::os::Thread thread2(ReaderWriterLockTest2_SubThread2, &rwlock, g_ThreadStack2, 0);

    // 初期化後の状態チェック
    CheckReaderWriterLockHeld( &rwlock, false, false, false );

    // Write ロック -> Read ロック -> Write ロック解除
    g_Sequence = 0;
    nn::os::AcquireWriteLock( &rwlock );
    nn::os::AcquireReadLock( &rwlock );
    nn::os::ReleaseWriteLock( &rwlock );
    CheckReaderWriterLockHeld( &rwlock, true, false, true );

    thread1.Start();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(30) );

    thread2.Start();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(30) );

    // Read ロック解除
    EXPECT_TRUE( ++g_Sequence == 3 );
    nn::os::ReleaseReadLock( &rwlock );
    thread1.Wait();
    thread2.Wait();

    EXPECT_TRUE( ++g_Sequence == 6 );

    // 破棄直前の状態チェック
    CheckReaderWriterLockHeld( &rwlock, false, false, false );

    // 破棄
    nn::os::FinalizeReaderWriterLock( &rwlock );
}

//-----------------------------------------------------------------------------
// WriteLock と ReadLock の簡単なストレステスト
// 下記のテストを一定回数繰り返す。
//
// 1) core0 のメインスレッドで WriteLock 取得中にグローバル変数を書き換え
// 2) core1 のサブスレッドで ReadLock 取得中にグローバル変数の内容チェック
// 3) core2 のサブスレッドで ReadLock 取得中にグローバル変数の内容チェック
//
//-----------------------------------------------------------------------------

int g_RwlockMiniStressCounter1 = 0;
int g_RwlockMiniStressCounter2 = 0;
int g_RwlockMiniStressCounter3 = 0;
uint64_t g_RwlockReaderLockCount = 0;
bool g_RwlockExitRequest = false;

void ReaderWriterLockMiniStressTest_SubThread(void* arg)
{
    // ここに来た時点で Read ロック済み
    auto* pRwlock = reinterpret_cast<nn::os::ReaderWriterLockType*>(arg);

    // Read ロックとグローバル変数の同値チェック
    while (!g_RwlockExitRequest)
    {
        {
            nn::os::AcquireReadLock( pRwlock );
            ++g_RwlockReaderLockCount;

            EXPECT_EQ(g_RwlockMiniStressCounter1, g_RwlockMiniStressCounter2);
            EXPECT_EQ(g_RwlockMiniStressCounter2, g_RwlockMiniStressCounter3);
            EXPECT_EQ(g_RwlockMiniStressCounter3, g_RwlockMiniStressCounter1);

            nn::os::ReleaseReadLock( pRwlock );
        }
        if (nn::os::TryAcquireReadLock( pRwlock ))
        {
            ++g_RwlockReaderLockCount;

            EXPECT_EQ(g_RwlockMiniStressCounter1, g_RwlockMiniStressCounter2);
            EXPECT_EQ(g_RwlockMiniStressCounter2, g_RwlockMiniStressCounter3);
            EXPECT_EQ(g_RwlockMiniStressCounter3, g_RwlockMiniStressCounter1);

            nn::os::ReleaseReadLock( pRwlock );
        }
    }
}

TEST(ReaderWriterLockMiniStressTest, ReaderWriterLockMiniStressTest)
{
    // 初期化
    nn::os::ReaderWriterLockType rwlock;
    nn::os::InitializeReaderWriterLock( &rwlock );
    g_RwlockExitRequest = false;

    auto coreNumber = nn::os::GetCurrentCoreNumber();
    nnt::os::Thread thread1(ReaderWriterLockMiniStressTest_SubThread, &rwlock, g_ThreadStack1, 0, (coreNumber + 1) % 3);
    nnt::os::Thread thread2(ReaderWriterLockMiniStressTest_SubThread, &rwlock, g_ThreadStack2, 0, (coreNumber + 2) % 3);

    // 初期化後の状態チェック
    CheckReaderWriterLockHeld( &rwlock, false, false, false );

    // 簡単なストレステスト
    thread1.Start();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(30) );

    thread2.Start();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(30) );

    auto startTick = nn::os::GetSystemTick();

    // Write ロック
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    for (int i=0; i<50 * 1000; ++i)
#elif defined(NN_BUILD_CONFIG_OS_WIN)
    for (int i=0; i<1 * 1000; ++i)
#else
    for (int i=0; i<10 * 1000; ++i)
#endif
    {
        {
            nn::os::AcquireWriteLock( &rwlock );

            ++g_RwlockMiniStressCounter1;
            ++g_RwlockMiniStressCounter2;
            ++g_RwlockMiniStressCounter3;

            nn::os::ReleaseWriteLock( &rwlock );
        }
        // ReadLock より WriteLock が優先されるため、適度に隙間を作る
        nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(1) );

        if (nn::os::TryAcquireWriteLock( &rwlock ))
        {
            ++g_RwlockMiniStressCounter3;
            ++g_RwlockMiniStressCounter2;
            ++g_RwlockMiniStressCounter1;

            nn::os::ReleaseWriteLock( &rwlock );
        }
        // ReadLock より WriteLock が優先されるため、適度に隙間を作る
        nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(1) );
    }

    // テスト終了要求
    g_RwlockExitRequest = true;
    thread1.Wait();
    thread2.Wait();

    EXPECT_EQ(g_RwlockMiniStressCounter1, g_RwlockMiniStressCounter2);
    EXPECT_EQ(g_RwlockMiniStressCounter2, g_RwlockMiniStressCounter3);
    EXPECT_EQ(g_RwlockMiniStressCounter3, g_RwlockMiniStressCounter1);

    // テスト時間の出力
    // NX64 Develop ビルド環境でおよそ 4.1 秒
    auto endTick = nn::os::GetSystemTick();
    NN_LOG("WriterLockCount = %lld\n", g_RwlockMiniStressCounter1);
    NN_LOG("ReaderLockCount = %lld\n", g_RwlockReaderLockCount);
    NN_LOG("Elapsed time = %lld(msec)\n", (endTick - startTick).ToTimeSpan().GetMilliSeconds());

    // 状態チェックと破棄
    CheckReaderWriterLockHeld( &rwlock, false, false, false );
    nn::os::FinalizeReaderWriterLock( &rwlock );
}

//-----------------------------------------------------------------------------

}}} // namespace nnt::os::readWriteLock

