﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Mutex 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/nn_Windows.h>
#endif

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <thread>
#endif
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "test_MutexHelper.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

namespace nnt { namespace os { namespace mutex {

//---------------------------------------------------------------------------

void
log_mutex()
{
    NNT_OS_LOG("mutex.lockHistory=0x%08x\n", nn::os::GetCurrentThread()->_lockHistory);
}

void
msg_death()
{
    log_mutex();
    NNT_OS_LOG(NN_TEXT("次の API 発行で DEATH が成立することがテスト期待値\n"));
}

//---------------------------------------------------------------------------
// 単独の LockMutex() テスト

TEST(LockMutex, test_LockMutex1)
{
    // LockMutex 再帰許可
    doTest1(true,                       // Mutex 再帰許可は true
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_LOCK,              // Thread1 は LockMutex()
            true,                       // Thread1 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(LockMutex, test_LockMutex2)
{
    // LockMutex 再帰禁止
    doTest1(false,                      // Mutex 再帰許可は false
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_LOCK,              // Thread1 は LockMutex()
            true,                       // Thread1 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(LockMutex, test_LockMutex3)
{
    // LockMutex 再帰許可（連続２回ロック）
    doTest1(true,                       // Mutex 再帰許可は true
            true,                       // Mutex 初期ロック状態は true
            THR_CALL_LOCK,              // Thread1 は LockMutex()
            true,                       // Thread1 での API 返値
            2,                          // Mutex ネストカウンタは 2
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(LockMutex, test_LockMutex4)
{
    // LockMutex 再帰禁止（連続２回ロック）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(false,                  // Mutex 再帰許可は false
                true,                   // Mutex 初期ロック状態は true
                THR_CALL_LOCK,          // Thread1 は LockMutex()
                false,                  // Thread1 での API 返値
                1,                      // Mutex ネストカウンタは 1
                THR_STATE_EXITED)       // Thread1 は終了
    , "");
}

TEST(LockMutex, test_LockMutex5)
{
    // LockMutex 再帰許可（２スレッドでロック）
    doTest2(true,                       // Mutex 再帰許可は true
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_LOCK,              // Thread1 は LockMutex()
            THR_WAIT_LOCK,              // Thread2 は LockMutex()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_WAITING2,         // Thread1 は終了待ち
            THR_STATE_WAITING);         // Thread2 は待ち状態
}

TEST(LockMutex, test_LockMutex6)
{
    // LockMutex 再帰禁止（２スレッドでロック）
    doTest2(false,                      // Mutex 再帰許可は false
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_LOCK,              // Thread1 は LockMutex()
            THR_WAIT_LOCK,              // Thread2 は LockMutex()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_WAITING2,         // Thread1 は終了
            THR_STATE_WAITING);         // Thread2 は待ち状態
}

//---------------------------------------------------------------------------
// 単独の TryLockMutex() テスト

TEST(TryLockMutex, test_TryLockMutex1)
{
    // TryLockMutex 再帰許可
    doTest1(true,                       // Mutex 再帰許可は true
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_TRY_LOCK,          // Thread1 は TryLockMutex()
            true,                       // Thread1 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_EXITED);          // Thread1 は待ち状態
}

TEST(TryLockMutex, test_TryLockMutex2)
{
    // TryLockMutex 再帰禁止
    doTest1(false,                      // Mutex 再帰許可は false
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_TRY_LOCK,          // Thread1 は TryLockMutex()
            true,                       // Thread1 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryLockMutex, test_TryLockMutex3)
{
    // TryLockMutex 再帰許可（連続２回ロック）
    doTest1(true,                       // Mutex 再帰許可は true
            true,                       // Mutex 初期ロック状態は true
            THR_CALL_TRY_LOCK,          // Thread1 は TryLockMutex()
            true,                       // Thread1 での API 返値
            2,                          // Mutex ネストカウンタは 2
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryLockMutex, test_TryLockMutex4)
{
    // TryLockMutex 再帰禁止（連続２回ロック）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(false,                  // Mutex 再帰許可は false
                true,                   // Mutex 初期ロック状態は true
                THR_CALL_TRY_LOCK,      // Thread1 は TryLockMutex()
                false,                  // Thread1 での API 返値
                1,                      // Mutex ネストカウンタは 1
                THR_STATE_EXITED)       // Thread1 は終了
    , "");
}

TEST(TryLockMutex, test_TryLockMutex5)
{
    // TryLockMutex 再帰許可（２スレッドでロック）
    doTest2(true,                       // Mutex 再帰許可は true
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_TRY_LOCK,          // Thread1 は TryLockMutex()
            THR_CALL_TRY_LOCK,          // Thread2 は TryLockMutex()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_EXITED,           // Thread1 は終了待ち
            THR_STATE_EXITED);          // Thread2 は待ち状態
}

TEST(TryLockMutex, test_TryLockMutex6)
{
    // TryLockMutex 再帰禁止（２スレッドでロック）
    doTest2(false,                      // Mutex 再帰許可は false
            false,                      // Mutex 初期ロック状態は false
            THR_CALL_TRY_LOCK,          // Thread1 は TryLockMutex()
            THR_CALL_TRY_LOCK,          // Thread2 は TryLockMutex()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            1,                          // Mutex ネストカウンタは 1
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は待ち状態
}


//---------------------------------------------------------------------------
// 単独の UnlockMutex() テスト

TEST(UnlockMutex, test_UnlockMutex1)
{
    // UnlockMutex 再帰許可
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(true,                   // Mutex 再帰許可は true
                false,                  // Mutex 初期ロック状態は false
                THR_CALL_UNLOCK,        // Thread1 は UnlockMutex()
                false,                  // Thread1 での API 返値
                0,                      // Mutex ネストカウンタは 0
                THR_STATE_EXITED)       // Thread1 は待ち状態
    , "");
}

TEST(UnlockMutex, test_UnlockMutex2)
{
    // UnlockMutex 再帰禁止
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(false,                  // Mutex 再帰許可は false
                false,                  // Mutex 初期ロック状態は false
                THR_CALL_UNLOCK,        // Thread1 は UnlockMutex()
                false,                  // Thread1 での API 返値
                0,                      // Mutex ネストカウンタは 0
                THR_STATE_EXITED)       // Thread1 は終了
    , "");
}

TEST(UnlockMutex, test_UnlockMutex3)
{
    // UnlockMutex 再帰許可
    doTest1(true,                       // Mutex 再帰許可は true
            true,                       // Mutex 初期ロック状態は true
            THR_CALL_UNLOCK,            // Thread1 は UnlockMutex()
            true,                       // Thread1 での API 返値
            0,                          // Mutex ネストカウンタは 0
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(UnlockMutex, test_UnlockMutex4)
{
    // UnlockMutex 再帰禁止
    doTest1(false,                      // Mutex 再帰許可は false
            true,                       // Mutex 初期ロック状態は true
            THR_CALL_UNLOCK,            // Thread1 は UnlockMutex()
            true,                       // Thread1 での API 返値
            0,                          // Mutex ネストカウンタは 0
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(UnlockMutex, test_UnlockMutex5)
{
    // UnlockMutex 再帰許可（２スレッドでロック）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest2(true,                   // Mutex 再帰許可は true
                false,                  // Mutex 初期ロック状態は false
                THR_CALL_UNLOCK,        // Thread1 は UnlockMutex()
                THR_CALL_UNLOCK,        // Thread2 は UnlockMutex()
                false,                  // Thread1 での API 返値
                false,                  // Thread2 での API 返値
                0,                      // Mutex ネストカウンタは 0
                THR_STATE_EXITED,       // Thread1 は終了待ち
                THR_STATE_EXITED)       // Thread2 は待ち状態
    , "");
}

TEST(UnlockMutex, test_UnlockMutex6)
{
    // UnlockMutex 再帰禁止（２スレッドでロック）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest2(true,                   // Mutex 再帰許可は true
                true,                   // Mutex 初期ロック状態は true
                THR_CALL_UNLOCK,        // Thread1 は UnlockMutex()
                THR_CALL_UNLOCK,        // Thread2 は UnlockMutex()
                true,                   // Thread1 での API 返値
                false,                  // Thread2 での API 返値
                0,                      // Mutex ネストカウンタは 0
                THR_STATE_EXITED,       // Thread1 は終了
                THR_STATE_EXITED)       // Thread2 は待ち状態
    , "");
}

//---------------------------------------------------------------------------
// Lock 獲得中に他スレッドから Unlock してもエラー

TEST(CombinedMutexTest, test_LockAndUnlock1)
{
    // Thread1 で Lock()、Thread2 で Unlock()
    EXPECT_DEATH_IF_SUPPORTED(
        doTest2(true,                   // Mutex 再帰許可は true
                false,                  // Mutex 初期ロック状態は false
                THR_CALL_LOCK,          // Thread1 は LockMutex()
                THR_CALL_UNLOCK,        // Thread2 は UnlockMutex()
                true,                   // Thread1 での API 返値
                false,                  // Thread2 での API 返値
                1,                      // Mutex ネストカウンタは 1
                THR_STATE_WAITING2,     // Thread1 は待ち解除して終了
                THR_STATE_EXITED)       // Thread2 は終了
    , "");
}

TEST(CombinedMutexTest, test_LockAndUnlock2)
{
    // Thread1 で Lock()、Thread2 で Unlock()
    EXPECT_DEATH_IF_SUPPORTED(
        doTest2(true,                   // Mutex 再帰許可は true
                true,                   // Mutex 初期ロック状態は true
                THR_CALL_LOCK,          // Thread1 は LockMutex()
                THR_CALL_UNLOCK,        // Thread2 は UnlockMutex()
                true,                   // Thread1 での API 返値
                false,                  // Thread2 での API 返値
                2,                      // Mutex ネストカウンタは 2
                THR_STATE_WAITING2,     // Thread1 は待ち解除して終了
                THR_STATE_EXITED)       // Thread2 は終了
    , "");
}

//---------------------------------------------------------------------------
// Mutex のロックレベル機能の通常テスト

TEST(MutexLockLevelTest, test_NonRecursiveLockTest)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    // ロックレベル 1, 3, 5 の Mutex を順番に獲得（全て Non-Recursive）

    nn::os::MutexType   mutex1, mutex3, mutex5;

    nn::os::InitializeMutex( &mutex1, false, 1 );
    nn::os::InitializeMutex( &mutex3, false, 3 );
    nn::os::InitializeMutex( &mutex5, false, 5 );

    // ロックレベル検査を含む LockMutex の獲得
    log_mutex(); nn::os::LockMutex( &mutex1 );
    log_mutex(); nn::os::LockMutex( &mutex3 );
    log_mutex(); nn::os::LockMutex( &mutex5 );
    log_mutex(); nn::os::UnlockMutex( &mutex5 );
    log_mutex(); nn::os::UnlockMutex( &mutex3 );
    log_mutex(); nn::os::UnlockMutex( &mutex1 );
    log_mutex();
    NNT_OS_LOG("LockLevelTest: LockMutex and UnlockMutex test Passed.\n");

    // ロックレベル検査を含む TryLockMutex の獲得
    log_mutex(); EXPECT_TRUE( nn::os::TryLockMutex( &mutex1 ) );
    log_mutex(); EXPECT_TRUE( nn::os::TryLockMutex( &mutex3 ) );
    log_mutex(); EXPECT_TRUE( nn::os::TryLockMutex( &mutex5 ) );
    log_mutex(); nn::os::UnlockMutex( &mutex5 );
    log_mutex(); nn::os::UnlockMutex( &mutex3 );
    log_mutex(); nn::os::UnlockMutex( &mutex1 );
    log_mutex();
    NNT_OS_LOG("LockLevelTest: TryLockMutex and UnlockMutex test Passed.\n");

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}


TEST(MutexLockLevelTest, test_RecursiveLockTest)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    // ロックレベル 2, 3, 4 の Mutex を順番に獲得（全て Non-Recursive）

    nn::os::MutexType   mutex2, mutex3, mutex4;

    nn::os::InitializeMutex( &mutex2, true, 2 );
    nn::os::InitializeMutex( &mutex3, true, 3 );
    nn::os::InitializeMutex( &mutex4, true, 4 );

    // ロックレベル検査を含む LockMutex の獲得
    log_mutex(); nn::os::LockMutex( &mutex2 );
    log_mutex(); nn::os::LockMutex( &mutex3 );
    log_mutex(); nn::os::LockMutex( &mutex3 );
    log_mutex(); nn::os::LockMutex( &mutex4 );
    log_mutex(); nn::os::LockMutex( &mutex4 );
    log_mutex(); nn::os::LockMutex( &mutex4 );
    log_mutex(); nn::os::UnlockMutex( &mutex4 );
    log_mutex(); nn::os::UnlockMutex( &mutex4 );
    log_mutex(); nn::os::UnlockMutex( &mutex4 );
    log_mutex(); nn::os::UnlockMutex( &mutex3 );
    log_mutex(); nn::os::UnlockMutex( &mutex3 );
    log_mutex(); nn::os::UnlockMutex( &mutex2 );
    log_mutex();
    NNT_OS_LOG("LockLevelTest: LockMutex and UnlockMutex test Passed.\n");

    // ロックレベル検査を含む TryLockMutex の獲得
    log_mutex(); EXPECT_TRUE( nn::os::TryLockMutex( &mutex2 ) );
    log_mutex(); EXPECT_TRUE( nn::os::TryLockMutex( &mutex3 ) );
    log_mutex(); EXPECT_TRUE( nn::os::TryLockMutex( &mutex3 ) );
    log_mutex(); EXPECT_TRUE( nn::os::TryLockMutex( &mutex4 ) );
    log_mutex(); EXPECT_TRUE( nn::os::TryLockMutex( &mutex4 ) );
    log_mutex(); EXPECT_TRUE( nn::os::TryLockMutex( &mutex4 ) );
    log_mutex(); nn::os::UnlockMutex( &mutex4 );
    log_mutex(); nn::os::UnlockMutex( &mutex4 );
    log_mutex(); nn::os::UnlockMutex( &mutex4 );
    log_mutex(); nn::os::UnlockMutex( &mutex3 );
    log_mutex(); nn::os::UnlockMutex( &mutex3 );
    log_mutex(); nn::os::UnlockMutex( &mutex2 );
    log_mutex();
    NNT_OS_LOG("LockLevelTest: TryLockMutex and UnlockMutex test Passed.\n");

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// Mutex のロックレベル機能の異常検知テスト

int
testImpl_NonRecursiveLockTest()
{
    // ロックレベル 1 の Mutex を順番に獲得（全て Non-Recursive）

    nn::os::MutexType   mutex1a, mutex1b;

    nn::os::InitializeMutex( &mutex1a, false, 1 );
    nn::os::InitializeMutex( &mutex1b, false, 1 );

    // ロックレベル検査を含む LockMutex の獲得
    log_mutex(); nn::os::LockMutex( &mutex1a );
    msg_death(); nn::os::LockMutex( &mutex1b );
    log_mutex(); nn::os::UnlockMutex( &mutex1b );
    log_mutex(); nn::os::UnlockMutex( &mutex1a );
    log_mutex();

    return 0;
}

TEST(MutexLockLevelAssertionTest, test_NonRecursiveLockTest)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    EXPECT_DEATH_IF_SUPPORTED( testImpl_NonRecursiveLockTest(), "" );
    NNT_OS_LOG("LockLevelDeathTest: LockMutex and UnlockMutex test Passed.\n");

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

//----------

int
testImpl_NonRecursiveTryLockTest()
{
    // ロックレベル 1 の Mutex を順番に獲得（全て Non-Recursive）

    nn::os::MutexType   mutex1a, mutex1b;

    nn::os::InitializeMutex( &mutex1a, false, 1 );
    nn::os::InitializeMutex( &mutex1b, false, 1 );

    // ロックレベル検査を含む LockMutex の獲得
    log_mutex(); nn::os::TryLockMutex( &mutex1a );
    msg_death(); nn::os::TryLockMutex( &mutex1b );
    log_mutex(); nn::os::UnlockMutex( &mutex1b );
    log_mutex(); nn::os::UnlockMutex( &mutex1a );
    log_mutex();

    return 0;
}

TEST(MutexLockLevelAssertionTest, test_NonRecursiveTryLockTest)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    EXPECT_DEATH_IF_SUPPORTED( testImpl_NonRecursiveTryLockTest(), "" );
    NNT_OS_LOG("LockLevelDeathTest: TryLockMutex and UnlockMutex test Passed.\n");

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

//----------

int
testImpl_RecursiveLockTest()
{
    // ロックレベル 2 の Mutex を順番に獲得（全て Recursive）

    nn::os::MutexType   mutex2a, mutex2b;

    nn::os::InitializeMutex( &mutex2a, true, 2 );
    nn::os::InitializeMutex( &mutex2b, true, 2 );

    // ロックレベル検査を含む LockMutex の獲得
    log_mutex(); nn::os::LockMutex( &mutex2a );
    log_mutex(); nn::os::LockMutex( &mutex2a );
    msg_death(); nn::os::LockMutex( &mutex2b );
    log_mutex(); nn::os::UnlockMutex( &mutex2b );
    log_mutex(); nn::os::UnlockMutex( &mutex2a );
    log_mutex(); nn::os::UnlockMutex( &mutex2a );
    log_mutex();

    return 0;
}

TEST(MutexLockLevelAssertionTest, test_RecursiveLockTest)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    EXPECT_DEATH_IF_SUPPORTED( testImpl_RecursiveLockTest(), "" );
    NNT_OS_LOG("LockLevelDeathTest: LockMutex and UnlockMutex test Passed.\n");

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

//----------

int
testImpl_RecursiveTryLockTest()
{
    // ロックレベル 2 の Mutex を順番に獲得（全て Recursive）

    nn::os::MutexType   mutex2a, mutex2b;

    nn::os::InitializeMutex( &mutex2a, true, 2 );
    nn::os::InitializeMutex( &mutex2b, true, 2 );

    // ロックレベル検査を含む LockMutex の獲得
    log_mutex(); nn::os::TryLockMutex( &mutex2a );
    log_mutex(); nn::os::TryLockMutex( &mutex2a );
    msg_death(); nn::os::TryLockMutex( &mutex2b );
    log_mutex(); nn::os::UnlockMutex( &mutex2b );
    log_mutex(); nn::os::UnlockMutex( &mutex2a );
    log_mutex(); nn::os::UnlockMutex( &mutex2a );
    log_mutex();

    return 0;
}

TEST(MutexLockLevelAssertionTest, test_RecursiveTryLockTest)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    EXPECT_DEATH_IF_SUPPORTED( testImpl_RecursiveTryLockTest(), "" );
    NNT_OS_LOG("LockLevelDeathTest: TryLockMutex and UnlockMutex test Passed.\n");

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

//----------

int
testImpl_RecursiveNonRecursiveLockTest()
{
    // ロックレベル 3 の Mutex を順番に獲得（Recursive -> NonRecursive）

    nn::os::MutexType   mutex3a, mutex3b;

    nn::os::InitializeMutex( &mutex3a, true,  3 );
    nn::os::InitializeMutex( &mutex3b, false, 3 );

    // ロックレベル検査を含む LockMutex の獲得
    log_mutex(); nn::os::LockMutex( &mutex3a );
    log_mutex(); nn::os::LockMutex( &mutex3a );
    msg_death(); nn::os::LockMutex( &mutex3b );
    log_mutex(); nn::os::UnlockMutex( &mutex3b );
    log_mutex(); nn::os::UnlockMutex( &mutex3a );
    log_mutex(); nn::os::UnlockMutex( &mutex3a );
    log_mutex();

    return 0;
}

TEST(MutexLockLevelAssertionTest, test_RecursiveNonRecursiveLockTest)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    EXPECT_DEATH_IF_SUPPORTED( testImpl_RecursiveNonRecursiveLockTest(), "" );
    NNT_OS_LOG("LockLevelDeathTest: LockMutex and UnlockMutex test Passed.\n");

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

//----------

int
testImpl_RecursiveNonRecursiveTryLockTest()
{
    // ロックレベル 3 の Mutex を順番に獲得（Recursive -> NonRecursive）

    nn::os::MutexType   mutex3a, mutex3b;

    nn::os::InitializeMutex( &mutex3a, true,  3 );
    nn::os::InitializeMutex( &mutex3b, false, 3 );

    // ロックレベル検査を含む TryLockMutex の獲得
    log_mutex(); nn::os::TryLockMutex( &mutex3a );
    log_mutex(); nn::os::TryLockMutex( &mutex3a );
    msg_death(); nn::os::TryLockMutex( &mutex3b );
    log_mutex(); nn::os::UnlockMutex( &mutex3b );
    log_mutex(); nn::os::UnlockMutex( &mutex3a );
    log_mutex(); nn::os::UnlockMutex( &mutex3a );
    log_mutex();

    return 0;
}

TEST(MutexLockLevelAssertionTest, test_RecursiveNonRecursiveTryLockTest)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    EXPECT_DEATH_IF_SUPPORTED( testImpl_RecursiveNonRecursiveTryLockTest(), "" );
    NNT_OS_LOG("LockLevelDeathTest: TryLockMutex and UnlockMutex test Passed.\n");

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

//----------

int
testImpl_NonRecursiveRecursiveLockTest()
{
    // ロックレベル 3 の Mutex を順番に獲得（Recursive -> NonRecursive）

    nn::os::MutexType   mutex3a, mutex3b;

    nn::os::InitializeMutex( &mutex3a, false, 3 );
    nn::os::InitializeMutex( &mutex3b, true,  3 );

    // ロックレベル検査を含む LockMutex の獲得
    log_mutex(); nn::os::LockMutex( &mutex3a );
    msg_death(); nn::os::LockMutex( &mutex3b );
    log_mutex(); nn::os::UnlockMutex( &mutex3b );
    log_mutex(); nn::os::UnlockMutex( &mutex3a );
    log_mutex();

    return 0;
}

TEST(MutexLockLevelAssertionTest, test_NonRecursiveRecursiveLockTest)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    EXPECT_DEATH_IF_SUPPORTED( testImpl_RecursiveNonRecursiveLockTest(), "" );
    NNT_OS_LOG("LockLevelDeathTest: LockMutex and UnlockMutex test Passed.\n");

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

//----------

int
testImpl_NonRecursiveRecursiveTryLockTest()
{
    // ロックレベル 3 の Mutex を順番に獲得（Recursive -> NonRecursive）

    nn::os::MutexType   mutex3a, mutex3b;

    nn::os::InitializeMutex( &mutex3a, false, 3 );
    nn::os::InitializeMutex( &mutex3b, true,  3 );

    // ロックレベル検査を含む TryLockMutex の獲得
    log_mutex(); nn::os::TryLockMutex( &mutex3a );
    msg_death(); nn::os::TryLockMutex( &mutex3b );
    log_mutex(); nn::os::UnlockMutex( &mutex3b );
    log_mutex(); nn::os::UnlockMutex( &mutex3a );
    log_mutex();

    return 0;
}

TEST(MutexLockLevelAssertionTest, test_NonRecursiveRecursiveTryLockTest)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    EXPECT_DEATH_IF_SUPPORTED( testImpl_RecursiveNonRecursiveTryLockTest(), "" );
    NNT_OS_LOG("LockLevelDeathTest: TryLockMutex and UnlockMutex test Passed.\n");

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// TORIAEZU: for SIGLO-4262
//
//  通常では #if 0 にして、このテストは実施しないでおく。
//  SIGLO-4262 の不具合再現を確認したい場合だけ、
//  ここを #if 1 にして Mutex の単独テストとして実行する。
//
//  VisualStudio 上で試す場合は、CTRL+F5 ではなく F5 で実行し、ログの最後で、
//  「(-----): 静的オブジェクトのデストラクタ内で LockMutex が可能か ... OK」
//  が表示されていれば、不具合を回避していることになる。
//
//  不具合を回避できていない場合は以下のような表示になる。
//  「(-----): 静的オブジェクトのデストラクタ内で LockMutex が可能か
//    SDK Assertion Failure: 'mutex->_nestCount >= 1' at ～ in nn::os::LockMutex.
//    内部エラー：ミューテックスの内部状態が異常です。」
//
#if 0
#if defined(NN_BUILD_CONFIG_OS_WIN32)
//---------------------------------------------------------------------------
// 静的オブジェクト内での Mutex 操作テスト

class TestStaticObject
{
public:
    // コンストラクタ
    TestStaticObject()
    {
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("静的オブジェクトのデストラクタ内で LockMutex が可能かをテスト。\n"));

        nn::os::InitializeMutex( &m_Mutex, true, 0 );
    }

    // デストラクタ
    ~TestStaticObject()
    {
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("静的オブジェクトのデストラクタ内で LockMutex が可能か"));

        nn::os::LockMutex( &m_Mutex );
        nn::os::UnlockMutex( &m_Mutex );

        CheckBool( true );

        nn::os::FinalizeMutex( &m_Mutex );
    }

private:
    nn::os::MutexType   m_Mutex;
};

TEST(LockMutexInStaticObject, test_LockMutexInStaticObject)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    {   // 静的オブジェクトを構築して強制終了する
        static TestStaticObject s_Object;

        // 自分自身のウィンドウに対して WM_CLOSE メッセージを送信して強制終了
        HWND hWnd = ::GetConsoleWindow();
        ::SendMessage(hWnd, WM_CLOSE, 0 , 0);
    }

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}
#endif /* defined(NN_BUILD_CONFIG_OS_WIN32) */
#endif

#if defined(NN_BUILD_CONFIG_OS_WIN32)
//---------------------------------------------------------------------------
// std::thread で使用するテスト用スレッド関数
//
void StdThreadTestFunction()
{
    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッドが起動しました。\n"));

    nn::os::MutexType   mutex;
    nn::os::MutexType   recursiveMutex;
    nn::os::InitializeMutex( &mutex, false, 0 );
    nn::os::InitializeMutex( &recursiveMutex, true, 0 );


    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッド内で mutex を操作"));
    nn::os::LockMutex( &mutex );
    nn::os::UnlockMutex( &mutex );
    CheckBool( true );

    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッド内で recursiveMutex を操作"));
    nn::os::LockMutex( &recursiveMutex );
    nn::os::LockMutex( &recursiveMutex );
    nn::os::LockMutex( &recursiveMutex );
    nn::os::LockMutex( &recursiveMutex );
    nn::os::UnlockMutex( &recursiveMutex );
    nn::os::UnlockMutex( &recursiveMutex );
    nn::os::UnlockMutex( &recursiveMutex );
    nn::os::UnlockMutex( &recursiveMutex );
    CheckBool( true );

    nn::os::FinalizeMutex( &mutex );
    nn::os::FinalizeMutex( &recursiveMutex );

    NNT_OS_LOG(NN_TEXT("sub:  std::thread で生成されたスレッドが終了します。\n"));
}

//---------------------------------------------------------------------------
// std::thread でスレッドを作成し、std::thread::join() する。
//
TEST(StdThreadTest, test_StdThreadAndJoin)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        NNT_OS_LOG(NN_TEXT("main: std::thread でスレッドを生成します。\n"));

        std::thread stdThread( StdThreadTestFunction );

        NNT_OS_LOG(NN_TEXT("main: std::thread で生成したスレッドに join します。\n"));

        stdThread.join();

        NNT_OS_LOG(NN_TEXT("main: std::thread のテストを終了します。\n"));
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// std::thread でスレッドを作成し、std::thread::detach() する。
//
TEST(StdThreadTest, test_StdThreadAndDetach)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        NNT_OS_LOG(NN_TEXT("main: std::thread でスレッドを生成します。\n"));

        std::thread stdThread( StdThreadTestFunction );

        NNT_OS_LOG(NN_TEXT("main: std::thread で生成したスレッドを detach します。\n"));

        stdThread.detach();

        NNT_OS_LOG(NN_TEXT("main: std::thread のテストを終了します。\n"));
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}
#endif /* defined(NN_BUILD_CONFIG_OS_WIN32) */


//---------------------------------------------------------------------------
// nn::os::Mutex クラスの NonRecursive テスト
//

NN_ALIGNAS(4096) char   g_ThreadStack[8192];
int g_MutexClassTestCounter;

void test_MutexClassTestThreadNonRecursive(void* arg)
{
    nn::os::Mutex& mutex = *reinterpret_cast<nn::os::Mutex*>( arg );

    for (int i=0; i<50000; ++i)
    {
        {   // lock_guard を利用
            std::lock_guard<nn::os::Mutex>  lock( mutex );

            ++g_MutexClassTestCounter;
            nn::os::YieldThread();
        }
        nn::os::YieldThread();

        {   // unique_lock を利用
            std::unique_lock<nn::os::Mutex>  lock( mutex );

            ++g_MutexClassTestCounter;
            nn::os::YieldThread();
        }
        nn::os::YieldThread();
    }
}

TEST(MutexClassTest, test_MutexClassNonRecursive)
{
    CLEAR_GOOGLE_TEST();

    // ミューテックスの構築（引数１つのコンストラクタで初期化）
    nn::os::Mutex   mutex( false );
    g_MutexClassTestCounter = 0;

    // スレッドの作成
    nn::os::ThreadType  thread;
    nn::os::CreateThread( &thread, test_MutexClassTestThreadNonRecursive, &mutex, g_ThreadStack, sizeof(g_ThreadStack), nn::os::GetThreadPriority(nn::os::GetCurrentThread()));

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("Mutex クラスの NonRecursive テスト\n"));
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("親スレッドと子スレッドで同じグローバル変数をインクリメント\n"));

    // テスト本体（スレッド実行開始）
    int countTryLock = 0;
    nn::os::StartThread( &thread );
    for (int i=0; i<50000; ++i)
    {
        // メンバ変数の Lock() を利用
        {
            mutex.Lock();

            ++g_MutexClassTestCounter;
            nn::os::YieldThread();

            mutex.Unlock();
        }

        // メンバ変数の TryLock() を利用
        {
            while (!mutex.TryLock())
            {
                // TryLock が成功するまでループするだけ
                ++countTryLock;
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
            }

            ++g_MutexClassTestCounter;
            nn::os::YieldThread();

            mutex.Unlock();
        }
    }

    // スレッドの終了待ち
    nn::os::WaitThread( &thread );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("グローバル変数カウンタが期待値通りか"));
    CheckParam( g_MutexClassTestCounter, 200000 );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("TryLock の空回り数＝%d\n"), countTryLock);

    // スレッドの破棄
    nn::os::DestroyThread( &thread );

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// nn::os::Mutex クラスの Recursive テスト
//

void test_MutexClassTestThreadRecursive(void* arg)
{
    nn::os::Mutex& mutex = *reinterpret_cast<nn::os::Mutex*>( arg );

    for (int i=0; i<50000; ++i)
    {
        mutex.Lock();

        ++g_MutexClassTestCounter;
        nn::os::YieldThread();

        {
            while (!mutex.TryLock())
            {
                // 成功するまでループ
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
            }

            ++g_MutexClassTestCounter;
            nn::os::YieldThread();

            mutex.Unlock();
        }
        mutex.Unlock();
        nn::os::YieldThread();
    }
}

TEST(MutexClassTest, test_MutexClassRecursive)
{
    CLEAR_GOOGLE_TEST();

    // ミューテックスの構築（引数２つのコンストラクタで初期化）
    nn::os::Mutex   mutex( true, 0 );
    g_MutexClassTestCounter = 0;

    // スレッドの作成
    nn::os::ThreadType  thread;
    nn::os::CreateThread( &thread, test_MutexClassTestThreadRecursive, &mutex, g_ThreadStack, sizeof(g_ThreadStack), nn::os::GetThreadPriority(nn::os::GetCurrentThread()));

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("Mutex クラスの Recursive テスト\n"));
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("親スレッドと子スレッドで同じグローバル変数をインクリメント\n"));

    // テスト本体（スレッド実行開始）
    int countTryLock = 0;
    nn::os::StartThread( &thread );
    for (int i=0; i<50000; ++i)
    {
        // メンバ変数の Lock() を利用
        {
            mutex.Lock();

            ++g_MutexClassTestCounter;
            nn::os::YieldThread();

            // メンバ変数の TryLock() を利用
            {
                while (!mutex.TryLock())
                {
                    // TryLock が成功するまでループするだけ
                    ++countTryLock;
                    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
                }

                ++g_MutexClassTestCounter;
                nn::os::YieldThread();

                mutex.Unlock();
            }
            mutex.Unlock();
        }
    }

    // スレッドの終了待ち
    nn::os::WaitThread( &thread );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("グローバル変数カウンタが期待値通りか"));
    CheckParam( g_MutexClassTestCounter, 200000 );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("TryLock の空回り数＝%d"), countTryLock);
    CheckBool( countTryLock == 0 );

    // スレッドの破棄
    nn::os::DestroyThread( &thread );

    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// nn::os::MutexType 構造体の静的初期化のテスト
//

nn::os::MutexType   g_StaticMutex1 = NN_OS_MUTEX_INITIALIZER(false);
nn::os::MutexType   g_StaticMutex2 = NN_OS_MUTEX_INITIALIZER(true);

void test_StaticMutexTestThread(void* arg)
{
    int& failedCountOfTryLock = *reinterpret_cast<int*>(arg);

    while (failedCountOfTryLock < 10)
    {
        nn::os::LockMutex( &g_StaticMutex1 );
        nn::os::YieldThread();

        nn::os::UnlockMutex( &g_StaticMutex1 );
        nn::os::YieldThread();

        nn::os::LockMutex( &g_StaticMutex2 );
        nn::os::YieldThread();

        nn::os::UnlockMutex( &g_StaticMutex2 );
        nn::os::YieldThread();
    }
}

TEST(MutexTypeStaticInitializerTest, test_MutexTypeStaticInitializer)
{
    CLEAR_GOOGLE_TEST();

    // 静的初期化された MutexType オブジェクトのダンプ
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("静的初期化された g_StaticMutex1 の内部状態（チェックなし）\n"));
    for (size_t i=0; i<sizeof(nn::os::MutexType) / sizeof(uint32_t); ++i)
    {
        SEQ_NONE();
        NNT_OS_LOG("g_StaticMutex1[%d]= 0x%08x\n", i, ((uint32_t*)&g_StaticMutex1)[i]);
    }

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("静的初期化された g_StaticMutex2 の内部状態（チェックなし）\n"));
    for (size_t i=0; i<sizeof(nn::os::MutexType) / sizeof(uint32_t); ++i)
    {
        SEQ_NONE();
        NNT_OS_LOG("g_StaticMutex2[%d]= 0x%08x\n", i, ((uint32_t*)&g_StaticMutex2)[i]);
    }

    // スレッドの作成
    int failedCountOfTryLock = 0;
    int count = 0;
    nn::os::ThreadType  thread;
    nn::os::CreateThread( &thread, test_StaticMutexTestThread, &failedCountOfTryLock, g_ThreadStack, sizeof(g_ThreadStack), nn::os::GetThreadPriority(nn::os::GetCurrentThread()));
    nn::os::StartThread( &thread );

    // 静的初期化された MutexType オブジェクトを利用するテスト
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("静的初期化された g_StaticMutex で保護されたカウンタの操作\n"));
    while (failedCountOfTryLock < 10)
    {
        bool ret1 = nn::os::TryLockMutex( &g_StaticMutex1 );
        if (!ret1)
        {
            ++failedCountOfTryLock;
            nn::os::LockMutex( &g_StaticMutex1 );
        }
        nn::os::YieldThread();

        bool ret2 = nn::os::TryLockMutex( &g_StaticMutex2 );
        if (!ret2)
        {
            ++failedCountOfTryLock;
            nn::os::LockMutex( &g_StaticMutex2 );
        }
        nn::os::YieldThread();

        ++count;

        nn::os::UnlockMutex( &g_StaticMutex2 );
        nn::os::YieldThread();
        nn::os::UnlockMutex( &g_StaticMutex1 );
    }

    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("TryLockMutex() に 10 回失敗するまでのループ数： %d\n"), count);

    // 再帰ロックの確認
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("静的初期化された g_StaticMutex2 が再帰可能か"));
    nn::os::LockMutex( &g_StaticMutex2 );
    nn::os::LockMutex( &g_StaticMutex2 );
    nn::os::LockMutex( &g_StaticMutex2 );
    nn::os::UnlockMutex( &g_StaticMutex2 );
    nn::os::UnlockMutex( &g_StaticMutex2 );
    nn::os::UnlockMutex( &g_StaticMutex2 );
    CheckBool( true );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("静的初期化された MutexType オブジェクトがファイナライズ可能か"));
    nn::os::FinalizeMutex( &g_StaticMutex1 );
    nn::os::FinalizeMutex( &g_StaticMutex2 );
    CheckBool( true );

    // エージングテスト用にもう一度初期化状態に戻しておく
    nn::os::InitializeMutex( &g_StaticMutex1, false, 0 );
    nn::os::InitializeMutex( &g_StaticMutex2, true,  0 );

    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
//  Mutex クラスの型変換関数のテスト
//  ここでは、Mutex クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  ここでは、GetBase() および operator MutexType&() の動作テストを行なう。
//
TEST(MutexClass, test_MutexClassTestTypeExchange)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("mutex インスタンスの生成\n"));
        nn::os::Mutex mutex(false);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("mutex.GetBase() で MutexType オブジェクトを取得"));
        nn::os::MutexType*  mutexType = mutex.GetBase();
        CheckBool( mutexType != NULL );

        // mutex クラスを使ってロックを獲得
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("mutex.Lock() でロックを獲得"));
        mutex.Lock();
        CheckBool( true );

        // mutexType オブジェクトを使ってアンロックを実施
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("mutexType を使ってアンロックを実施"));
        nn::os::UnlockMutex(mutexType);
        CheckBool( true );

        // operator MutexType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator MutexType&() を使ってオブジェクトの参照を取得"));
        nn::os::MutexType& mutexTypeRefer = mutex;
        CheckBool( &mutexTypeRefer == mutexType );

        // operator const MutexType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator const MutexType&() を使ってオブジェクトの参照を取得"));
        const nn::os::MutexType& mutexTypeReferConst = mutex;
        CheckBool( &mutexTypeReferConst == mutexType );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// nn::os::MutexType および Mutex クラスの IsMutexLockedByCurrentThread() テスト
//
nn::os::MutexType   g_MutexTypeForIsLockedTest;
nn::os::Mutex       g_MutexClassForIsLockedTest(false);
nn::os::Barrier     g_BarrierForIsLockedTest(2);

void SubThreadForMutexLockedByCurrentThreadTest(void* arg)
{
    NN_UNUSED(arg);

    // ２つのミューテックスをロック
    nn::os::LockMutex(&g_MutexTypeForIsLockedTest);
    g_MutexClassForIsLockedTest.Lock();

    g_BarrierForIsLockedTest.Await();
    g_BarrierForIsLockedTest.Await();

    // ２つのミューテックスをアンロック
    g_MutexClassForIsLockedTest.Unlock();
    nn::os::UnlockMutex(&g_MutexTypeForIsLockedTest);
}

TEST(MutexLockedByCurrentThread, test_IsMutexLockedByCurrentThread)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    nn::os::InitializeMutex(&g_MutexTypeForIsLockedTest, false, 0);

    // 子スレッドを生成
    nn::os::ThreadType  thread;
    nn::os::CreateThread( &thread, SubThreadForMutexLockedByCurrentThreadTest, NULL, g_ThreadStack, sizeof(g_ThreadStack), nn::os::GetThreadPriority(nn::os::GetCurrentThread()));
    nn::os::StartThread( &thread );

    g_BarrierForIsLockedTest.Await();

    // ここからが Mutex 所有権のテスト
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("MutexType の自スレッド所有チェック\n"));

    {
        nn::os::MutexType   mutex;
        nn::os::InitializeMutex(&mutex, false, 0);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("初期化直後の MutexType の IsMutexLockedByCurrentThread() は false"));
        CheckBool( nn::os::IsMutexLockedByCurrentThread(&mutex) == false );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ロック直後の MutexType の IsMutexLockedByCurrentThread() は true"));
        nn::os::LockMutex(&mutex);
        CheckBool( nn::os::IsMutexLockedByCurrentThread(&mutex) == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("アンロック直後の MutexType の IsMutexLockedByCurrentThread() は false"));
        nn::os::UnlockMutex(&mutex);
        CheckBool( nn::os::IsMutexLockedByCurrentThread(&mutex) == false );
    }

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("MutexType の他スレッド所有チェック\n"));

    {
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("他スレッドでロックされた MutexType の IsMutexLockedByCurrentThread() は false"));
        CheckBool( nn::os::IsMutexLockedByCurrentThread(&g_MutexTypeForIsLockedTest) == false );
    }

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("Mutex クラスの自スレッド所有チェック\n"));

    {
        nn::os::Mutex   mutex(false);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("初期化直後の Mutex インスタンスの IsLockedByCurrentThread() は false"));
        CheckBool( mutex.IsLockedByCurrentThread() == false );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("ロック直後の Mutex インスタンスの IsLockedByCurrentThread() は true"));
        mutex.Lock();
        CheckBool( mutex.IsLockedByCurrentThread() == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("アンロック直後の Mutex インスタンスの IsLockedByCurrentThread() は false"));
        mutex.Unlock();
        CheckBool( mutex.IsLockedByCurrentThread() == false );
    }

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("Mutex クラスの他スレッド所有チェック\n"));

    {
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("他スレッドでロックされた Mutex インスタンスの IsLockedByCurrentThread() は false"));
        CheckBool( g_MutexClassForIsLockedTest.IsLockedByCurrentThread() == false );
    }

    g_BarrierForIsLockedTest.Await();

    // 子スレッドを削除
    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

// ロック無し Mutex の速度を計測
#define NNT_OS_MLAUL(m)     do { \
    (m).Lock(); \
    (m).Unlock(); \
    } while( NN_STATIC_CONDITION(0) )
TEST(MutexWithoutLockSpeedTest, testMutexWithoutLockSpeed)
{
    const int loopCount = 10;
    const int unitSize = 50000;
    const int showSize = 1000;

    int64_t sum = 0;
    int64_t max = 0;
    int64_t min = INT64_MAX;
    int64_t leg;
    int64_t start;
    nn::os::Mutex m(false);

    // 時間計測
    for (int l = 0; l < loopCount; ++l)
    {
        start = nn::os::GetSystemTick().GetInt64Value();
        for (int u = 0; u < unitSize; ++u)
        {
            NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m);
            NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m);
            NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m);
            NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m); NNT_OS_MLAUL(m);
        }
        leg = (nn::os::GetSystemTick().GetInt64Value() - start);
        sum += leg;

        // 最悪値と最善値を記録
        if (leg > max)
        {
            max = leg;
        }
        if (leg < min)
        {
            min = leg;
        }
    }

    // 値を補正し、showSize あたりの回数にする
    sum -= max + min;
    sum = sum * showSize / (unitSize * 20 * (loopCount - 2));

    // 結果の表示
    nn::os::Tick tick( sum );
    NNT_OS_LOG("Mutex without Lock : time=%lldus (tick=%lld) per %d calls.\n",
        tick.ToTimeSpan().GetMicroSeconds(), tick.GetInt64Value(), showSize);
}

// アンロック状態における IsMutexLockedByCurrentThread の速度を計測
#define NNT_OS_IMLBCT(m)    ((m).IsLockedByCurrentThread())
TEST(IsMutexLockedByCurrentThreadWithUnlockSpeedTest, testIsMutexLockedByCurrentThreadWithUnlockSpeed)
{
    const int loopCount = 10000;

    double sum = 0;
    double max = 0;
    double min = 10000;
    double leg;
    int64_t start;
    nn::os::Mutex m(false);

    // 時間計測
    for (int l = 0; l < loopCount; l++)
    {
        start = nn::os::GetSystemTick().GetInt64Value();

        NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m);
        NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m);
        NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m);
        NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m); NNT_OS_IMLBCT(m);

        leg = (nn::os::GetSystemTick().GetInt64Value() - start) / 20.0;
        sum += leg;

        // 最悪値と最善値を記録
        if (leg > max)
        {
            max = leg;
        }
        if (leg < min)
        {
            min = leg;
        }
    }

    // 値を補正し、1000回あたりの回数にする
    sum -= max + min;
    sum = sum * 1000 / (loopCount - 2);

    // 結果の表示
    nn::os::Tick tick(static_cast<int64_t>(sum));
    NNT_OS_LOG("IsMutexLockedByCurrentThread with Unlock : time=%lldus (tick=%lld) per %d calls.\n",
        tick.ToTimeSpan().GetMicroSeconds(), tick.GetInt64Value(), 1000);
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

//---------------------------------------------------------------------------
//  SIGLO-11040 のテスト
//
//  非キャッシュ領域に同期オブジェクトを配置し、チェックに引っ掛かるかを確認します。
//  Debug か Develop ビルドでのみチェックは行われます。
//
#if 0
NN_ALIGNAS(4096) char g_DataForPuttingSyncObject[4096];

TEST(PutSyncObjectOnUncachedAreaTest, test_PutSyncObjectOnUncachedArea)
{
    // Unchaced 領域に変更する
    nn::os::SetMemoryAttribute(
        reinterpret_cast<uintptr_t>(g_DataForPuttingSyncObject),
        sizeof(g_DataForPuttingSyncObject),
        nn::os::MemoryAttribute_Uncached );

    // placement new で上記領域に同期オブジェクトを設置
    nn::os::Mutex* pMutex = new( g_DataForPuttingSyncObject ) nn::os::Mutex( false );

    NN_UNUSED( pMutex );
}
#endif

#endif  // defined(NN_BUILD_CONFIG_OS_HORIZON)

}}} // namespace nnt::os::mutex

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of Mutex APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of Mutex APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
