﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  MultipleWait 専用のテストヘルパー
//---------------------------------------------------------------------------

#pragma once

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nnt/nntest.h>
#endif

//---------------------------------------------------------------------------

namespace nnt { namespace os { namespace multiWait {

const int   priThread1  = 24;
const int   priThread2  = 24;

//---------------------------------------------------------------------------
//  単体テストフレームワーク
//---------------------------------------------------------------------------
//【機能概要】
//  ２つのスレッドと、１つのセマフォ、イベント、メッセージキューを使用し、
//  それぞれ指定された前提条件となるリソース状態を整えてから、
//  指定された API を発行し、その返値および、スレッドや多重待ちオブジェクト
//  の事後状態を期待値と比較することで、様々な条件下の単体テストを実施する。
//
//  指定した前提条件となるオブジェクト状態への遷移は公開 API のみで行ない、
//  オブジェクトの中身を直接書き換えたりはしない。ただし、事後状態を
//  参照する場合には、オブジェクトの中身を直接参照する場合もある。
//
//  全てのオブジェクトは、１回のテスト毎に Initialize/Finalize が行なわれる。
//  スレッド１が必ず最初に動作を開始し、スレッド２が続いて動作を開始する。
//
//  Semaphore     の初期状態は、カウンタ 0、最大カウント 1。
//  Event         の初期状態は、非シグナル状態、自動クリア。
//  MessageQueue1 の初期状態は、バッファ数 4、空っぽ状態。
//  MessageQueue2 の初期状態は、バッファ数 4、満杯状態。
//
//【使用方法】
//  以下の関数１回につき、１つの機能テストを実施する。
//  １回の関数実行により、シーケンス番号が +100 ずつ進む。
//
//    1) １スレッド用
//        void doTest1(th1, rslt1, th1ex);
//
//    2) ２スレッド用
//        void doTest2(th1, th2, rslt1, rslt2, th1ex, th2ex);
//
//  各引数は以下のような意味を持つ。
//
//      th1:    スレッド１のアクション
//      th2:    スレッド２のアクション
//      rslt1:  スレッド１でのテスト対象 API の返値（期待値）
//      rslt2:  スレッド２でのテスト対象 API の返値（期待値）
//      th1ex:  スレッド１の事後状態（期待値）
//      th2ex:  スレッド２の事後状態（期待値）
//
//  スレッド状態を示す th1/th2/th1ex/th2ex には以下の状態を指定する。
//
//      THR_NOTUSE                  そのスレッドは使用しない
//
//      THR_WAIT_ANY                WaitAny() の発行
//      THR_WAIT_TIMED_ANY          TimedWaitAny() の発行
//      THR_WAIT_TIMED_ANY_MINUS    TimedWaitAny() の発行（負のタイムアウト値）
//
//      THR_CALL_ANY                WaitAny() の発行
//      THR_CALL_TRY_ANY            TryWaitAny() の発行
//      THR_CALL_TIMED_ANY          TimedWaitAny() の発行
//      THR_CALL_TIMED_ANY_MINUS    TimedWaitAny() の発行（負のタイムアウト値）
//
//      THR_CALL_REL_SEM            ReleaseSemaphore() の発行
//      THR_CALL_SIG_EVENT          SignalEvent() の発行
//
//      THR_CALL_SEND_MSGQUE        SendMessageQueue() の発行
//      THR_CALL_TRY_SEND_MSGQUE    TrySendMessageQueue() の発行
//      THR_CALL_TIMED_SEND_MSGQUE  TimedSendMessageQueue() の発行
//
//      THR_CALL_JAM_MSGQUE         JamMessageQueue() の発行
//      THR_CALL_TRY_JAM_MSGQUE     TryJamMessageQueue() の発行
//      THR_CALL_TIMED_JAM_MSGQUE   TimedJamMessageQueue() の発行
//
//      THR_CALL_RECV_MSGQUE        ReceiveMessageQueue() の発行
//      THR_CALL_TRY_RECV_MSGQUE    TryReceiveMessageQueue() の発行
//      THR_CALL_TIMED_RECV_MSGQUE  TimedReceiveMessageQueue() の発行
//
//      THR_CALL_SIGNAL_ALL         Sem/Evt/MsgQueSnd/MsgQueRcv 全てシグナル化
//
//  スレッドの最終状態を示す th1ex/th2ex には以下の状態を指定する。
//
//      THR_STATE_NONE              スレッド未使用
//      THR_STATE_EXITED            終了状態
//      THR_STATE_WAITING           待ち状態を継続中
//      THR_STATE_TIMEDOUT_EXITED   タイムアウトを検知して終了
//
//  rslt1/rslt2 には bool を指定する。
//---------------------------------------------------------------------------

enum TestThreadAction
{
    THR_NOTUSE          = 0,    // 初期化のみ（スレッドを使用しない）

    THR_WAIT_ANY,               // WaitAny() の発行
    THR_WAIT_TIMED_ANY,         // TimedWaitAny() の発行
    THR_WAIT_TIMED_ANY_MINUS,   // TimedWaitAny() の発行（負のタイムアウト値）

    THR_CALL_ANY,               // WaitAny() の発行
    THR_CALL_TRY_ANY,           // TryWaitAny() の発行
    THR_CALL_TIMED_ANY,         // TimedWaitAny() の発行
    THR_CALL_TIMED_ANY_MINUS,   // TimedWaitAny() の発行（負のタイムアウト値）

    THR_CALL_REL_SEM,           // ReleaseSemaphore() の発行
    THR_CALL_SIG_EVENT,         // SignalEvent() の発行

    THR_CALL_SEND_MSGQUE,       // SendMessageQueue() の発行
    THR_CALL_TRY_SEND_MSGQUE,   // TrySendMessageQueue() の発行
    THR_CALL_TIMED_SEND_MSGQUE, // TimedSendMessageQueue() の発行

    THR_CALL_JAM_MSGQUE,        // JamMessageQueue() の発行
    THR_CALL_TRY_JAM_MSGQUE,    // TryJamMessageQueue() の発行
    THR_CALL_TIMED_JAM_MSGQUE,  // TimedJamMessageQueue() の発行

    THR_CALL_RECV_MSGQUE,       // ReceiveMessageQueue() の発行
    THR_CALL_TRY_RECV_MSGQUE,   // TryReceiveMessageQueue() の発行
    THR_CALL_TIMED_RECV_MSGQUE, // TimedReceiveMessageQueue() の発行

    THR_CALL_SIGNAL_ALL,        // Sem/Evt/MsgQueSnd/MsgQueRcv 全てシグナル化
};

enum TestThreadLastState
{
    THR_STATE_NONE      = 0,    // スレッド未使用
    THR_STATE_WAITING   = 1,    // 待ち状態を継続中
    THR_STATE_EXITED    = 2,    // 終了状態
    THR_STATE_TIMEDOUT  = 3,    // タイムアウトを検知して終了
    THR_STATE_MAX,
};

enum TestSyncPoint
{
    NNT_POINT_THREAD1_START     = 1,
    NNT_POINT_THREAD2_START,
    NNT_POINT_THREAD1_CALLED,
    NNT_POINT_THREAD2_CALLED,
    NNT_POINT_THREAD1_END,
    NNT_POINT_THREAD2_END,
};

//---------------------------------------------------------------------------

class   MultipleWaitHelper
{
public:
    MultipleWaitHelper()    {}
    ~MultipleWaitHelper()   {}

    // テスト実行
    void    doTest( TestThreadAction    th1,
                    TestThreadAction    th2,
                    bool                rslt1,
                    bool                rslt2,
                    TestThreadLastState th1ex,
                    TestThreadLastState th2ex,
                    const char*         functionName,
                    int                 line);

    // 多重待ちの初期化／破棄
    void    InitializeMultipleWait();
    void    FinalizeMultipleWait();

    // スレッドの実行開始
    void    StartThread1( TestThreadAction );
    void    StartThread2( TestThreadAction );

    // スレッド１情報セット ＆ 取得
    void    SetThreadInfo1( TestThreadAction    action,
                            TestThreadLastState expectState,
                            bool                expect)
    {
        m_TestThreadAction1         = action;
        m_TestThreadExpectState1    = expectState;
        m_expect1                   = expect;
    }
    TestThreadAction    GetThreadAction1()
    {
        return  m_TestThreadAction1;
    }
    TestThreadLastState GetThreadExpectState1()
    {
        return  m_TestThreadExpectState1;
    }
    bool        GetThreadExpect1()
    {
        return  m_expect1;
    }

    // スレッド２情報セット ＆ 取得
    void    SetThreadInfo2( TestThreadAction    action,
                            TestThreadLastState expectState,
                            bool                expect)
    {
        m_TestThreadAction2         = action;
        m_TestThreadExpectState2    = expectState;
        m_expect2                   = expect;
    }
    TestThreadAction    GetThreadAction2()
    {
        return  m_TestThreadAction2;
    }
    TestThreadLastState GetThreadExpectState2()
    {
        return  m_TestThreadExpectState2;
    }
    bool        GetThreadExpect2()
    {
        return  m_expect2;
    }

    // 多重待ちとスレッドをクリーンアップ（待ち解除）
    void    Cleanup();

    // スレッドの状態チェック（Thread の内部実装に依存）
    void    ThreadStateCheck( const nn::os::ThreadType&,
                              TestThreadAction,
                              TestThreadLastState,
                              bool,
                              bool );

    // 多重待ち対象オブジェクトの状態チェック（MultipleWait の内部実装に依存）
    void    MultipleWaitCheck( TestThreadAction th1, TestThreadAction th2);

    // MultipleWait 関連 API を発行
    bool    CallMultipleWaitApi( TestThreadAction       action,
                                 int                    seq,
                                 TestThreadLastState    expectState,
                                 bool                   expect);

    // TesThreadState によるシーケンスカウンタの進み量を返す
    int     GetLastSeq( TestThreadLastState, TestThreadLastState );

    // OS-API 発行後の result の格納と取得
    bool GetApiResult1()
    {
        return m_resultOfApi1;
    }
    bool GetApiResult2()
    {
        return m_resultOfApi2;
    }
    void SetApiResult1(bool result)
    {
        m_resultOfApi1 = result;
    }
    void SetApiResult2(bool result)
    {
        m_resultOfApi2 = result;
    }

    bool IsCleanuped()
    {
        return m_isCleanuped;
    }

private:
    void CreateThread1( TestThreadAction state );
    void CreateThread2( TestThreadAction state );
    void DestroyThread1( TestThreadAction state );
    void DestroyThread2( TestThreadAction state );
    void InitializeMessage();
    void FinalizeMessage();

    nn::os::ThreadType*     m_mainThread;
    nn::os::ThreadType      m_Thread1;
    nn::os::ThreadType      m_Thread2;

    nn::os::SemaphoreType       m_Semaphore;
    nn::os::EventType           m_Event;
    nn::os::MessageQueueType    m_MessageQueue1;
    nn::os::MessageQueueType    m_MessageQueue2;

    TestThreadAction        m_TestThreadAction1;
    TestThreadAction        m_TestThreadAction2;
    TestThreadLastState     m_TestThreadExpectState1;
    TestThreadLastState     m_TestThreadExpectState2;

    bool                    m_expect1;
    bool                    m_expect2;
    bool                    m_resultOfApi1;
    bool                    m_resultOfApi2;
    bool                    m_isCleanuped;
};

//---------------------------------------------------------------------------

extern void ThreadFunc1(void* exinf);
extern void ThreadFunc2(void* exinf);

extern  MultipleWaitHelper  g_MWHelper;

#define doTest1(th1, rslt1, th1ex)         \
        g_MWHelper.doTest((th1), (THR_NOTUSE), (rslt1), (true), (th1ex), (THR_STATE_NONE), __FUNCTION__, __LINE__)

#define doTest2(th1, th2, rslt1, rslt2, th1ex, th2ex)    \
        g_MWHelper.doTest((th1), (th2), (rslt1), (rslt2), (th1ex), (th2ex), __FUNCTION__, __LINE__)

}}} // namespace nnt::os::multiWait

