﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  MultipleWait 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/os/os_SdkMultipleWaitApi.h>
#include "../Common/test_Helper.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <thread>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include "test_MultipleWait-os.win32.h"
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include "test_MultipleWait-os.horizon.h"
    #include <nn/svc/svc_Base.h>
    #include <nn/svc/svc_Server.h>
    #include <nn/svc/svc_Handle.h>
#else
    #error "未サポートの OS 種別が指定されています。"
#endif

//---------------------------------------------------------------------------

namespace nnt { namespace os { namespace multiWait {

extern char g_ThreadStack[ 0x4000 ];

#if defined( NN_BUILD_CONFIG_OS_WIN32 ) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_BDSLIMX6 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SOC_TEGRA_X1 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_JUNO )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_SMMA53 ))
//---------------------------------------------------------------------------
//  Initialize -> Link -> Unlink -> Finalize の確認
//  割込みイベントを含むテストケース。
//
TEST(MultiWaitSequentialTest, test_SequentialTestWithKernelObjectForSystemProgram)
{
    nnt::os::multiWait::InterruptGenerator irqGeneratorForMW;

    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     holder1;
    nn::os::MultiWaitHolderType     holder2;
    nn::os::MultiWaitHolderType     holder3;
    nn::os::MultiWaitHolderType*    signaledHolder = NULL;
    nn::os::EventType               event;
    nn::os::SemaphoreType           semaphore;
    nn::os::InterruptEventType      irqEvent;

    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    // 最初は非シグナル状態のイベントを用意
    nn::os::InitializeEvent(&event, false, nn::os::EventClearMode_ManualClear);

    // 初期カウント 0 のセマフォを用意
    nn::os::InitializeSemaphore(&semaphore, 0, 1);

    // 100msec おきにシグナル化される ManualReset の割込みイベント
    nn::os::InitializeInterruptEvent(&irqEvent, InterruptNameTest2, nn::os::EventClearMode_ManualClear);

    // 多重待ちオブジェクトリストを構築
    nn::os::InitializeMultiWait( &multiWait );

    nn::os::InitializeMultiWaitHolder( &holder1, &event );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder1 );
    nn::os::SetMultiWaitHolderUserData( &holder1, 0x11111111 );

    nn::os::InitializeMultiWaitHolder( &holder2, &semaphore );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder2 );
    nn::os::SetMultiWaitHolderUserData( &holder2, 0x22222222 );

    nn::os::InitializeMultiWaitHolder( &holder3, &irqEvent );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder3 );
    nn::os::SetMultiWaitHolderUserData( &holder3, 0x33333333 );

    // テストその１
    // event, sem, irqEvent のうち、irqEvent で多重待ち解除するケース
    // irqEvent は多重待ちに入る前に一旦クリアされる。
    // WaitAny, TryWaitAny, TimedWaitAny 全てをテスト
    NNT_OS_LOG(NN_TEXT("テストその１： 割込みイベントで待ち解除するケース\n"));
    for (int i = 0; i < 3; ++i)
    {
        SEQ_NONE();

        // WaitAny を発行（irqEvent がシグナル化されるまで待機）
        ClearTimerInterruptStatus();
        nn::os::ClearInterruptEvent( &irqEvent );
        switch (i)
        {
        case 0:
                NNT_OS_LOG("WaitAny( {event, semaphore, irqEvent} )");
                signaledHolder = nn::os::WaitAny( &multiWait );
                break;
        case 1:
                NNT_OS_LOG("TryWaitAny( {event, semaphore, irqEvent} )");
                signaledHolder = nn::os::TryWaitAny( &multiWait );
                break;
        case 2:
                NNT_OS_LOG("TimedWaitAny( {event, semaphore, irqEvent} )");
                signaledHolder = nn::os::TimedWaitAny( &multiWait,
                                        nn::TimeSpan::FromMilliSeconds(500));
                break;
        default:
                signaledHolder = NULL;
                break;
        }

        if (i == 1)
        {
            NNT_OS_LOG(" expect=false");
            CheckBool( signaledHolder == NULL );
        }
        else
        {
            // シグナルが成立している Holder オブジェクトが正しいか？
            NNT_OS_LOG(" expect=true");
            CheckBool( signaledHolder == &holder3 );

            // ユーザデータ値が正しいか？
            auto value = nn::os::GetMultiWaitHolderUserData(signaledHolder);
            SEQ_NONE();
            NNT_OS_LOG("    SignaledHolder's UserData=0x%08x", value);
            CheckBool( value == 0x33333333 );
        }
    }


    // テストその２
    // event, sem, irqEvent のうち、irqEvent で多重待ち解除するケース
    // irqEvent は多重待ちに入る前から既にシグナル状態。
    // WaitAny, TryWaitAny, TimedWaitAny 全てをテスト
    NNT_OS_LOG(NN_TEXT("テストその２： 割込みイベントで待ち解除するケース（事前シグナル化）\n"));
    for (int i = 0; i < 3; ++i)
    {
        SEQ_NONE();

        // WaitAny を発行（irqEvent は既にシグナル状態なので即リターン）
        switch (i)
        {
        case 0:
                NNT_OS_LOG("WaitAny( {event, semaphore, irqEvent} )");
                signaledHolder = nn::os::WaitAny( &multiWait );
                break;
        case 1:
                NNT_OS_LOG("TryWaitAny( {event, semaphore, irqEvent} )");
                signaledHolder = nn::os::TryWaitAny( &multiWait );
                break;
        case 2:
                NNT_OS_LOG("TimedWaitAny( {event, semaphore, irqEvent} )");
                signaledHolder = nn::os::TimedWaitAny( &multiWait,
                                        nn::TimeSpan::FromMilliSeconds(500));
                break;
        default:
                signaledHolder = NULL;
                break;
        }

        // シグナルが成立している Holder オブジェクトが正しいか？
        NNT_OS_LOG(" expect=true");
        CheckBool( signaledHolder == &holder3 );

        // ユーザデータ値が正しいか？
        auto value = nn::os::GetMultiWaitHolderUserData(signaledHolder);
        SEQ_NONE();
        NNT_OS_LOG("    SignaledHolder's UserData=0x%08x", value);
        CheckBool( value == 0x33333333 );
    }

    // テストその３
    // event, sem, irqEvent のうち、semaphore で多重待ち解除するケース
    // semaphore は多重待ちに入ってから 50msec 後ぐらいにシグナル化。
    // WaitAny, TryWaitAny, TimedWaitAny 全てをテスト
    NNT_OS_LOG(NN_TEXT("テストその３： セマフォで待ち解除するケース\n"));
    for (int i = 0; i < 3; ++i)
    {
        nn::os::ThreadType  thread;
        nn::os::CreateThread( &thread, [](void* p)
        {
            auto* sem = reinterpret_cast<nn::os::SemaphoreType*>( p );

            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(50) );
            nn::os::ReleaseSemaphore( sem );
        },
        &semaphore, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);

        SEQ_NONE();

        // WaitAny を発行（semaphore がシグナル状態なのですぐにリターン）
        ClearTimerInterruptStatus();
        nn::os::ClearInterruptEvent( &irqEvent );

        signaledHolder = nn::os::WaitAny( &multiWait );

        ClearTimerInterruptStatus();
        nn::os::ClearInterruptEvent( &irqEvent );

        nn::os::StartThread( &thread );

        // WaitAny を発行（irqEvent がシグナル化されるまで待機）
        ClearTimerInterruptStatus();
        nn::os::ClearInterruptEvent( &irqEvent );
        switch (i)
        {
        case 0:
                NNT_OS_LOG("WaitAny( {event, semaphore, irqEvent} )");
                signaledHolder = nn::os::WaitAny( &multiWait );
                break;
        case 1:
                NNT_OS_LOG("TryWaitAny( {event, semaphore, irqEvent} )");
                signaledHolder = nn::os::TryWaitAny( &multiWait );
                break;
        case 2:
                NNT_OS_LOG("TimedWaitAny( {event, semaphore, irqEvent} )");
                signaledHolder = nn::os::TimedWaitAny( &multiWait,
                                        nn::TimeSpan::FromMilliSeconds(500));
                break;
        default:
                signaledHolder = NULL;
                break;
        }

        if (i == 1)
        {
            NNT_OS_LOG(" expect=false");
            CheckBool( signaledHolder == NULL );
        }
        else
        {
            // シグナルが成立している Holder オブジェクトが正しいか？
            NNT_OS_LOG(" expect=true");
            CheckBool( signaledHolder == &holder2 );

            // ユーザデータ値が正しいか？
            auto value = nn::os::GetMultiWaitHolderUserData(signaledHolder);
            SEQ_NONE();
            NNT_OS_LOG("    SignaledHolder's UserData=0x%08x", value);
            CheckBool( value == 0x22222222 );
        }

        nn::os::AcquireSemaphore( &semaphore );

        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );
    }


    // テストその４
    // event, sem, irqEvent のうち、semaphore で多重待ち解除するケース
    // semaphore は多重待ちに入る前から既にシグナル状態。
    // WaitAny, TryWaitAny, TimedWaitAny 全てをテスト
    NNT_OS_LOG(NN_TEXT("テストその４： セマフォで待ち解除するケース（事前シグナル化）\n"));
    for (int i = 0; i < 3; ++i)
    {
        SEQ_NONE();

        // WaitAny を発行（semaphore がシグナル状態なのですぐにリターン）
        ClearTimerInterruptStatus();
        nn::os::ClearInterruptEvent( &irqEvent );

        signaledHolder = nn::os::WaitAny( &multiWait );

        ClearTimerInterruptStatus();
        nn::os::ClearInterruptEvent( &irqEvent );
        nn::os::ReleaseSemaphore( &semaphore );

        // WaitAny を発行（irqEvent がシグナル化されるまで待機）
        ClearTimerInterruptStatus();
        nn::os::ClearInterruptEvent( &irqEvent );
        switch (i)
        {
        case 0:
                NNT_OS_LOG("WaitAny( {event, semaphore, irqEvent} )");
                signaledHolder = nn::os::WaitAny( &multiWait );
                break;
        case 1:
                NNT_OS_LOG("TryWaitAny( {event, semaphore, irqEvent} )");
                signaledHolder = nn::os::TryWaitAny( &multiWait );
                break;
        case 2:
                NNT_OS_LOG("TimedWaitAny( {event, semaphore, irqEvent} )");
                signaledHolder = nn::os::TimedWaitAny( &multiWait,
                                        nn::TimeSpan::FromMilliSeconds(500));
                break;
        default:
                signaledHolder = NULL;
                break;
        }

        // シグナルが成立している Holder オブジェクトが正しいか？
        NNT_OS_LOG(" expect=true");
        CheckBool( signaledHolder == &holder2 );

        // ユーザデータ値が正しいか？
        auto value = nn::os::GetMultiWaitHolderUserData(signaledHolder);
        SEQ_NONE();
        NNT_OS_LOG("    SignaledHolder's UserData=0x%08x", value);
        CheckBool( value == 0x22222222 );

        nn::os::AcquireSemaphore( &semaphore );
    }


    // テストその５
    // event, sem, irqEvent の多重待ちでタイムアウトするケース
    NNT_OS_LOG(NN_TEXT("テストその５： 多重待ちでタイムアウトするケース\n"));
    {
        SEQ_NONE();
        NNT_OS_LOG("TimedWaitAny( {event, semaphore, irqEvent} )");

        // TimedWaitAny を発行
        ClearTimerInterruptStatus();
        nn::os::ClearInterruptEvent( &irqEvent );

        signaledHolder = nn::os::WaitAny( &multiWait );

        ClearTimerInterruptStatus();
        nn::os::ClearInterruptEvent( &irqEvent );

        signaledHolder = nn::os::TimedWaitAny( &multiWait, nn::TimeSpan::FromMilliSeconds(20));

        // シグナルが成立している Holder オブジェクトが正しいか？
        NNT_OS_LOG(" expect=false");
        CheckBool( signaledHolder == NULL );
    }

    // リストから外す
    nn::os::UnlinkMultiWaitHolder( &holder1 );
    nn::os::UnlinkMultiWaitHolder( &holder2 );
    nn::os::UnlinkMultiWaitHolder( &holder3 );

    // ファイナライズ
    nn::os::FinalizeMultiWaitHolder( &holder1 );
    nn::os::FinalizeMultiWaitHolder( &holder2 );
    nn::os::FinalizeMultiWaitHolder( &holder3 );
    nn::os::FinalizeMultiWait( &multiWait );

    nn::os::FinalizeEvent( &event );
    nn::os::FinalizeSemaphore( &semaphore );
    nn::os::FinalizeInterruptEvent( &irqEvent );

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}   // NOLINT(readability/fn_size)
#endif

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
//---------------------------------------------------------------------------
//  多重待ちでセッションを待機するテスト
//  とりわけ、本テストでは、多重待ち中のセッションをクローズした際に、
//  OS ライブラリの中で ABORT しないことをテストする。
//
void SubThreadForMultiWaitWithSessionClose(void *arg)
{
    nn::svc::Handle handle = nn::svc::Handle(reinterpret_cast<uintptr_t>(arg));

    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );

    // クライアントセッションをクローズ
    auto result = nn::svc::CloseHandle( handle );
    EXPECT_TRUE( result.IsSuccess() );
}

TEST(MultiWaitWithSessionClose, test_MultiWaitWithSessionClose)
{
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("多重待ち中にクライアントセッションをクローズするテスト\n"));

    // セッションの作成
    nn::svc::Handle     serverSession;
    nn::svc::Handle     clientSession;
    auto result = nn::svc::CreateSession(&serverSession, &clientSession, false, 0);
    EXPECT_TRUE( result.IsSuccess() );

    // スレッドの作成
    nn::os::ThreadType  thread;
    nn::os::CreateThread( &thread, SubThreadForMultiWaitWithSessionClose, reinterpret_cast<void*>(static_cast<nnHandle>(clientSession).value), g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);

    // 多重待ちの準備
    nn::os::MultiWaitType   multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    nn::os::MultiWaitHolderType holder;
    nn::os::InitializeMultiWaitHolder(&holder, nn::os::NativeHandle(static_cast<nnHandle>(serverSession).value));
    nn::os::LinkMultiWaitHolder(&multiWait, &holder);

    // 多重待ち後、クライアントセッションをクローズ
    nn::os::StartThread( &thread );
    auto* signaledHolder = nn::os::WaitAny( &multiWait );
    EXPECT_TRUE( signaledHolder == &holder );

    // テスト終了
    nn::os::UnlinkMultiWaitHolder( &holder );
    nn::os::FinalizeMultiWaitHolder( &holder );
    nn::os::FinalizeMultiWait( &multiWait );

    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    nn::svc::CloseHandle( serverSession );

}   // NOLINT(readability/fn_size)
#endif

#if !defined(NN_BUILD_CONFIG_TOOLCHAIN_GCC)
TEST(MultiWaitUtility, test_WaitAnyVariation2)
{
    const nn::TimeSpan timeout = 0;
    {
        nn::os::SystemEvent w(nn::os::EventClearMode_ManualClear, true);
        EXPECT_TRUE(nn::os::TryWaitAny(w.GetBase()) == -1);
        EXPECT_TRUE(nn::os::TimedWaitAny(timeout, w.GetBase()) == -1);
        std::thread t([&]
        {
            w.Signal();
        });
        EXPECT_TRUE(nn::os::WaitAny(w.GetBase()) == 0);
        t.join();
        EXPECT_TRUE(nn::os::TryWaitAny(w.GetBase()) == 0);
    }
}
#endif // !defined(NN_BUILD_CONFIG_TOOLCHAIN_GCC)

}}} // namespace nnt::os::multiWait

