﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/nn_Windows.h>
#include "../Common/test_Helper.h"
#include "../Common/test_NamedPipe.h"

#include <winbase.h>

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
    #include "test_InterruptEvent-os.win32.h"
#else
    #error "未サポートの OS 種別が指定されています。"
#endif

namespace nnt { namespace os { namespace interruptEvent {

char InterruptNameTest1[32];
char InterruptNameTest2[32];
char InterruptNameTest3[32];

//---------------------------------------------------------------------------
//  割込みイベント用の Win32 プロセスの準備
//  100msec 毎にタイマーキューを起床し、必要に応じて各イベントをセットする
//
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//  周期タイマーのハンドラ（メンバ関数）
//
void InterruptGeneratorForWin32::TimerInterruptHandlerImpl()
NN_NOEXCEPT
{
    // For AutoReset
    ::SetEvent( m_Event1 );

    // For ManualReset
    ::SetEvent( m_Event2 );
}

//---------------------------------------------------------------------------
//  周期タイマーのハンドラ（Windows へのエクスポート用）
//
VOID WINAPI TimerInterruptHandler(PVOID param, BOOLEAN fired)
{
    NN_UNUSED(fired);

    auto* p = reinterpret_cast<InterruptGeneratorForWin32*>(param);
    p->TimerInterruptHandlerImpl();
}

//---------------------------------------------------------------------------
//  コンストラクタ
//
//  以下の前処理を行ないます。
//  - Win32 割込み通知用イベントの作成
//  - Win32 タイマーキューの作成
//  - Win32 タイマーキュータイマーの作成（周期タイマーが SetEvent する）
//
InterruptGeneratorForWin32::InterruptGeneratorForWin32()
NN_NOEXCEPT
{
    // Win32 割込み通知用イベントのイベント名を自動生成
    nnt::os::detail::CreateUniqueInterruptName( InterruptNameTest1, sizeof(InterruptNameTest1) );
    nnt::os::detail::CreateUniqueInterruptName( InterruptNameTest2, sizeof(InterruptNameTest2) );
    nnt::os::detail::CreateUniqueInterruptName( InterruptNameTest3, sizeof(InterruptNameTest3) );

    // Win32 割込み通知用イベントの作成
    //  （AutoReset 版でも Win32 Event は Manual にしておく必要がある。）
    m_Event1 = ::CreateEvent(NULL,
                             TRUE  /* ManualReset */,
                             FALSE /* Initially Not Signaled */,
                             reinterpret_cast<LPCTSTR>( InterruptNameTest1 ));
    NN_ASSERT(m_Event1 != NULL,
        NN_TEXT("Win32 Event1 の生成に失敗しました。"));


    // Win32 割込み通知用イベントの作成（ManualRest 版）
    m_Event2 = ::CreateEvent(NULL,
                             TRUE  /* ManualReset */,
                             FALSE /* Initially Not Signaled */,
                             reinterpret_cast<LPCTSTR>( InterruptNameTest2 ));
    NN_ASSERT(m_Event2 != NULL,
        NN_TEXT("Win32 Event2 の生成に失敗しました。"));


    // Win32 割込み通知用イベントの作成（非シグナル状態のままを維持）
    m_Event3 = ::CreateEvent(NULL,
                             TRUE  /* ManualReset */,
                             FALSE /* Initially Not Signaled */,
                             reinterpret_cast<LPCTSTR>( InterruptNameTest3 ));
    NN_ASSERT(m_Event3 != NULL,
        NN_TEXT("Win32 Event3 の生成に失敗しました。"));


    // Win32 タイマーキューの作成
    m_TimerQueue = CreateTimerQueue();
    NN_ASSERT(m_TimerQueue != NULL,
        NN_TEXT("Win32 TimerQueue の生成に失敗しました。"));


    // Win32 タイマーキュータイマーの作成
    BOOL ret = CreateTimerQueueTimer( &m_TimerQueueTimer,
                                      m_TimerQueue,
                                      &TimerInterruptHandler,
                                      this,
                                      100 /* msec */,
                                      100 /* msec */,
                                      0 );
    NN_ASSERT( ret != NULL,
        NN_TEXT("Win32 TimerQueueTimer の生成に失敗しました。"));
}

//---------------------------------------------------------------------------
//  デストラクタ
//
//  - Win32 タイマーキュータイマーの削除
//  - Win32 タイマーキューの削除
//  - Win32 割込み通知用イベントの削除
//
InterruptGeneratorForWin32::~InterruptGeneratorForWin32()
NN_NOEXCEPT
{
    // Win32 タイマーキュータイマーの削除
    ::DeleteTimerQueueTimer( m_TimerQueue, m_TimerQueueTimer, NULL );

    // Win32 タイマーキューの削除
    ::DeleteTimerQueueEx( m_TimerQueue, NULL );

    // Win32 割込み通知用イベントの削除
    ::CloseHandle( m_Event1 );
    ::CloseHandle( m_Event2 );
    ::CloseHandle( m_Event3 );
}


}}} // namespace nnt::os::interruptEvent

