﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Event 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/nn_Common.h>
#include <nn/nn_SdkText.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "test_EventHelper.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

#define EVENT_AUTO_CLEAR    nn::os::EventClearMode_AutoClear
#define EVENT_MANUAL_CLEAR  nn::os::EventClearMode_ManualClear

namespace nnt { namespace os { namespace event {

//---------------------------------------------------------------------------
// InitializeEvent 直後の初期状態チェック

TEST(InitializeEvent, test_initiallySignaled1)
{
    // InitializeEvent 初期状態チェック
    doTest1(false,                      // Event は Not Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_NONE);            // Thread1 は終了
}

TEST(InitializeSendEvent, test_initiallySignaled2)
{
    // InitializeEvent 初期状態チェック
    doTest1(false,                      // Event は Not Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_NONE);            // Thread1 は終了
}

TEST(InitializeSendEvent, test_initiallyNotSignaled1)
{
    // InitializeEvent 初期状態チェック
    doTest1(true,                       // Event は Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            true,                       // Event は Signaled
            THR_STATE_NONE);            // Thread1 は終了
}

TEST(InitializeSendEvent, test_initiallyNotSignaled2)
{
    // InitializeEvent 初期状態チェック
    doTest1(true,                       // Event は Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            true,                       // Event は Signaled
            THR_STATE_NONE);            // Thread1 は終了
}

//---------------------------------------------------------------------------
// 単独の SignalEvent() テスト

TEST(SignalEvent, test_NotSignalToSignal)
{
    // SignalEvent 非シグナルからシグナルへ
    doTest1(false,                      // Event は Not Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_CALL_SIGNAL,            // Thread1 は SignalEvent()
            true,                       // Thread1 での API 返値
            true,                       // Event は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(SignalEvent, test_SignalToSignal)
{
    // SignalEvent シグナルからシグナルへ
    doTest1(true,                       // Event は Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_CALL_SIGNAL,            // Thread1 は SignalEvent()
            true,                       // Thread1 での API 返値
            true,                       // Event は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

//---------------------------------------------------------------------------

// 単独の ClearEvent() テスト

TEST(ClearEvent, test_NotSignalToNotSignal1)
{
    // SignalEvent 非シグナルからシグナルへ
    doTest1(false,                      // Event は Not Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_CALL_CLEAR,             // Thread1 は ClearEvent()
            true,                       // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(ClearEvent, test_NotSignalToNotSignal2)
{
    // SignalEvent 非シグナルからシグナルへ
    doTest1(false,                      // Event は Not Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_CALL_CLEAR,             // Thread1 は ClearEvent()
            true,                       // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(ClearEvent, test_SignalToNotSignal1)
{
    // SignalEvent シグナルからシグナルへ
    doTest1(true,                       // Event は Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_CALL_CLEAR,             // Thread1 は ClearEvent()
            true,                       // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(ClearEvent, test_SignalToNotSignal2)
{
    // SignalEvent シグナルからシグナルへ
    doTest1(true,                       // Event は Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_CALL_CLEAR,             // Thread1 は ClearEvent()
            true,                       // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

//---------------------------------------------------------------------------
// 単独の WaitEvent() AutoClear テスト

TEST(WaitEventAutoClear, test_WaitEventAutoClear1)
{
    // AUTO_CLEAR/Signaled のイベントに対して WaitEvent 発行
    doTest1(true,                       // Event は Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_CALL_WAIT,              // Thread1 は WaitEvent()
            true,                       // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(WaitEventAutoClear, test_WaitEventAutoClear2)
{
    // AUTO_CLEAR/NotSignaled のイベントに対して WaitEvent 発行
    doTest1(false,                      // Event は Not Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitEvent()
            true,                       // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_WAITING);         // Thread1 はシグナル待ち
}

TEST(WaitEventAutoClear, test_WaitEventAutoClear3)
{
    // AUTO_CLEAR/Signaled のイベントに対して WaitEvent 発行×２スレッド
    doTest2(true,                       // Event は Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_CALL_WAIT,              // Thread1 は WaitEvent()
            THR_WAIT_WAIT,              // Thread1 は WaitEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_WAITING);             // Thread2 はシグナル待ち
}

TEST(WaitEventAutoClear, test_WaitEventAutoClear4)
{
    // AUTO_CLEAR/NotSignaled のイベントに対して WaitEvent 発行×２スレッド
    doTest2(false,                      // Event は Not Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitEvent()
            THR_WAIT_WAIT,              // Thread1 は WaitEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_WAITING,          // Thread1 はシグナル待ち
            THR_STATE_WAITING);         // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の WaitEvent() ManualClear テスト

TEST(WaitEventManualClear, test_WaitEventManualClear1)
{
    // MANUAL_CLEAR/Signaled のイベントに対して WaitEvent 発行
    doTest1(true,                       // Event は Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_CALL_WAIT,              // Thread1 は WaitEvent()
            true,                       // Thread1 での API 返値
            true,                       // Event は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(WaitEventManualClear, test_WaitEventManualClear2)
{
    // MANUAL_CLEAR/NotSignaled のイベントに対して WaitEvent 発行
    doTest1(false,                      // Event は Not Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitEvent()
            true,                       // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_WAITING);         // Thread1 はシグナル待ち
}

TEST(WaitEventManualClear, test_WaitEventManualClear3)
{
    // MANUAL_CLEAR/Signaled のイベントに対して WaitEvent 発行×２スレッド
    doTest2(true,                       // Event は Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_CALL_WAIT,              // Thread1 は WaitEvent()
            THR_CALL_WAIT,              // Thread1 は WaitEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // Event は Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(WaitEventManualClear, test_WaitEventManualClear4)
{
    // MANUAL_CLEAR/NotSignaled のイベントに対して WaitEvent 発行×２スレッド
    doTest2(false,                      // Event は Not Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitEvent()
            THR_WAIT_WAIT,              // Thread1 は WaitEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_WAITING,          // Thread1 はシグナル待ち
            THR_STATE_WAITING);         // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の TryWaitEvent() AutoClear テスト

TEST(TryWaitEventAutoClear, test_TryWaitEventAutoClear1)
{
    // AUTO_CLEAR/Signaled のイベントに対して TryWaitEvent 発行
    doTest1(true,                       // Event は Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitEvent()
            true,                       // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryWaitEventAutoClear, test_TryWaitEventAutoClear2)
{
    // AUTO_CLEAR/NotSignaled のイベントに対して TryWaitEvent 発行
    doTest1(false,                      // Event は Not Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitEvent()
            false,                      // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED);          // Thread1 はシグナル待ち
}

TEST(TryWaitEventAutoClear, test_TryWaitEventAutoClear3)
{
    // AUTO_CLEAR/Signaled のイベントに対して TryWaitEvent 発行×２スレッド
    doTest2(true,                       // Event は Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitEvent()
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitEvent()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 はシグナル待ち
}

TEST(TryWaitEventAutoClear, test_TryWaitEventAutoClear4)
{
    // AUTO_CLEAR/NotSignaled のイベントに対して TryWaitEvent 発行×２スレッド
    doTest2(false,                      // Event は Not Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitEvent()
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitEvent()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED,           // Thread1 はシグナル待ち
            THR_STATE_EXITED);          // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の TryWaitEvent() ManualClear テスト

TEST(TryWaitEventManualClear, test_TryWaitEventManualClear1)
{
    // MANUAL_CLEAR/Signaled のイベントに対して TryWaitEvent 発行
    doTest1(true,                       // Event は Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitEvent()
            true,                       // Thread1 での API 返値
            true,                       // Event は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryWaitEventManualClear, test_TryWaitEventManualClear2)
{
    // MANUAL_CLEAR/NotSignaled のイベントに対して TryWaitEvent 発行
    doTest1(false,                      // Event は Not Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitEvent()
            false,                      // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED);          // Thread1 はシグナル待ち
}

TEST(TryWaitEventManualClear, test_TryWaitEventManualClear3)
{
    // MANUAL_CLEAR/Signaled のイベントに対して TryWaitEvent 発行×２スレッド
    doTest2(true,                       // Event は Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitEvent()
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // Event は Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(TryWaitEventManualClear, test_TryWaitEventManualClear4)
{
    // MANUAL_CLEAR/NotSignaled のイベントに対して TryWaitEvent 発行×２スレッド
    doTest2(false,                      // Event は Not Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitEvent()
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitEvent()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED,           // Thread1 はシグナル待ち
            THR_STATE_EXITED);          // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の TimedWaitEvent() AutoClear テスト

TEST(TimedWaitEventAutoClear, test_TimedWaitEventAutoClear1)
{
    // AUTO_CLEAR/Signaled のイベントに対して TimedWaitEvent 発行
    doTest1(true,                       // Event は Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_CALL_TIMED_WAIT,        // Thread1 は TimedWaitEvent()
            true,                       // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedWaitEventAutoClear, test_TimedWaitEventAutoClear2)
{
    // AUTO_CLEAR/NotSignaled のイベントに対して TimedWaitEvent 発行
    doTest1(false,                      // Event は Not Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitEvent()
            false,                      // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

TEST(TimedWaitEventAutoClear, test_TimedWaitEventAutoClear3)
{
    // AUTO_CLEAR/Signaled のイベントに対して TimedWaitEvent 発行×２スレッド
    doTest2(true,                       // Event は Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_CALL_TIMED_WAIT,        // Thread1 は TimedWaitEvent()
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitEvent()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}

TEST(TimedWaitEventAutoClear, test_TimedWaitEventAutoClear4)
{
    // AUTO_CLEAR/NotSignaled のイベントに対して TimedWaitEvent 発行×２スレッド
    doTest2(false,                      // Event は Not Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitEvent()
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitEvent()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_TIMEDOUT,         // Thread1 はタイムアウト
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}

//---------------------------------------------------------------------------
// 単独の TimedWaitEvent() ManualClear テスト

TEST(TimedWaitEventManualClear, test_TimedWaitEventManualClear1)
{
    // MANUAL_CLEAR/Signaled のイベントに対して TimedWaitEvent 発行
    doTest1(true,                       // Event は Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_CALL_TIMED_WAIT,        // Thread1 は TimedWaitEvent()
            true,                       // Thread1 での API 返値
            true,                       // Event は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedWaitEventManualClear, test_TimedWaitEventManualClear2)
{
    // MANUAL_CLEAR/NotSignaled のイベントに対して TimedWaitEvent 発行
    doTest1(false,                      // Event は Not Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitEvent()
            false,                      // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

TEST(TimedWaitEventManualClear, test_TimedWaitEventManualClear3)
{
    // MANUAL_CLEAR/Signaled のイベントに対して TimedWaitEvent 発行×２スレッド
    doTest2(true,                       // Event は Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_CALL_TIMED_WAIT,        // Thread1 は TimedWaitEvent()
            THR_CALL_TIMED_WAIT,        // Thread1 は TimedWaitEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // Event は Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(TimedWaitEventManualClear, test_TimedWaitEventManualClear4)
{
    // MANUAL_CLEAR/NotSignaled のイベントに対して TimedWaitEvent 発行×２スレッド
    doTest2(false,                      // Event は Not Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitEvent()
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitEvent()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_TIMEDOUT,         // Thread1 はタイムアウト
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}


// 負のタイムアウト値を許容することになったときのためにテストを残しておく
#if 0
//---------------------------------------------------------------------------
// 単独の TimedWaitEvent() テスト（負のタイムアウト値）

TEST(TimedWaitEventWithMinusTimeout, test_TimedWaitEventWithMinusTimeout1)
{
    // AUTO_CLEAR/Signaled のイベントに対して TimedWaitEvent 発行
    doTest1(true,                       // Event は Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_CALL_TIMED_WAIT_MINUS,  // Thread1 は負のタイムアウト値で TimedWaitEvent()
            true,                       // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedWaitEventWithMinusTimeout, test_TimedWaitEventWithMinusTimeout2)
{
    // AUTO_CLEAR/NotSignaled のイベントに対して TimedWaitEvent 発行
    doTest1(false,                      // Event は Not Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_WAIT_TIMED_WAIT_MINUS,  // Thread1 は負のタイムアウト値で TimedWaitEvent()
            false,                      // Thread1 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

#else
// 負のタイムアウト値を許容しない場合の DEATH TEST

TEST(TimedWaitEventWithMinusTimeoutDeathTest, test_TimedWaitEventWithMinusTimeoutDeathTest)
{
    // AUTO_CLEAR/Signaled のイベントに対して TimedWaitEvent 発行
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(true,                       // Event は Signaled
                EVENT_AUTO_CLEAR,           // Event は自動クリアモード
                THR_CALL_TIMED_WAIT_MINUS,  // Thread1 は負のタイムアウト値で TimedWaitEvent()
                true,                       // Thread1 での API 返値
                false,                      // Event は Not Signaled
                THR_STATE_EXITED)           // Thread1 は終了
    , "");
}

#endif


//---------------------------------------------------------------------------
// AUTO_CLEAR/Wait 待機中に Signal する（TimedWait 含む）

TEST(CombinedEventTestAutoClear, test_WaitAndSignalAutoClear)
{
    // Thread1 で WaitEvent()、Thread2 で SignalEvent()
    doTest2(false,                      // Event は Not Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitEvent()
            THR_CALL_SIGNAL,            // Thread2 は SignalEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedEventTestAutoClear, test_TimdeWaitAndSignalAutoClear)
{
    // Thread1 で WaitEvent()、Thread2 で SignalEvent()
    doTest2(false,                      // Event は Not Signaled
            EVENT_AUTO_CLEAR,           // Event は自動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitEvent()
            THR_CALL_SIGNAL,            // Thread2 は SignalEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // Event は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

//---------------------------------------------------------------------------
// MANUAL_CLEAR/Wait 待機中に Signal する（TimedWait 含む）

TEST(CombinedEventTestManualClear, test_WaitAndSignalManualClear)
{
    // Thread1 で WaitEvent()、Thread2 で SignalEvent()
    doTest2(false,                      // Event は Not Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitEvent()
            THR_CALL_SIGNAL,            // Thread2 は SignalEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // Event は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedEventTestManualClear, test_TimdeWaitAndSignalManualClear)
{
    // Thread1 で WaitEvent()、Thread2 で SignalEvent()
    doTest2(false,                      // Event は Not Signaled
            EVENT_MANUAL_CLEAR,         // Event は手動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitEvent()
            THR_CALL_SIGNAL,            // Thread2 は SignalEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // Event は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}


//---------------------------------------------------------------------------
//  Event クラスのテストその１
//  ここでは、Event クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  test_EventClassTestAutoClear ではブロックしない AutoClear イベントの
//  テストを実施する。
//
TEST(EventClass, test_EventClassTestAutoClear)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  timeout = nn::TimeSpan::FromMicroSeconds(1);

        SEQ_SET(4000);
        NNT_OS_LOG(NN_TEXT("main: event インスタンスの生成（AutoClear）\n"));
        nn::os::Event event(nn::os::EventClearMode_AutoClear);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.TryWait() が false を返す"));
        CheckBool( event.TryWait() == false );

        // Wait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.Signal() に成功する\n"));
        event.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.Wait() に成功する\n"));
        event.Wait();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.TryWait() が false を返す"));
        CheckBool( event.TryWait() == false );

        // TryWait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.Signal() に成功する\n"));
        event.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.TryWait() が true を返す"));
        CheckBool( event.TryWait() == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.TryWait() が false を返す"));
        CheckBool( event.TryWait() == false );

        // TimedWait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.Signal() に成功する\n"));
        event.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.TimedWait() が true を返す"));
        CheckBool( event.TimedWait(timeout) == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.TimedWait(0) が false を返す"));
        CheckBool( event.TimedWait(nn::TimeSpan(0)) == false );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  Event クラスのテストその２
//  ここでは、Event クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  test_EventClassTestManualClear ではブロックしない ManualClear イベントの
//  テストを実施する。
//
TEST(EventClass, test_EventClassTestManualClear)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  timeout = nn::TimeSpan::FromMicroSeconds(1);

        SEQ_SET(4100);
        NNT_OS_LOG(NN_TEXT("main: event インスタンスの生成（ManualClear）\n"));
        nn::os::Event event(nn::os::EventClearMode_ManualClear);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.TryWait() が false を返す"));
        CheckBool( event.TryWait() == false );

        // Wait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.Signal() に成功する\n"));
        event.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.Wait() に成功する\n"));
        event.Wait();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.TryWait() が true を返す"));
        CheckBool( event.TryWait() == true );

        // TryWait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.Signal() に成功する\n"));
        event.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.TryWait() が true を返す"));
        CheckBool( event.TryWait() == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.TryWait() が true を返す"));
        CheckBool( event.TryWait() == true );

        // TimedWait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.Signal() に成功する\n"));
        event.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.TimedWait() が true を返す"));
        CheckBool( event.TimedWait(timeout) == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.TimedWait(0) が true を返す"));
        CheckBool( event.TimedWait(nn::TimeSpan(0)) == true );

        // Clear の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.Clear() に成功する\n"));
        event.Clear();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: event.TryWait() が false を返す"));
        CheckBool( event.TryWait() == false );

    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  Event クラスのテストその３
//  ここでは、Event クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  test_EventClassTestBlock ではブロック系のテストを２スレッド使って実施する。
//
NN_ALIGNAS( 4096 ) char g_StackOfThread[ 8192 ];

void EventClassTestThread(void* arg)
{
    auto&              event    = *static_cast<nn::os::Event*>(arg);
    const nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds(100);

    for (int i = 0; i < 2; ++i)
    {
        nn::os::SleepThread( interval );

        SEQ_CHECK(4202 + i * 3);
        NNT_OS_LOG(NN_TEXT("sub:  event.Signal() でシグナル化\n"));
        event.Signal();
    }
}

TEST(EventClass, test_EventClassTestBlock)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  timeout = nn::TimeSpan::FromSeconds(1);

        SEQ_SET(4200);
        NNT_OS_LOG(NN_TEXT("main: Event インスタンスの生成"));
        nn::os::Event event(nn::os::EventClearMode_AutoClear);

        // テスト用の子スレッドを作成
        nn::os::ThreadType  thread;
        {
            auto pri    = nn::os::GetThreadPriority(nn::os::GetCurrentThread());
            auto result = nn::os::CreateThread( &thread,
                                                &EventClassTestThread,
                                                &event,
                                                g_StackOfThread,
                                                sizeof(g_StackOfThread),
                                                pri + 1);
            CheckBool( result.IsSuccess() );
        }

        // 子スレッドを起動
        nn::os::StartThread( &thread );

        // Step1: Wait で待機して子スレッドから起床
        SEQ_CHECK(4201);
        NNT_OS_LOG(NN_TEXT("main: evnet.Wait() で待機\n"));
        event.Wait();
        SEQ_CHECK(4203);
        NNT_OS_LOG(NN_TEXT("main: -> 起床\n"));

        // Step2: TimedWait で待機し、子スレッドからシグナルしてもらう
        SEQ_CHECK(4204);
        NNT_OS_LOG(NN_TEXT("main: event.TimedWait() で待機\n"));
        event.TimedWait(timeout);
        SEQ_CHECK(4206);
        NNT_OS_LOG(NN_TEXT("main: -> 起床\n"));

        // 子スレッドの終了待ち
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  Event クラスの型変換関数のテスト
//  ここでは、Event クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  ここでは、GetBase() および operator EventType&() の動作テストを行なう。
//
TEST(EventClass, test_EventClassTestTypeExchange)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        SEQ_SET(4300);
        NNT_OS_LOG(NN_TEXT("event インスタンスの生成（AutoClear）\n"));
        nn::os::Event event(nn::os::EventClearMode_AutoClear);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.GetBase() で EventType オブジェクトを取得"));
        nn::os::EventType*  eventType = event.GetBase();
        CheckBool( eventType != NULL );

        // 初期シグナル状態が false であることを確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("event.TryWait() で非シグナル状態であることを確認"));
        CheckBool( event.TryWait() == false );

        // EventType にてセットし、TryWait で true を確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("eventType を使ってシグナル状態にセット"));
        nn::os::SignalEvent(eventType);
        CheckBool( event.TryWait() == true );

        // EventType にてクリアし、TryWait で false を確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("eventType を使って非シグナル状態にクリア"));
        nn::os::ClearEvent(eventType);
        CheckBool( event.TryWait() == false );

        // operator EventType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator EventType&() を使ってオブジェクトの参照を取得"));
        nn::os::EventType& eventTypeRefer = event;
        CheckBool( &eventTypeRefer == eventType );

        // operator const EventType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator const EventType&() を使ってオブジェクトの参照を取得"));
        const nn::os::EventType& eventTypeReferConst = event;
        CheckBool( &eventTypeReferConst == eventType );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


}}} // namespace nnt::os::event

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of Event APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of Event APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
