﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_MemoryAttribute.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_UserExceptionHandler.h>

#include <nnt/nntest.h>
#include "../Common/test_SetjmpWithFpuContext.h"
#include "../Common/test_Helper.h"


namespace nnt { namespace os { namespace debug {

//---------------------------------------------------------------------------
//  メモリの属性なしチェック API のテスト
//---------------------------------------------------------------------------

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
TEST(AddressSpaceNoAttribute, test_AddressSpaceNoAttribute)
{
    uintptr_t address;
    const auto size = nn::os::MemoryPageSize * 2;
    NN_STATIC_ASSERT(size <= nn::os::MemoryHeapUnitSize);

    // メモリヒープを設定
    NNT_OS_LOG("Setup memory heap and memoryblock\n");
    EXPECT_TRUE( nn::os::SetMemoryHeapSize( nn::os::MemoryHeapUnitSize ).IsSuccess() );
    EXPECT_TRUE( nn::os::AllocateMemoryBlock(&address, nn::os::MemoryHeapUnitSize).IsSuccess() );
    NNT_OS_LOG("Memory block address=0x%p\n", address);
    void* pAddress = reinterpret_cast<void*>(address);

    // 上記領域の属性チェック
    EXPECT_TRUE( !nn::os::IsMemoryLocked(address, nn::os::MemoryHeapUnitSize) );
    EXPECT_TRUE( !nn::os::IsMemoryLocked(pAddress, nn::os::MemoryHeapUnitSize) );

    // TransferMemory を作成
    {
        NNT_OS_LOG("Create transfer memory\n");
        // address を uintptr_t のまま渡すと Attach 相当のコンストラクタが
        // 呼ばれてしまうので注意。何か対策考えた方がよいかも。
        nn::os::TransferMemory trans(reinterpret_cast<void*>(address), size, nn::os::MemoryPermission_ReadWrite);
        EXPECT_FALSE( !nn::os::IsMemoryLocked(address, size) );
        EXPECT_FALSE( !nn::os::IsMemoryLocked(pAddress, size) );

        // TransferMemory をマップ
        NNT_OS_LOG("Map transfer memory\n");
        void* transMapAddress;
        NN_ABORT_UNLESS_RESULT_SUCCESS( trans.Map(&transMapAddress, nn::os::MemoryPermission_ReadWrite) );
        EXPECT_FALSE( !nn::os::IsMemoryLocked(address, size) );
        EXPECT_FALSE( !nn::os::IsMemoryLocked(pAddress, size) );
        EXPECT_TRUE( !nn::os::IsMemoryLocked(reinterpret_cast<uintptr_t>(transMapAddress), size) );
        EXPECT_TRUE( !nn::os::IsMemoryLocked(transMapAddress, size) );
        NNT_OS_LOG("map address=0x%p\n", transMapAddress);
//*(volatile char *)0 = 0;

        // TransferMemory をアンマップ
        NNT_OS_LOG("Unmap transfer memory\n");
        trans.Unmap();
        EXPECT_FALSE( !nn::os::IsMemoryLocked(address, size) );
        EXPECT_FALSE( !nn::os::IsMemoryLocked(pAddress, size) );

        // TransferMemory を破棄
        NNT_OS_LOG("Destroy transfer memory\n");
    }
    EXPECT_TRUE( !nn::os::IsMemoryLocked(address, size) );
    EXPECT_TRUE( !nn::os::IsMemoryLocked(pAddress, size) );

    {
        // Uncached に設定（メモリブロックの後半のみ）
        NNT_OS_LOG("Set memory attribute UNCAHCED\n");
        auto halfSize = size / 2;
        nn::os::SetMemoryAttribute(address + halfSize, halfSize, nn::os::MemoryAttribute_Uncached);
        EXPECT_TRUE( !nn::os::IsMemoryLocked(address, halfSize) );
        EXPECT_TRUE( !nn::os::IsMemoryLocked(pAddress, halfSize) );
        EXPECT_FALSE( !nn::os::IsMemoryLocked(address + halfSize, halfSize) );
        EXPECT_FALSE( !nn::os::IsMemoryLocked(reinterpret_cast<void*>(address + halfSize), halfSize) );
        EXPECT_FALSE( !nn::os::IsMemoryLocked(address, size) );
        EXPECT_FALSE( !nn::os::IsMemoryLocked(pAddress, size) );

        // Cached に戻す
        NNT_OS_LOG("Set memory attribute NORMAL\n");
        nn::os::SetMemoryAttribute(address, size, nn::os::MemoryAttribute_Normal);
        EXPECT_TRUE( !nn::os::IsMemoryLocked(address, size) );
        EXPECT_TRUE( !nn::os::IsMemoryLocked(pAddress, size) );
    }

    // ページサイズ等にアライメントされていなくても正しくチェックできるか
    EXPECT_TRUE( !nn::os::IsMemoryLocked(address + 1, 1) );
    EXPECT_TRUE( !nn::os::IsMemoryLocked(reinterpret_cast<void*>(address + 1), 1) );

    // メモリヒープを返却
    nn::os::FreeMemoryBlock(address, nn::os::MemoryHeapUnitSize);
    EXPECT_TRUE( !nn::os::IsMemoryLocked(address, nn::os::MemoryHeapUnitSize) );
    EXPECT_TRUE( !nn::os::IsMemoryLocked(pAddress, nn::os::MemoryHeapUnitSize) );
    EXPECT_TRUE( nn::os::SetMemoryHeapSize( 0 ).IsSuccess() );
}
#endif /* defined(NN_BUILD_CONFIG_OS_HORIZON) */

//---------------------------------------------------------------------------

}}} // namespace nnt::os::debug

