﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nnt/nntest.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #error "Win32 ビルドではコンパイル不要です"
#endif  // !defined(NN_BUILD_CONFIG_OS_WIN32)

#include <nn/svc/svc_ThreadLocalRegion.h>
#include <nn/os/os_SdkMemoryAllocatorForThreadLocal.h>


//-----------------------------------------------------------------------------
// Currently, it's built by PIC mode in both clang and gcc.
#define NNT_OS_BUILD_PIC

//-----------------------------------------------------------------------------

extern char __tdata_start[];
extern char __tdata_end[];
extern char __tbss_start[];
extern char __tbss_end[];

extern "C" void* get_tls_init_image(size_t module_id)
{
    NN_UNUSED(module_id);
    return reinterpret_cast<void*>(__tdata_start);
}

extern "C" size_t get_tls_init_size(size_t module_id)
{
    NN_UNUSED(module_id);
    return reinterpret_cast<uintptr_t>(__tdata_end) - reinterpret_cast<uintptr_t>(__tdata_start);
}

extern "C" size_t get_tls_size(size_t module_id)
{
    NN_UNUSED(module_id);
    return reinterpret_cast<uintptr_t>(__tbss_end) - reinterpret_cast<uintptr_t>(__tdata_start);
}

//-----------------------------------------------------------------------------

extern "C" void *allocate_tls (size_t module_id)
{
    char *mem = reinterpret_cast<char*>( nnosAllocateMemoryForThreadLocal( get_tls_size(module_id) ) );
    NN_ABORT_UNLESS(mem != NULL);

    // initial image of .tdata section
    std::memcpy (mem,
                 get_tls_init_image(module_id),
                 get_tls_init_size(module_id));

    // initial image of .tbss  section
    std::memset (mem + get_tls_init_size(module_id),
                 0,
                 get_tls_size(module_id) - get_tls_init_size(module_id));

    return mem;
}

extern "C" uintptr_t __get_tp()
{
    return nn::svc::GetThreadLocalRegion()->thread_pointer;
}

extern "C" void __set_tp(uintptr_t* tp)
{
    nn::svc::GetThreadLocalRegion()->thread_pointer = reinterpret_cast<uintptr_t>(tp);
#if defined NN_BUILD_CONFIG_CPU_ARM64
    asm volatile(" msr tpidr_el0, %0" :: "r"(tp) : "memory");
#else
    asm volatile(" mcr p15, 0, %0, c13, c0, 2" :: "r"(tp) : "memory");
#endif
}

//-----------------------------------------------------------------------------

#if defined(NN_BUILD_CONFIG_COMPILER_CLANG)
extern "C" int __cxa_thread_atexit(void (*func)(void *), void *obj, void *dso_symbol)
#elif defined(NN_BUILD_CONFIG_COMPILER_GCC)
extern "C" int __cxa_thread_atexit_impl(void (*func)(void *), void *obj, void *dso_symbol)
#endif
{
    NN_UNUSED( func );
    NN_UNUSED( obj );
    NN_UNUSED( dso_symbol );
    return 0;
}

#if defined(NNT_OS_BUILD_NONPIC)
//-----------------------------------------------------------------------------
// non-pic 環境での TLS 変数のアドレス取得関数（ただし r0 以外は破壊禁止）
//
extern "C" void* __aeabi_read_tp(int offset)
{
    NN_UNUSED( offset );
    return NULL;
}
#endif

#if defined(NNT_OS_BUILD_PIC)
//-----------------------------------------------------------------------------
// pic 環境での TLS 変数のアドレス取得関数
// IHI0045D_ABI_addenda.pdf の資料を参照のこと。
//
extern "C" void *__tls_get_addr (size_t* got)
{
    size_t module_id = got[0];
    size_t offset    = got[1];

    NN_SDK_ASSERT( module_id > 0 );

    // Get TCB region
    uintptr_t* tcb = reinterpret_cast<uintptr_t*>( __get_tp() );
    if (tcb == NULL)
    {
        // Lazy allocation
        tcb = reinterpret_cast<uintptr_t*>( nnosAllocateMemoryForThreadLocal( 16 ) );    // size is provisional
        NN_ABORT_UNLESS(tcb != NULL);
        __set_tp(tcb);
    }

    // Get dtv region
    uintptr_t* dtv = reinterpret_cast<uintptr_t*>( tcb[0] );
    if (dtv == NULL)
    {
        // Lazy allocation
        uintptr_t count = nnosGetModuleCountMax();
        dtv = reinterpret_cast<uintptr_t*>( nnosAllocateMemoryForThreadLocal( (1 + count) * sizeof(uintptr_t) ) );
        NN_ABORT_UNLESS(dtv != NULL);

        dtv[0] = count;   // gen
        std::memset( dtv + 1, 0, count * sizeof(uintptr_t) );

        tcb[0] = (uintptr_t)dtv;
    }

    // Get tls region
    char *tls_block = reinterpret_cast<char*>( dtv[module_id] );
    if (tls_block == NULL)
    {
        // Lazy allocation
        tls_block      = reinterpret_cast<char*>( allocate_tls(module_id) );
        dtv[module_id] = reinterpret_cast<uintptr_t>( tls_block );
    }

    return tls_block + offset;
}
#endif /* defined(NNT_OS_BUILD_PIC) */

//-----------------------------------------------------------------------------

