﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Test Helper 機能
//
//  テストを実装する上で必要となる以下の機能を提供する。
//  ・シーケンスチェッカー
//  ・各個別テストにおける合否判定および集計
//  ・時間待ち機能（Win32 で実装）
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <new>
#include <nn/init.h>
#include <nn/mem.h>
#include <nn/os/os_MemoryAllocatorForThreadLocal.h>
#include "../Common/test_Helper.h"

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
namespace {

#if defined(NNT_OS_SYSTEM_PROGRAM_TEST)
    const size_t                TlsBufferSize = 1 * 1024 * 1024;
#else
    const size_t                TlsBufferSize = 64 * 1024 * 1024;
#endif
    NN_ALIGNAS(4096) uint8_t    g_TlsBuffer[ TlsBufferSize ];
    nn::util::TypedStorage<nn::mem::StandardAllocator,sizeof(nn::mem::StandardAllocator),NN_ALIGNOF(nn::mem::StandardAllocator)>    g_TlsAllocator;

    void* CustomAllocatorForThreadLocal(size_t size, size_t alignment) NN_NOEXCEPT
    {
        // SIGLO-78877:
        // os::StartThread() せずに os::DestroyThread() した場合に、
        // thread_local 用のアロケータからこれらを呼ぶようなコードがあると
        // 多重ユーザ例外でループする問題のテスト
        nn::os::GetThreadNamePointer( nn::os::GetCurrentThread() );
        nn::os::GetThreadPriority( nn::os::GetCurrentThread() );
        nn::os::GetThreadCurrentPriority( nn::os::GetCurrentThread() );

        return Get(g_TlsAllocator).Allocate(size, alignment);
    }

    void CustomDeallocatorForThreadLocal(void* p, size_t size) NN_NOEXCEPT
    {
        // SIGLO-78877:
        // os::StartThread() せずに os::DestroyThread() した場合に、
        // thread_local 用のデアロケータからこれらを呼ぶようなコードがあると
        // 多重ユーザ例外でループする問題のテスト
        nn::os::GetThreadNamePointer( nn::os::GetCurrentThread() );
        nn::os::GetThreadPriority( nn::os::GetCurrentThread() );
        nn::os::GetThreadCurrentPriority( nn::os::GetCurrentThread() );

        NN_UNUSED(size);
        if (p)
        {
            Get(g_TlsAllocator).Free(p);
        }
    }
}
#endif  // NN_BUILD_CONFIG_OS_HORIZON

//---------------------------------------------------------------------------
//  デフォルトの nninitStartup() が適用されないように空関数として定義。
extern "C" void nninitStartup()
{
    static uint8_t MallocBuffer[1024 * 1024];
    nn::init::InitializeAllocator(MallocBuffer, sizeof(MallocBuffer));

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    new( &Get(g_TlsAllocator) ) nn::mem::StandardAllocator;
    Get(g_TlsAllocator).Initialize(g_TlsBuffer, TlsBufferSize);

    // コンパイラのスレッドローカル実装用のメモリアロケータの登録
    nn::os::SetMemoryAllocatorForThreadLocal(
                            CustomAllocatorForThreadLocal,
                            CustomDeallocatorForThreadLocal);

#endif  // NN_BUILD_CONFIG_OS_HORIZON
}

//---------------------------------------------------------------------------
//  テスト用シーケンスチェッククラス
//---------------------------------------------------------------------------

void Sequence::Error(const char* filename, int line)
{
    NN_LOG("\n*** Error! seq=%d (%s() at %d)\n", m_next, filename, line);
    CheckBool( false );
}

void Sequence::Check(int num, const char* filename, int line)
{

    if (m_next > num)
    {
        NN_LOG("\n*** Sequence Error! expect=%d actual=%d (%s() at %d)\n", m_next, num, filename, line);
        CheckBool( false );
    }

    m_next = num;
    Print();
}

void Sequence::Skip(int num, const char* filename, int line)
{

    if (m_next > num)
    {
        NN_LOG("\n*** Sequence Error! expect=%d actual=%d (%s() at %d)\n", m_next, num, filename, line);
        CheckBool( false );
    }

    m_next = num;
    Print();
    NNT_OS_LOG("skip\n");
}

void Sequence::WaitUntil(int num, int next, const char* filename, int line)
{
    int loop = 0;

    while (m_next != num)
    {
        if (++loop >= 100)
        {
            NN_LOG("\n*** Sequence Timeout Error! expect=%d actual=%d (%s() at %d)\n", m_next, num, filename, line);
            CheckBool( false );
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromNanoSeconds( NNT_CALIBRATE_RATE() ) );
    }

    nn::os::SleepThread( nn::TimeSpan::FromNanoSeconds( NNT_CALIBRATE_RATE() * 2 ) );
    Set(next, filename, line);
}

void Sequence::SyncWaitUnsafe(int bitpos, const char* filename, int line)
{
    int loop = 0;

    NNT_MEMORY_BARRIER();
    while ((m_signal & (0x1 << bitpos)) == 0)
    {
        if (++loop >= 100)
        {
            NN_LOG("\n*** Sequence Timeout Error! signal=0x%08x (%s() at %d)\n", m_signal, filename, line);
            CheckBool( false );
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromNanoSeconds( NNT_CALIBRATE_RATE() ) );
        NNT_MEMORY_BARRIER();
    }
}

void Sequence::SyncSignalUnsafe(int bitpos)
{
    NNT_MEMORY_BARRIER();
    m_signal |= (0x1 << bitpos);
    NNT_MEMORY_BARRIER();
}

void Sequence::SyncAckUnsafe(int bitpos)
{
    NNT_MEMORY_BARRIER();
    m_ack |= (0x1 << bitpos);
    NNT_MEMORY_BARRIER();
}

void Sequence::SyncDelay(int count, int bitpos)
{
    for (int i = 0; i < count; ++i)
    {
        NNT_MEMORY_BARRIER();
        if (m_ack & (0x1 << bitpos))
        {
            return;
        }
        nn::os::SleepThread( nn::TimeSpan::FromNanoSeconds( NNT_CALIBRATE_RATE() ) );
    }
}


Sequence    g_Sequence;

//---------------------------------------------------------------------------
//  テスト用結果判定クラス
//---------------------------------------------------------------------------

void CountResult::Check(bool state)
{
    if (state == true)
    {
        m_gross_ok++;
        m_total_ok++;
        m_num_ok++;
        NNT_OS_LOG(" ... OK\n");
    }
    else
    {
        m_gross_ng++;
        m_total_ng++;
        m_num_ng++;
        NNT_OS_LOG(" ... NG\n");
        if (g_Sequence.GetFunctionName())
        {
            NNT_OS_LOG("<< func=\x22%s()\x22 line=%d >>\n",
                g_Sequence.GetFunctionName(), g_Sequence.GetLineNumber());
        }
    }
}

void CountResult::Show()
{
    NN_LOG("\n");
    NN_LOG("----------------------------\n");
    NN_LOG("TOTAL: OK= %d\n", m_total_ok);
    NN_LOG("       NG= %d\n", m_total_ng);
    NN_LOG("    Retry= %d\n", m_total_retry);
    NN_LOG("\n");
}

void CountResult::ShowGross()
{
    NN_LOG("\n");
    NN_LOG("----------------------------\n");
    NN_LOG("TOTAL: OK= %d\n", m_gross_ok);
    NN_LOG("       NG= %d\n", m_gross_ng);
    NN_LOG("    Retry= %d\n", m_gross_retry);
    NN_LOG("\n");
}

CountResult g_Result;

//---------------------------------------------------------------------------

