﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Barrier 専用のテストヘルパー
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "test_BarrierHelper.h"


//---------------------------------------------------------------------------

namespace nnt { namespace os {
namespace barrier {

NN_ALIGNAS( 4096 ) char g_StackOfThread1[ 16384 ];
NN_ALIGNAS( 4096 ) char g_StackOfThread2[ 16384 ];

int             g_seq;
BarrierHelper   g_BarrierHelper;

//---------------------------------------------------------------------------
// スレッド１の生成
void BarrierHelper::CreateThread1( TestThreadAction state )
{
    if (state == THR_NOTUSE)
    {
        return;
    }

    SEQ_NONE();
    NNT_OS_LOG("Create Thread1:");
    nn::Result err = nn::os::CreateThread(
                        &m_Thread1,                     // Thread オブジェクト
                        &ThreadFunc1,                   // エントリ関数
                        reinterpret_cast<void*>(0x1111),// 引数
                        g_StackOfThread1,               // スタックアドレス
                        16384,                          // スタックサイズ
                        priThread1);                    // 優先度

    CheckBool( err.IsSuccess() );
}

// スレッド２の Initialize
void BarrierHelper::CreateThread2( TestThreadAction state )
{
    if (state == THR_NOTUSE)
    {
        return;
    }

    SEQ_NONE();
    NNT_OS_LOG("Create Thread2:");
    nn::Result err = nn::os::CreateThread(
                      &m_Thread2,                     // Thread オブジェクト
                      &ThreadFunc2,                   // エントリ関数
                      reinterpret_cast<void*>(0x2222),// 引数
                      g_StackOfThread2,               // スタックアドレス
                      16384,                          // スタックサイズ
                      priThread2);                    // 優先度

    CheckBool( err.IsSuccess() );
}

//---------------------------------------------------------------------------
// スレッド１の破棄
void BarrierHelper::DestroyThread1( TestThreadAction state )
{
    if (state == THR_NOTUSE)
    {
        return;
    }

    SEQ_NONE();
    NNT_OS_LOG("Destroy Thread1:");
    nn::os::DestroyThread( &m_Thread1 );

    CheckBool( true );
}

// スレッド２の破棄
void BarrierHelper::DestroyThread2( TestThreadAction state )
{
    if (state == THR_NOTUSE)
    {
        return;
    }

    SEQ_NONE();
    NNT_OS_LOG("Destroy Thread2:");
    nn::os::DestroyThread( &m_Thread2 );

    CheckBool( true );
}


//---------------------------------------------------------------------------
// Barrier オブジェクトの Initialize
void BarrierHelper::InitializeBarrier(int numThreads)
{
    SEQ_NONE();
    NNT_OS_LOG("Initialize Barrier numThreads=%d", numThreads);

    nn::os::InitializeBarrier( &m_Barrier,      // Barrier オブジェクト
                               numThreads);     // 待ち合わせスレッド数

    CheckBool( true );
}

//---------------------------------------------------------------------------
// Barrier オブジェクトのクリーンナップ（待ちスレッドを解除）
void BarrierHelper::Cleanup()
{
    // Cleanup 後の状態へ
    m_isCleanuped = true;

    // ダミーでバリア待ちに入る
    if (m_Barrier._numThreads > 0)
    {
        AwaitBarrier( &m_Barrier );
    }

    // ２つのスレッドが本当に終了するまで待つ
    if ((GetThreadExpectState1() != THR_STATE_NONE) &&
        (m_Thread1._state == nn::os::ThreadType::State_Started) )
    {
        nn::os::WaitThread( &m_Thread1 );
    }
    if ((GetThreadExpectState2() != THR_STATE_NONE) &&
        (m_Thread2._state == nn::os::ThreadType::State_Started) )
    {
        nn::os::WaitThread( &m_Thread2 );
    }
}

//---------------------------------------------------------------------------
// Barrier オブジェクトの Finalize
void BarrierHelper::FinalizeBarrier()
{
    SEQ_NONE();
    NNT_OS_LOG("Finalize Barrier:");
    nn::os::FinalizeBarrier( &m_Barrier );

    CheckBool( true );
}

//---------------------------------------------------------------------------
//  スレッド実行の開始
void BarrierHelper::StartThread1( TestThreadAction state )
{
    SEQ_NONE();

    if (state != THR_NOTUSE)
    {
        NNT_OS_LOG("Start Thread1:");
        nn::os::StartThread( &m_Thread1 );     // Thread オブジェクト
        CheckBool( true );
    }
    else
    {
        NNT_OS_LOG("Thread1 is not used\n");
    }
}

void BarrierHelper::StartThread2( TestThreadAction state )
{
    SEQ_NONE();

    if (state != THR_NOTUSE)
    {
        NNT_OS_LOG("Start Thread2:");
        nn::os::StartThread( &m_Thread2 );     // Thread オブジェクト
        CheckBool( true );
    }
    else
    {
        NNT_OS_LOG("Thread2 is not used\n");
    }
}

//---------------------------------------------------------------------------
// バリアの内部状態のチェック（非シグナル状態にしてリターン）
void    BarrierHelper::BarrierCheck( int expectNumThreads )
{
    // 待機スレッド数のチェック
    CheckParam( m_Barrier._numThreads, expectNumThreads );
}

//---------------------------------------------------------------------------
// スレッド状態のチェック
void    BarrierHelper::ThreadStateCheck( const nn::os::ThreadType&  pThread,
                                         TestThreadAction           action,
                                         TestThreadLastState        expect,
                                         bool                       apiResult,
                                         bool                       apiExpect )
{
    NN_UNUSED( action );

    // 現在のスレッド状態を表示
    switch (pThread._state)
    {
        case nn::os::ThreadType::State_Initialized:
            {
                NNT_OS_LOG(" INITIALIZED");
            }
            break;

        case nn::os::ThreadType::State_Started:
            {
                NNT_OS_LOG(" STARTED");
            }
            break;

        case nn::os::ThreadType::State_Exited:
            {
                NNT_OS_LOG(" EXITED");
            }
            break;

        case nn::os::ThreadType::State_NotInitialized:
            {
                NNT_OS_LOG(" NotInitialized");
            }
            break;

        default:
            {
                NNT_OS_LOG(" UNKNOWN");
            }
            break;
    }


    // スレッドの事後期待値を検査
    switch (expect)
    {
        // スレッド未使用
        case THR_STATE_NONE:
            {
                CheckBool( true );
            }
            break;

        // EXITED
        case THR_STATE_EXITED:
            {
                CheckParam( pThread._state, nn::os::ThreadType::State_Exited );
                NNT_OS_LOG("(-----): API return value:");
                CheckParam( apiResult, apiExpect );
            }
            break;

        // WAITING
        case THR_STATE_WAITING:
            {
                CheckParam( pThread._state, nn::os::ThreadType::State_Started );
            }
            break;

        // タイムアウトを検知して終了
        case THR_STATE_TIMEDOUT:
            {
                CheckParam( pThread._state, nn::os::ThreadType::State_Exited );
                NNT_OS_LOG("(-----): API return value:");
                CheckParam( apiResult, apiExpect );
            }
            break;

        // 想定外のエラー（テスト不具合）
        default:
            {
                NN_ASSERT( false );
            }
            break;
    }
}

//---------------------------------------------------------------------------
//  OS-API 発行
//---------------------------------------------------------------------------

bool    BarrierHelper::CallBarrierApi( TestThreadAction     action,
                                       int                  seq)
{
    bool    result = true;

    SEQ_CHECK( seq );
    NNT_OS_LOG("Call ");

    // スレッド事前条件の検査
    switch (action)
    {
        // NOTUSE 系（スレッドを使用しない）
        case THR_NOTUSE:
            {
                NN_ASSERT( false );
            }
            break;

        // AwaitBarrier() 待ち状態（無期限）
        case THR_WAIT_WAIT:
            {
                NNT_OS_LOG("AwaitBarrier() goes WAIT\n");
                nn::os::AwaitBarrier( &m_Barrier );
            }
            break;

        // AwaitBarrier() の発行
        case THR_CALL_WAIT:
            {
                NNT_OS_LOG("AwaitBarrier()\n");
                nn::os::AwaitBarrier( &m_Barrier );
            }
            break;

        // 想定外のエラー（テスト不具合）
        default:
            {
                NN_ASSERT( false );
            }
            break;
    }

    return result;
}

//---------------------------------------------------------------------------
//  テスト用スレッド関数の本体
//---------------------------------------------------------------------------
void    ThreadFunc1(void* exinf)
{
    TestThreadAction    action      = g_BarrierHelper.GetThreadAction1();
    TestThreadLastState expectState = g_BarrierHelper.GetThreadExpectState1();
    bool                expect      = g_BarrierHelper.GetThreadExpect1();
    bool                result;

    NN_UNUSED( exinf );

    SEQ_SYNC_WAIT( NNT_POINT_THREAD1_START );

    // スレッド起動
    SEQ_CHECK( g_seq + 1 );
    NNT_OS_LOG("Begin of thread1:");
    CheckBool( true );

    // Thread1 から API を発行
    g_BarrierHelper.SetApiResult1( false );
    result = g_BarrierHelper.CallBarrierApi(action, g_seq + 2);
    g_BarrierHelper.SetApiResult1( result );

    // Cleanuped ならスレッド終了（THR_STATE_WAITING だった場合など）
    if ( g_BarrierHelper.IsCleanuped() )
    {
        NNT_OS_LOG("(-----): Thread1: Canceled\n");
        return;
    }

    SEQ_SYNC_ACK( NNT_POINT_THREAD1_CALLED );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD1_CALLED );

    // スレッドの事後状態別の後処理
    switch (expectState)
    {
        // API から普通に戻ってきた場合
        case THR_STATE_EXITED:
            {
                SEQ_CHECK( g_seq + 5 );
                NNT_OS_LOG("Thread1: return from API:");
                CheckBool( result <= expect );
            }
            break;

        case THR_STATE_TIMEDOUT:
            break;

        // 想定外のエラー（テスト不具合）
        case THR_STATE_WAITING:
        default:
            CheckBool( false );
            break;
    }

    // スレッド終了
    SEQ_CHECK( expectState == THR_STATE_TIMEDOUT ? g_seq + 9 : g_seq + 6 );
    NNT_OS_LOG("End of thread1:");
    CheckBool( true );

    SEQ_SYNC_ACK( NNT_POINT_THREAD1_END );
}


void    ThreadFunc2(void* exinf)
{
    TestThreadAction    action      = g_BarrierHelper.GetThreadAction2();
    TestThreadLastState expectState = g_BarrierHelper.GetThreadExpectState2();
    bool                expect      = g_BarrierHelper.GetThreadExpect2();
    bool                result;

    NN_UNUSED( exinf );

    SEQ_SYNC_WAIT( NNT_POINT_THREAD2_START );

    // スレッド起動
    SEQ_CHECK( g_seq + 3 );
    NNT_OS_LOG("Begin of thread2:");
    CheckBool( true );

    // Thread2 から API を発行
    g_BarrierHelper.SetApiResult2( false );
    result = g_BarrierHelper.CallBarrierApi(action, g_seq + 4);
    g_BarrierHelper.SetApiResult2( result );

    // Cleanuped ならスレッド終了（THR_STATE_WAITING だった場合など）
    if ( g_BarrierHelper.IsCleanuped() )
    {
        NNT_OS_LOG("(-----): Thread2: Canceled\n");
        return;
    }

    SEQ_SYNC_ACK( NNT_POINT_THREAD2_CALLED );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD2_CALLED );

    // スレッドの事後状態別の後処理
    switch (expectState)
    {
        // API から普通に戻ってきた場合
        case THR_STATE_EXITED:
            {
                SEQ_CHECK( g_seq + 7 );
                NNT_OS_LOG("Thread2: return from API:");
                CheckParam( result, expect );
            }
            break;

        case THR_STATE_TIMEDOUT:
            break;

        // 想定外のエラー（テスト不具合）
        case THR_STATE_WAITING:
        default:
            CheckBool( false );
            break;
    }

    // スレッド終了
    SEQ_CHECK( expectState == THR_STATE_TIMEDOUT? g_seq + 10 : g_seq + 8 );
    NNT_OS_LOG("End of thread2:");
    CheckBool( true );

    SEQ_SYNC_ACK( NNT_POINT_THREAD2_END );
}

//---------------------------------------------------------------------------

int    BarrierHelper::GetLastSeq( TestThreadLastState th1ex,
                                  TestThreadLastState th2ex )
{
    static const int lastSeqTable[THR_STATE_MAX][THR_STATE_MAX] =
    {
    //  th2: NONE,WAITING, EXITED, TIMEDOUT
        {       0,      0,      0,      0,      },  // NONE
        {       2,      4,      8,      10,     },  // WAITING
        {       6,      6,      8,      10,     },  // EXITED
        {       9,      9,      9,      10,     },  // TIMEDOUT
    };                                              // th1

    return g_seq + lastSeqTable[th1ex][th2ex];
}

//---------------------------------------------------------------------------
//  doTest() 関数
//---------------------------------------------------------------------------

void    BarrierHelper::doTest( int                  numThreads,
                               TestThreadAction     th1,
                               TestThreadAction     th2,
                               bool                 rslt1,
                               bool                 rslt2,
                               int                  expectNumThreads,
                               TestThreadLastState  th1ex,
                               TestThreadLastState  th2ex,
                               const char*          functionName,
                               int                  line)
{
    // シーケンス番号を次の 100番台 へ進める
    g_seq = g_Sequence.Get();
    g_seq = ((g_seq + 100) / 100) * 100;

    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    // テスト用 API の発行元情報を退避
    g_Sequence.SetFunctionName( functionName );
    g_Sequence.SetLineNumber( line );

    // Cleanup 前の状態へ
    m_isCleanuped = false;
    g_BarrierHelper.SetApiResult1( true );
    g_BarrierHelper.SetApiResult2( true );

    // スレッドオブジェクトの Initialize
    g_seq = (g_seq / 100) * 100;
    SEQ_SET( g_seq );
    NNT_OS_LOG("Start Test\n");
    CreateThread1( th1 );
    CreateThread2( th2 );

    // バリアオブジェクトの Initialize
    InitializeBarrier( numThreads );

    // スレッドの状態を構築
    SetThreadInfo1( th1, th1ex, rslt1 );
    SetThreadInfo2( th2, th2ex, rslt2 );

    // 順番にテストシーケンスを指示する
    SEQ_SYNC_INIT();

    StartThread1( th1 );
    StartThread2( th2 );
    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_START );
    SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD1_CALLED );

    if (th2ex != THR_STATE_NONE)
    {
        SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_START );
        SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD2_CALLED );
    }

    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_CALLED );
    SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD1_END );

    if (th2ex != THR_STATE_NONE)
    {
        SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_CALLED );
        SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD2_END );
    }

    // タイムアウト系の API は、もう片方のスレッドで同期が成立しない場合、
    // この時点で既にタイムアウトが成立し、対象スレッドは
    // THR_TIMEDOUT_EXITED 状態という形で ExitThread() 済みになっている。

    // スレッド１の状態を検査
    SEQ_WAIT_UNTIL( GetLastSeq(th1ex, th2ex), g_seq + 11 );
    NNT_OS_LOG("Thread1 state check:");
    ThreadStateCheck( m_Thread1, th1, th1ex, GetApiResult1(), rslt1 );

    // スレッド２の状態を検査
    SEQ_NONE();
    NNT_OS_LOG("Thread2 state check:");
    ThreadStateCheck( m_Thread2, th2, th2ex, GetApiResult2(), rslt2 );

    // バリアの状態を検査
    SEQ_NONE();
    NNT_OS_LOG("Barrier state check:");
    BarrierCheck( expectNumThreads );

    // バリア状態を初期状態へ
    SEQ_NONE();
    NNT_OS_LOG("Cleanup Barrier and Threads\n");
    Cleanup();

    // バリアオブジェクトの Finalize
    FinalizeBarrier();

    // スレッドの破棄
    DestroyThread1( th1 );
    DestroyThread2( th2 );

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

}   // namespace barrier
}}  // namespace nnt::os

