﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Barrier 関連機能のテスト
//---------------------------------------------------------------------------

// windows.h で min/max が定義されるのを抑制する
#ifndef NOMINMAX
#define NOMINMAX
#endif

#include "../Common/test_Pragma.h"

#include <nn/nn_Common.h>
#include <nn/nn_SdkText.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "../Common/test_TargetInfo.h"
#include "test_BarrierHelper.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

namespace nnt { namespace os { namespace barrier {

//---------------------------------------------------------------------------
// 単独の AwaitBarrier() テスト

TEST(AwaitBarrier, test_AwaitThread0)
{
    // AwaitBarrier() × 0 回
    doTest1(2,                          // 待ち合わせスレッド数
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            0,                          // 待機スレッド数
            THR_STATE_NONE);            // Thread1 は終了
}

TEST(AwaitBarrier, test_AwaitThread1)
{
    // AwaitBarrier() × 1 回
    doTest1(2,                          // 待ち合わせスレッド数
            THR_WAIT_WAIT,              // Thread1 は AwaitBarrier()
            true,                       // Thread1 での API 返値
            1,                          // 待機スレッド数
            THR_STATE_WAITING);         // Thread1 は待ち
}

TEST(AwaitBarrier, test_AwaitThread2)
{
    // AwaitBarrier() × 2 回
    doTest2(2,                          // 待ち合わせスレッド数
            THR_WAIT_WAIT,              // Thread1 は AwaitBarrier()
            THR_CALL_WAIT,              // Thread2 は AwaitBarrier()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            0,                          // 待機スレッド数
            THR_STATE_EXITED,           // Thread1 は待ち→終了
            THR_STATE_EXITED);          // Thread2 は終了
}

//---------------------------------------------------------------------------
// 複数スレッドから繰り返し Await

const int BarrierTestMaxThreadNum    = MaxCoreNum * 2 + 1;     // テスト可能な最大スレッド数
const int BarrierTestThreadStackSize = 8192;               // スタックサイズ

/*!
  @struct       ThreadParam
  @brief        test_BarrierClassTestThreadFunc に渡すテスト用パラメータの構造体です。
*/
struct ThreadParam
{
    nn::os::ThreadType    thread;
    nn::os::BarrierType*  pBarrier;                                 //!< テスト対象のバリアのポインタ
    int                   threadIndex;                              //!< スレッド番号
    int                   loopNum;                                  //!< ループ回数
    bool                  flag;
};

// スレッドプール
ThreadParam           g_ThreadPool[BarrierTestMaxThreadNum];
NN_ALIGNAS(4096) char g_ThreadStack[BarrierTestMaxThreadNum][BarrierTestThreadStackSize];

static_assert( (BarrierTestThreadStackSize % nn::os::StackRegionAlignment) == 0, "");


void BarrierTestThreadFunc(void* arg)
{
    ThreadParam& param = *reinterpret_cast<ThreadParam*>( arg );

    // サブスレッドが実行されたことのフラグを立てる
    param.flag = true;

    // 指定された回数だけ繰り返し Await
    for (int i=0; i<param.loopNum; i++)
    {
        nn::os::AwaitBarrier(param.pBarrier);
    }
}

void BarrierTestAwait(int threadNum, int loopNum, bool affinity, bool mainFirst=true)
{
    NN_ASSERT( threadNum <= BarrierTestMaxThreadNum );
    NN_ASSERT( nn::os::GetThreadPriority(nn::os::GetCurrentThread()) > 0 );

    // スレッド優先度の設定
    int subThreadPriority = nn::os::GetThreadPriority(nn::os::GetCurrentThread());
    if (mainFirst)
    {
        ++subThreadPriority;
    }
    else
    {
        --subThreadPriority;
    }

    // バリアの初期化
    nn::os::BarrierType barrier;
    nn::os::InitializeBarrier(&barrier, threadNum);

    // テスト用サブスレッドの作成（メインスレッドは自スレッド）
    for (int i=0; i<threadNum; ++i)
    {
        ThreadParam& param = g_ThreadPool[i];
        param.threadIndex = i;
        param.loopNum     = loopNum;
        param.pBarrier    = &barrier;
        param.flag        = false;
        if (affinity)
        {
            nn::os::CreateThread( &param.thread, BarrierTestThreadFunc, &param, g_ThreadStack[i], BarrierTestThreadStackSize, subThreadPriority, param.threadIndex % MaxCoreNum);
        }
        else
        {
            nn::os::CreateThread( &param.thread, BarrierTestThreadFunc, &param, g_ThreadStack[i], BarrierTestThreadStackSize, subThreadPriority);
        }
        nn::os::StartThread( &param.thread );
    }

    // サブスレッドの待ち合わせを待つ
    for (int i=0; i<threadNum; ++i)
    {
        nn::os::WaitThread( &g_ThreadPool[i].thread );
    }

    // 全サブスレッドでフラグが立てられている（実行済みである）ことを確認
    for (int i=0; i<threadNum; ++i)
    {
        CheckBool( g_ThreadPool[i].flag == true );
    }

    // スレッドの破棄
    for (int i=0; i<threadNum; ++i)
    {
        nn::os::DestroyThread( &g_ThreadPool[i].thread );
    }

    // バリアの Finalize
    nn::os::FinalizeBarrier(&barrier);
}

TEST(AwaitBarrier, test_BarrierAwait1Threads)
{
    CLEAR_GOOGLE_TEST();

    NNT_OS_LOG(NN_TEXT("1 スレッドで 1 回だけ待ち合わせ\n"));
    BarrierTestAwait(1, 1, false);

    NNT_OS_LOG(NN_TEXT("1 スレッドで 50000 回待ち合わせ\n"));
    BarrierTestAwait(1, 50000, false);

    JUDGE_GOOGLE_TEST();
}

TEST(AwaitBarrier, test_BarrierAwait2Threads)
{
    CLEAR_GOOGLE_TEST();

    NNT_OS_LOG(NN_TEXT("2 スレッドで 1 回だけ待ち合わせ\n"));
    BarrierTestAwait(2, 1, false, true);

    NNT_OS_LOG(NN_TEXT("2 スレッドで 50000 回待ち合わせ\n"));
    BarrierTestAwait(2, 50000, false, true);

    JUDGE_GOOGLE_TEST();
}

TEST(AwaitBarrier, test_BarrierAwait3Threads)
{
    CLEAR_GOOGLE_TEST();

    NNT_OS_LOG(NN_TEXT("3 スレッドで 1 回だけ待ち合わせ\n"));
    BarrierTestAwait(3, 1, false);

    NNT_OS_LOG(NN_TEXT("3 スレッドで 50000 回待ち合わせ\n"));
    BarrierTestAwait(3, 50000, false);

    JUDGE_GOOGLE_TEST();
}

TEST(AwaitBarrier, test_BarrierAwaitAffinity)
{
    CLEAR_GOOGLE_TEST();

    NNT_OS_LOG(NN_TEXT("%d スレッドで 1 回だけ待ち合わせ＆各CPUコアに 1 スレッドずつ生成\n"), MaxCoreNum);
    BarrierTestAwait(MaxCoreNum, 1, true);

    NNT_OS_LOG(NN_TEXT("%d スレッドで 1 回だけ待ち合わせ＆各CPUコアに 2 スレッドずつ生成\n"), MaxCoreNum * 2);
    BarrierTestAwait(MaxCoreNum * 2, 1, true);

    JUDGE_GOOGLE_TEST();
}

TEST(AwaitBarrier, test_BarrierAwaitCheckThreadOrder)
{
    CLEAR_GOOGLE_TEST();

    NNT_OS_LOG(NN_TEXT("%d スレッドで 1 回だけ待ち合わせ（メインスレッド優先）\n"), MaxCoreNum);
    BarrierTestAwait(MaxCoreNum, 1, false, true);

    NNT_OS_LOG(NN_TEXT("%d スレッドで 1 回だけ待ち合わせ（サブスレッド優先）\n"), MaxCoreNum);
    BarrierTestAwait(MaxCoreNum, 1, false, false);

    NNT_OS_LOG(NN_TEXT("%d スレッドで 1 回だけ待ち合わせ＆各CPUコアに 1 スレッドずつ生成（メインスレッド優先）\n"), MaxCoreNum);
    BarrierTestAwait(MaxCoreNum, 1, true, true);

    NNT_OS_LOG(NN_TEXT("%d スレッドで 1 回だけ待ち合わせ＆各CPUコアに 1 スレッドずつ生成（サブスレッド優先）\n"), MaxCoreNum);
    BarrierTestAwait(MaxCoreNum, 1, true, false);

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// DEATH TEST

TEST(BarrierDeathTest, test_BarrierDeathTestInitialize)
{
    CLEAR_GOOGLE_TEST();

    nn::os::BarrierType barrier;

    // Initialize の事前条件を満たさない場合を確認
    EXPECT_DEATH_IF_SUPPORTED( nn::os::InitializeBarrier(&barrier, std::numeric_limits<int>::min()), "" );
    EXPECT_DEATH_IF_SUPPORTED( nn::os::InitializeBarrier(&barrier, -1), "" );
    EXPECT_DEATH_IF_SUPPORTED( nn::os::InitializeBarrier(&barrier,  0), "" );

    JUDGE_GOOGLE_TEST();
}

nn::os::ThreadType    g_ThreadFinalizeDeathTest;
NN_ALIGNAS(4096) char g_ThreadFinalizeDeathTestStack[BarrierTestThreadStackSize];

void BarrierTestInvalidFinalizeThreadFunc(void* arg)
{
    nn::os::BarrierType& barrier = *reinterpret_cast<nn::os::BarrierType*>( arg );
    nn::os::AwaitBarrier(&barrier);
}

TEST(BarrierDeathTest, test_BarrierDeathTestFinalize)
{
    CLEAR_GOOGLE_TEST();

    // バリアの初期化
    nn::os::BarrierType barrier;
    nn::os::InitializeBarrier(&barrier, 2);

    // サブスレッドの準備
    nn::os::CreateThread( &g_ThreadFinalizeDeathTest, BarrierTestInvalidFinalizeThreadFunc, &barrier, g_ThreadFinalizeDeathTestStack, BarrierTestThreadStackSize, nn::os::GetThreadPriority(nn::os::GetCurrentThread()) - 1);
    nn::os::StartThread( &g_ThreadFinalizeDeathTest );

    // サブスレッドが Await するまで待つ
    while (barrier._numThreads == 0)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    // Await したスレッドがいる状態で Finalize
    EXPECT_DEATH_IF_SUPPORTED( nn::os::FinalizeBarrier(&barrier), "" );

    // 終了処理（EXPECT_DEATH がサポートされていない環境向け）
    nn::os::AwaitBarrier(&barrier);
    nn::os::WaitThread( &g_ThreadFinalizeDeathTest );
    nn::os::DestroyThread( &g_ThreadFinalizeDeathTest );
    nn::os::FinalizeBarrier(&barrier);

    JUDGE_GOOGLE_TEST();
}

TEST(BarrierDeathTest, test_BarrierDeathTestAwait)
{
    CLEAR_GOOGLE_TEST();

    nn::os::BarrierType barrier;

    // Await の事前条件を満たさない場合を確認
    EXPECT_DEATH_IF_SUPPORTED( nn::os::AwaitBarrier(&barrier), "" );

    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// Barrier クラステスト

nn::os::ThreadType    g_ThreadClassTest;
NN_ALIGNAS(4096) char g_ThreadClassTestStack[BarrierTestThreadStackSize];
bool                  g_ClassTestFlag = false;

void BarrierClassTestThreadFunc(void* arg)
{
    nn::os::Barrier& barrier = *reinterpret_cast<nn::os::Barrier*>( arg );

    // サブスレッドが実行されたフラグを立てる
    g_ClassTestFlag = true;

    // Await
    barrier.Await();
}

TEST(BarrierClassTest, test_BarrierClassTest)
{
    CLEAR_GOOGLE_TEST();

    NNT_OS_LOG(NN_TEXT("バリアクラスでも待ち合わせできることを確認\n"));

    // バリアの初期化
    g_ClassTestFlag = false;
    nn::os::Barrier barrier(2);

    // サブスレッドの準備
    nn::os::CreateThread( &g_ThreadClassTest, BarrierClassTestThreadFunc, &barrier, g_ThreadClassTestStack, BarrierTestThreadStackSize, nn::os::GetThreadPriority(nn::os::GetCurrentThread()) + 1);
    nn::os::StartThread( &g_ThreadClassTest );

    // Await
    barrier.Await();

    // サブスレッドが実行済みであることを確認
    CheckBool( g_ClassTestFlag == true );

    // 終了処理
    nn::os::WaitThread( &g_ThreadClassTest );
    nn::os::DestroyThread( &g_ThreadClassTest );

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  Barrier クラスの型変換関数のテスト
//  ここでは、Barrier クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  ここでは、GetBase() および operator BarrierType&() の動作テストを行なう。
//
NN_ALIGNAS(4096) char   g_ThreadStackBarrierClass[4096];

void test_BarrierClassTypeExchangeTestThread(void* arg)
{
    auto*  barrierType = static_cast<nn::os::BarrierType*>(arg);

    // 引数で受取った BarrierType オブジェクトに対して Await する
    nn::os::AwaitBarrier(barrierType);
}

TEST(BarrierClass, test_BarrierClassTestTypeExchange)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("barrier インスタンスの生成\n"));
        nn::os::Barrier barrier(2);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("barrier.GetBase() で BarrierType オブジェクトを取得"));
        nn::os::BarrierType*  barrierType = barrier.GetBase();
        CheckBool( barrierType != NULL );

        // 子スレッドを作成
        // 子スレッドには直前で GetBase() した BarrierType オブジェクトを渡す
        nn::os::ThreadType  thread;
        nn::os::CreateThread( &thread, test_BarrierClassTypeExchangeTestThread, barrierType, g_ThreadStackBarrierClass, sizeof(g_ThreadStackBarrierClass), nn::os::DefaultThreadPriority);

        // 子スレッドをスタートしつつ、barrier クラスを使って待ち合わせる。
        // この時、子スレッドは BarrierType オブジェクトを使って待ち合わせる。
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("barrier.Await() と AwaitBarrier() で待ち合わせる"));
        nn::os::StartThread( &thread );
        barrier.Await();
        CheckBool( true );

        // 子スレッドの終了待ち
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );

        // operator BarrierType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator BarrierType&() を使ってオブジェクトの参照を取得"));
        nn::os::BarrierType& barrierTypeRefer = barrier;
        CheckBool( &barrierTypeRefer == barrierType );

        // operator const BarrierType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator const BarrierType&() を使ってオブジェクトの参照を取得"));
        const nn::os::BarrierType& barrierTypeReferConst = barrier;
        CheckBool( &barrierTypeReferConst == barrierType );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


}}} // namespace nnt::os::barrier

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of Barrier APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of Barrier APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
