﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/base/testBase_Exit.h>

#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_Result.public.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_UserAccountSystemSaveData.h>
#include <nn/fs/fs_SaveDataTransferVersion2.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_Utility.h>

#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_ApiForPrivate.h>
#include <nn/olsc/olsc_ApiForSystemService.h>
#include <nn/olsc/olsc_RemoteStorageController.h>
#include <nn/olsc/olsc_TransferTaskListController.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/olsc_Util.h>
#include <nn/olsc/srv/database/olsc_SaveDataArchiveInfoCache.h>
#include <nn/olsc/srv/util/olsc_Account.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/olsc/srv/transfer/olsc_ComponentFileUpload.h>
#include <nn/olsc/srv/transfer/olsc_ComponentFileDownload.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveUpload.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveDownload.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForAdministrators.h>

#include <nnt/fsUtil/testFs_util.h>
#include <nnt/fsUtil/testFs_util_function.h>
#include <nn/crypto/crypto_Md5Generator.h>
#include <nn/util/util_Base64.h>

#include "testOlsc_Util.h"
#include "testOlsc_SaveDataUtil.h"



using namespace nn;
using namespace nn::olsc;
using namespace nn::olsc::srv;

namespace {

    const size_t WorkBufferSize        = 4 * 1024 * 1024;
    const size_t SaveDataSize          = 64 *  1024;
    const size_t SaveDataJournalSize   = 64 *  1024;

    static const int MaxComponentFileCount = 64;
    static const int PartitionNum = 3;
    const std::string OriginalFileName = "original";
    const std::string DiffFileName     = "diff";
}

// 受け取った manager に Ksp をセットする
Result SetKeySeedPackage(fs::SaveDataTransferManagerForCloudBackUp& manager, nn::olsc::srv::SaveDataArchiveId id, nn::olsc::srv::NsaIdToken& nsaIdToken, CURL* curl, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    fs::SaveDataTransferManagerForCloudBackUp::Challenge challenge;
    NNT_OLSC_RESULT_DO(manager.GetChallenge(&challenge));

    fs::SaveDataTransferManagerForCloudBackUp::KeySeedPackage keySeedPackage;
    NNT_OLSC_RESULT_DO(olsc::srv::transfer::GetKeySeedPackage(
        &keySeedPackage, id, challenge,
        nsaIdToken, curl, workBuffer, workBufferSize, nullptr));
    NNT_OLSC_RESULT_DO(manager.SetKeySeedPackage(keySeedPackage));

    NN_RESULT_SUCCESS;
}

// 受け取った ComponentFileInfo のリストから InitialData の ComponentFileInfo を探して DL
Result GetInitialData(size_t* pOutInitialDataSize, fs::InitialDataForCloudBackUp* pOutInitialData, nn::olsc::srv::ComponentFileInfo cfList[], int cfCount, nn::olsc::srv::NsaIdToken& nsaIdToken, CURL* curl, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    // 受け取った cfList から、まず InitialData を検索
    nn::util::optional<olsc::srv::ComponentFileInfo> initialDataCf;
    nnt::olsc::FindElement(&initialDataCf, cfList, cfCount, [](olsc::srv::ComponentFileInfo i) {
        return i.clientArgument.chunkId == nn::fs::SaveDataChunkIdForInitialData;
    });
    EXPECT_TRUE(initialDataCf);

    // InitialData の URL を要求
    olsc::srv::ComponentFileInfo cf;
    NNT_OLSC_RESULT_DO(olsc::srv::transfer::GetComponentFileSignedUrlForDownload(&cf, initialDataCf->id, nsaIdToken, curl, workBuffer, WorkBufferSize, nullptr));

    // InitialData を 単独で DL
    NNT_OLSC_RESULT_DO(olsc::srv::transfer::RequestDownloadComponentFile(pOutInitialDataSize, pOutInitialData, fs::InitialDataForCloudBackUp::Size, cf.url, cf.saveDataChunkSize, curl, nullptr));

    NN_RESULT_SUCCESS;
}

Result TestExportSaveData(nn::olsc::srv::SaveDataArchiveInfo* pOutSda, nn::olsc::srv::NsaIdToken& nsaIdToken, account::Uid& uid, ApplicationId& appId, const fs::SaveDataInfo& infoSrc, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    auto curl = curl_easy_init();
    olsc::srv::SaveDataArchiveInfo sda;

    fs::SaveDataTransferManagerForCloudBackUp manager;
    std::unique_ptr<fs::ISaveDataDivisionExporter> exporter;

    const fs::SaveDataCommitId TestSaveDataCommitId = 0x4e728abeca603d6c;

    olsc::srv::SeriesInfo info;
    info.seriesId = 0x0001;
    info.commitId = TestSaveDataCommitId;
    time::PosixTime time = { 1413263600 };
    // uint32_t launchRequiredVersion = 10;

    NNT_OLSC_RESULT_DO(nn::olsc::srv::transfer::StartSaveDataArchiveUpload(&sda, uid, appId, info, static_cast<size_t>(infoSrc.saveDataSize), time, nsaIdToken, curl, workBuffer, WorkBufferSize, nullptr));
    DumpSaveDataArchiveInfo(sda);

    NNT_OLSC_RESULT_DO(manager.OpenSaveDataFullExporter(&exporter, infoSrc.saveDataSpaceId, infoSrc.saveDataId));
    NN_LOG("Division Count = %d\n", sda.numOfPartitions);
    exporter->SetDivisionCount(PartitionNum);
    std::unique_ptr<fs::ISaveDataChunkIterator> iter;
    NNT_OLSC_RESULT_DO(exporter->OpenSaveDataDiffChunkIterator(&iter));

    // 差分のある ComponentFileInfo について（新規なら全 cf について）ループ
    nn::os::Tick startTime = nn::os::GetSystemTick();
    size_t offsetNumber = 0;
    for (; !iter->IsEnd(); iter->Next())
    {
        auto index = iter->GetId();

        // CF 作成
        olsc::srv::ComponentFileInfo cf;
        olsc::srv::ClientArgument clientArgument;
        clientArgument.chunkId = index;
        olsc::srv::ComponentFileType type = (index == nn::fs::SaveDataChunkIdForInitialData) ? olsc::srv::ComponentFileType::Meta : olsc::srv::ComponentFileType::Save;

        NNT_OLSC_RESULT_DO(nn::olsc::srv::transfer::CreateComponentFile(&cf, sda.id, clientArgument, type, nsaIdToken, curl, workBuffer, WorkBufferSize, nullptr));
        DumpComponentFileInfo(cf);
        size_t size = 0;
        olsc::srv::SaveDataChunkDigest digest;
        NNT_OLSC_RESULT_DO(nn::olsc::srv::transfer::ExportComponentFile(&size, &digest, cf.url, exporter.get(), index, nsaIdToken, curl, workBuffer, WorkBufferSize, nullptr));
        NNT_OLSC_RESULT_DO(nn::olsc::srv::transfer::CompleteComponentFile(cf.id, size, digest, nsaIdToken, curl, workBuffer, WorkBufferSize, nullptr));
        offsetNumber++;
    }

    // TODO : fs から正式な値取得 API を提供されたら呼び出すように修正
    nn::fs::detail::KeySeed keySeedFs;
    nn::fs::detail::InitialDataMac macFs;
    NN_RESULT_DO(exporter->FinalizeFullExport(&keySeedFs, &macFs));
    nn::olsc::srv::KeySeed keySeed;
    memcpy(keySeed.data, keySeedFs.data, nn::olsc::srv::KeySeed::Size);
    nn::olsc::srv::InitialDataMac mac;
    memcpy(mac.data, macFs.data, nn::olsc::srv::InitialDataMac::Size);

    NNT_OLSC_RESULT_DO(nn::olsc::srv::transfer::FinishSaveDataArchiveUpload(sda.id, keySeed, mac, nsaIdToken, curl, workBuffer, WorkBufferSize, nullptr));
    nn::os::Tick endTime = nn::os::GetSystemTick();
    NN_LOG("\nUpload Time = %lldms\n", nn::os::ConvertToTimeSpan( endTime - startTime ).GetMilliSeconds());
    *pOutSda = sda;
    NN_RESULT_SUCCESS;
}

Result TestDiffExportSaveData(nn::olsc::srv::SaveDataArchiveInfo* pOutSda, nn::olsc::srv::NsaIdToken& nsaIdToken, account::Uid& uid, ApplicationId& appId, const fs::SaveDataInfo& infoSrc, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    auto curl = curl_easy_init();

    fs::SaveDataTransferManagerForCloudBackUp manager;
    std::unique_ptr<fs::ISaveDataDivisionExporter> exporter;

    const fs::SaveDataCommitId TestSaveDataCommitId = 0x4e728abeca603d6c;

    olsc::srv::SeriesInfo info;
    info.seriesId = 0x0001;
    info.commitId = TestSaveDataCommitId;
    time::PosixTime time = { 1413263600 };
    // uint32_t launchRequiredVersion = 15;

    // appId 指定で Sda 取得
    nn::util::optional<olsc::srv::SaveDataArchiveInfo> oriSda;
    NNT_OLSC_RESULT_DO(olsc::srv::transfer::RequestFixedSaveDataArchiveInfo(&oriSda, nsaIdToken, appId, curl, workBuffer, workBufferSize, nullptr));
    EXPECT_TRUE(oriSda);

    // manager に Ksp をセット
    NN_RESULT_DO(SetKeySeedPackage(manager, oriSda->id, nsaIdToken, curl, workBuffer, WorkBufferSize));

    nn::olsc::srv::ComponentFileInfo cfList[MaxComponentFileCount];
    int cfCount = 0;

    // sdaId 指定で CfList 取得
    NNT_OLSC_RESULT_DO(olsc::srv::transfer::RequestComponentFileInfoList(&cfCount,
        cfList, MaxComponentFileCount, oriSda->id, nsaIdToken, curl, workBuffer, workBufferSize, nullptr));
    EXPECT_TRUE(cfCount > 0);

    fs::InitialDataForCloudBackUp initialData = {};
    size_t initialDataSize = 0;

    NN_RESULT_DO(GetInitialData(&initialDataSize, &initialData, cfList, cfCount, nsaIdToken, curl, workBuffer, workBufferSize));

    // InitialData を exporter に入れる
    NNT_OLSC_RESULT_DO(manager.OpenSaveDataDiffExporter(&exporter, initialData, infoSrc.saveDataSpaceId, infoSrc.saveDataId));

    // サーバにセーブの duplicate をリクエスト
    olsc::srv::SaveDataArchiveInfo dupSda;
    NNT_OLSC_RESULT_DO(nn::olsc::srv::transfer::StartSaveDataArchiveDifferentialUpload(&dupSda, &cfCount, cfList, static_cast<size_t>(MaxComponentFileCount), oriSda->id,
        info, static_cast<size_t>(infoSrc.saveDataSize), time, nsaIdToken, curl, workBuffer, workBufferSize, nullptr));

    DumpSaveDataArchiveInfo(dupSda);

    exporter->SetDivisionCount(PartitionNum);
    std::unique_ptr<fs::ISaveDataChunkIterator> iter;
    NNT_OLSC_RESULT_DO(exporter->OpenSaveDataDiffChunkIterator(&iter));

    // 差分のある ComponentFileInfo についてループ
    nn::os::Tick startTime = nn::os::GetSystemTick();
    for (; !iter->IsEnd(); iter->Next())
    {
        auto index = iter->GetId();
        // 受け取った cfList から、指定された chunkId を検索
        nn::util::optional<olsc::srv::ComponentFileInfo> targetCf;
        nnt::olsc::FindElement(&targetCf, cfList, cfCount, [&index](olsc::srv::ComponentFileInfo i) {
            return i.clientArgument.chunkId == index;
        });
        EXPECT_TRUE(targetCf);

        olsc::srv::ComponentFileInfo cf = *targetCf;

        // CF 作成
        NNT_OLSC_RESULT_DO(nn::olsc::srv::transfer::UpdateComponentFile(&cf, targetCf->id, nsaIdToken, curl, workBuffer, workBufferSize, nullptr));
        size_t size = 0;
        olsc::srv::SaveDataChunkDigest digest;
        NNT_OLSC_RESULT_DO(nn::olsc::srv::transfer::ExportComponentFile(&size, &digest, cf.url, exporter.get(), index, nsaIdToken, curl, workBuffer, workBufferSize, nullptr));
        NNT_OLSC_RESULT_DO(nn::olsc::srv::transfer::CompleteComponentFile(cf.id, size, digest, nsaIdToken, curl, workBuffer, workBufferSize, nullptr));
    }
    // TODO : fs から正式な値取得 API を提供されたら呼び出すように修正
    nn::fs::detail::InitialDataMac macFs;
    NN_RESULT_DO(exporter->FinalizeDiffExport(&macFs));
    nn::olsc::srv::InitialDataMac mac;
    memcpy(mac.data, macFs.data, nn::olsc::srv::InitialDataMac::Size);
    NNT_OLSC_RESULT_DO(nn::olsc::srv::transfer::FinishSaveDataArchiveDifferentialUpload(dupSda.id, mac, nsaIdToken, curl, workBuffer, workBufferSize, nullptr));
    nn::os::Tick endTime = nn::os::GetSystemTick();
    NN_LOG("\nDiff Upload Time = %lldms\n", nn::os::ConvertToTimeSpan( endTime - startTime ).GetMilliSeconds());

    *pOutSda = dupSda;
    NN_RESULT_SUCCESS;
}

Result TestImportSaveData(olsc::srv::SaveDataArchiveInfo& sda, nn::olsc::srv::NsaIdToken& nsaIdToken, account::Uid uid, ApplicationId appId, fs::SaveDataSpaceId spaceId, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    auto curl = curl_easy_init();

    fs::SaveDataTransferManagerForCloudBackUp manager;
    // manager に Ksp をセット
    NN_RESULT_DO(SetKeySeedPackage(manager, sda.id, nsaIdToken, curl, workBuffer, workBufferSize));

    nn::olsc::srv::ComponentFileInfo cfList[MaxComponentFileCount];
    int cfCount = 0;

    // DL 開始
    NNT_OLSC_RESULT_DO(olsc::srv::transfer::RequestStartDownloadSaveDataArchive(&cfCount,
    cfList, MaxComponentFileCount,
    sda.id, nsaIdToken, curl, workBuffer, workBufferSize, nullptr));

    // InitialData 取得
    fs::InitialDataForCloudBackUp initialData = {};
    size_t initialDataSize = 0;
    NN_RESULT_DO(GetInitialData(&initialDataSize, &initialData, cfList, cfCount, nsaIdToken, curl, workBuffer, workBufferSize));

    // ローカルにセーブがあるかどうかによって Importer を分ける
    nn::util::optional<fs::SaveDataInfo> saveDataInfo;
    nnt::fs::util::FindSaveData(&saveDataInfo, spaceId, [&](fs::SaveDataInfo i) {
        return i.saveDataUserId._data[0] == uid._data[0] && i.saveDataUserId._data[1] == uid._data[1] && i.applicationId.value == appId.value;
    });

    std::unique_ptr<fs::ISaveDataDivisionImporter> importer;
    // ローカルにセーブがある場合は DiffImporter
    if(saveDataInfo)
    {
        NN_LOG("Open Diff Impoerter\n");
        NNT_OLSC_RESULT_DO(manager.OpenSaveDataDiffImporter(&importer, initialData, spaceId, saveDataInfo->saveDataId));
    }
    // ローカルにセーブがない場合は FullImporter
    else
    {
        NN_LOG("Open Full Impoerter\n");
        NNT_OLSC_RESULT_DO(manager.OpenSaveDataFullImporter(&importer, initialData, uid, spaceId));
    }

    int64_t requiredSize;
    NNT_OLSC_RESULT_DO(importer.get()->InitializeImport(&requiredSize));
    // TODO: セーブデータ新規作成・拡張のための空き容量不足のハンドリング
    NN_UNUSED(requiredSize);

    {
        std::unique_ptr<fs::ISaveDataChunkIterator> iter;
        NNT_OLSC_RESULT_DO(importer.get()->OpenSaveDataDiffChunkIterator(&iter));

        // 差分のある sf について（新規なら全 sf について）ループ
        for (; !iter->IsEnd(); iter->Next())
        {
            auto chunkId = iter->GetId();
            std::unique_ptr<fs::ISaveDataChunkImporter> chunkImporter;
            NNT_OLSC_RESULT_DO(importer.get()->OpenSaveDataChunkImporter(&chunkImporter, chunkId));
            nn::util::optional<olsc::srv::ComponentFileInfo> targetCf;
            nnt::olsc::FindElement(&targetCf, cfList, cfCount, [&chunkId](olsc::srv::ComponentFileInfo i) {
                return i.clientArgument.chunkId == chunkId;
            });
            EXPECT_TRUE(targetCf);

            NNT_OLSC_RESULT_DO(olsc::srv::transfer::RequestImportComponentFile(
                chunkImporter.get(), targetCf->url, targetCf->saveDataChunkSize, curl, workBuffer, workBufferSize, nullptr));
        }
    }
    NNT_OLSC_RESULT_DO(importer.get()->FinalizeImport());

    NNT_OLSC_RESULT_DO(olsc::srv::transfer::RequestFinishDownloadSaveDataArchive(sda.id, nsaIdToken, curl, workBuffer, workBufferSize, nullptr));

    NN_RESULT_SUCCESS;
}

TEST(OlscTransferUlAndDl, Simple)
{
    auto workBuffer = nnt::fs::util::AllocateBuffer(WorkBufferSize);
    auto uid = nnt::olsc::GetFirstUserId();

    // NSA ID Token 取得
    nn::olsc::srv::NsaIdToken nsaIdToken;

#if defined( NN_BUILD_CONFIG_OS_WIN )
    memset(&nsaIdToken, 0x00, sizeof(olsc::srv::NsaIdToken));
#else
    size_t nsaIdTokenSize;
    nn::util::Cancelable c;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(srv::util::GetNsaIdToken(&nsaIdTokenSize, nsaIdToken.data, sizeof(nsaIdToken.data), uid, c));
    NN_LOG("NsaIdToken: %s\n", nsaIdToken.data);
#endif
    ApplicationId appId = { nnt::olsc::BaseApplicationIdValue };
    nn::util::optional<fs::SaveDataInfo> infoSrc;

    fs::SaveDataSpaceId spaceId = fs::SaveDataSpaceId::User;

    // 同じアプリのセーブが残ってたら削除
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::DeleteTestSaveData(uid, appId, spaceId));

    // サーバーにある既存の Sda を削除
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::DeleteAllSaveDataArchiveFromServer(nsaIdToken, uid, workBuffer.get(), WorkBufferSize));

    // 新しいセーブデータを作成
    ASSERT_TRUE(nnt::olsc::CreateTestSaveData(OriginalFileName, uid, appId, SaveDataSize, SaveDataJournalSize, workBuffer.get(), WorkBufferSize).IsSuccess());
    nnt::fs::util::FindSaveData(&infoSrc, spaceId, [&](fs::SaveDataInfo i) {
        return i.saveDataUserId._data[0] == uid._data[0] && i.saveDataUserId._data[1] == uid._data[1] && i.applicationId.value == appId.value;
    });
    EXPECT_TRUE(infoSrc);

    // ハッシュを計算
    nnt::fs::util::Hash hashSrc;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::CalculateSaveDataHash(&hashSrc, uid, appId, workBuffer.get(), WorkBufferSize));

    // 作ったセーブデータをアップロード
    nn::olsc::srv::SaveDataArchiveInfo sda;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(TestExportSaveData(&sda, nsaIdToken, uid, appId, *infoSrc, workBuffer.get(), WorkBufferSize));

    // ローカルセーブ削除
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(spaceId, infoSrc->saveDataId));

    // 削除したのでマウントに失敗することを確認
    {
        ncm::ApplicationId ncmAppId;
        ncmAppId.value = appId.value;
        fs::UserId userId;
        userId._data[0] = uid._data[0];
        userId._data[1] = uid._data[1];
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::MountSaveData("save", ncmAppId, userId));
    }

    // アップロードしたセーブデータをダウンロード
    NNT_OLSC_EXPECT_RESULT_SUCCESS(TestImportSaveData(sda, nsaIdToken, uid, appId, spaceId,  workBuffer.get(), WorkBufferSize));

    // ダウンロードしたセーブのハッシュ計算
    nnt::fs::util::Hash hashDst;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::CalculateSaveDataHash(&hashDst, uid, appId, workBuffer.get(), WorkBufferSize));

    // ハッシュの一致確認
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(&hashSrc, &hashDst, sizeof(hashDst));

    // ローカルセーブ削除(後始末)
    nnt::fs::util::FindSaveData(&infoSrc, spaceId, [&](fs::SaveDataInfo i) {
        return i.saveDataUserId._data[0] == uid._data[0] && i.saveDataUserId._data[1] == uid._data[1] && i.applicationId.value == appId.value;
    });
    EXPECT_TRUE(infoSrc);
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(spaceId, infoSrc->saveDataId));
}
TEST(OlscTransferUlAndDl, DifferentialUl)
{
    auto workBuffer = nnt::fs::util::AllocateBuffer(WorkBufferSize);
    auto uid = nnt::olsc::GetFirstUserId();

    // NSA ID Token 取得
    nn::olsc::srv::NsaIdToken nsaIdToken;

#if defined( NN_BUILD_CONFIG_OS_WIN )
    memset(&nsaIdToken, 0x00, sizeof(olsc::srv::NsaIdToken));
#else
    size_t nsaIdTokenSize;
    nn::util::Cancelable c;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(srv::util::GetNsaIdToken(&nsaIdTokenSize, nsaIdToken.data, sizeof(nsaIdToken.data), uid, c));
    NN_LOG("NsaIdToken: %s\n", nsaIdToken.data);
#endif

    ApplicationId appId = { nnt::olsc::BaseApplicationIdValue };

    nn::util::optional<fs::SaveDataInfo> infoSrc;

    fs::SaveDataSpaceId spaceId = fs::SaveDataSpaceId::User;

    // サーバーにある既存の Sda を削除
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::DeleteAllSaveDataArchiveFromServer(nsaIdToken, uid, workBuffer.get(), WorkBufferSize));

    // 新しいセーブデータを作成
    ASSERT_TRUE(nnt::olsc::CreateTestSaveData(OriginalFileName, uid, appId, SaveDataSize, SaveDataJournalSize, workBuffer.get(), WorkBufferSize).IsSuccess());
    nnt::fs::util::FindSaveData(&infoSrc, spaceId, [&](fs::SaveDataInfo i) {
        return i.saveDataUserId._data[0] == uid._data[0] && i.saveDataUserId._data[1] == uid._data[1] && i.applicationId.value == appId.value;
    });
    EXPECT_TRUE(infoSrc);

    // 作ったセーブデータをアップロード
    nn::olsc::srv::SaveDataArchiveInfo sda;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(TestExportSaveData(&sda, nsaIdToken, uid, appId, *infoSrc, workBuffer.get(), WorkBufferSize));

    // 作ったセーブデータを更新
    ASSERT_TRUE(nnt::olsc::CreateTestSaveData(DiffFileName, uid, appId, SaveDataSize, SaveDataJournalSize, workBuffer.get(), WorkBufferSize).IsSuccess());

    nnt::fs::util::FindSaveData(&infoSrc, spaceId, [&](fs::SaveDataInfo i) {
        return i.saveDataUserId._data[0] == uid._data[0] && i.saveDataUserId._data[1] == uid._data[1] && i.applicationId.value == appId.value;
    });
    EXPECT_TRUE(infoSrc);

    // 更新後のハッシュを計算
    nnt::fs::util::Hash hashSrc;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::CalculateSaveDataHash(&hashSrc, uid, appId, workBuffer.get(), WorkBufferSize));

    // 更新したセーブデータを差分アップロード
    NNT_OLSC_EXPECT_RESULT_SUCCESS(TestDiffExportSaveData(&sda, nsaIdToken, uid, appId, *infoSrc, workBuffer.get(), WorkBufferSize));

    // ローカルセーブ削除
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(spaceId, infoSrc->saveDataId));

    // 削除したのでマウントに失敗することを確認
    {
        ncm::ApplicationId ncmAppId;
        ncmAppId.value = appId.value;
        fs::UserId userId;
        userId._data[0] = uid._data[0];
        userId._data[1] = uid._data[1];
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::MountSaveData("save", ncmAppId, userId));
    }

    // アップロードしたセーブデータをダウンロード
    NNT_OLSC_EXPECT_RESULT_SUCCESS(TestImportSaveData(sda, nsaIdToken, uid, appId, spaceId,  workBuffer.get(), WorkBufferSize));

    // ダウンロードしたセーブのハッシュ計算
    nnt::fs::util::Hash hashDst;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::CalculateSaveDataHash(&hashDst, uid, appId, workBuffer.get(), WorkBufferSize));

    // ハッシュの一致確認
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(&hashSrc, &hashDst, sizeof(hashDst));

    // ローカルセーブ削除(後始末)
    nnt::fs::util::FindSaveData(&infoSrc, spaceId, [&](fs::SaveDataInfo i) {
        return i.saveDataUserId._data[0] == uid._data[0] && i.saveDataUserId._data[1] == uid._data[1] && i.applicationId.value == appId.value;
    });
    EXPECT_TRUE(infoSrc);
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(spaceId, infoSrc->saveDataId));
}

TEST(OlscTransferUlAndDl, DifferentialDl)
{
    auto workBuffer = nnt::fs::util::AllocateBuffer(WorkBufferSize);
    auto uid = nnt::olsc::GetFirstUserId();

    // NSA ID Token 取得
    nn::olsc::srv::NsaIdToken nsaIdToken;

#if defined( NN_BUILD_CONFIG_OS_WIN )
    memset(&nsaIdToken, 0x00, sizeof(olsc::srv::NsaIdToken));
#else
    size_t nsaIdTokenSize;
    nn::util::Cancelable c;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(srv::util::GetNsaIdToken(&nsaIdTokenSize, nsaIdToken.data, sizeof(nsaIdToken.data), uid, c));
    NN_LOG("NsaIdToken: %s\n", nsaIdToken.data);
#endif
    ApplicationId appId = { nnt::olsc::BaseApplicationIdValue };

    nn::util::optional<fs::SaveDataInfo> infoSrc;

    fs::SaveDataSpaceId spaceId = fs::SaveDataSpaceId::User;

    // サーバーにある既存の Sda を削除
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::DeleteAllSaveDataArchiveFromServer(nsaIdToken, uid, workBuffer.get(), WorkBufferSize));

    // 新しいセーブデータを作成
    ASSERT_TRUE(nnt::olsc::CreateTestSaveData(OriginalFileName, uid, appId, SaveDataSize, SaveDataJournalSize, workBuffer.get(), WorkBufferSize).IsSuccess());
    nnt::fs::util::FindSaveData(&infoSrc, spaceId, [&](fs::SaveDataInfo i) {
        return i.saveDataUserId._data[0] == uid._data[0] && i.saveDataUserId._data[1] == uid._data[1] && i.applicationId.value == appId.value;
    });
    EXPECT_TRUE(infoSrc);

    // セーブのハッシュ計算
    nnt::fs::util::Hash hashSrc;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::CalculateSaveDataHash(&hashSrc, uid, appId, workBuffer.get(), WorkBufferSize));

    // 作ったセーブデータをアップロード
    nn::olsc::srv::SaveDataArchiveInfo sda;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(TestExportSaveData(&sda, nsaIdToken, uid, appId, *infoSrc, workBuffer.get(), WorkBufferSize));

    // 作ったセーブデータを更新
    ASSERT_TRUE(nnt::olsc::CreateTestSaveData(DiffFileName, uid, appId, SaveDataSize, SaveDataJournalSize, workBuffer.get(), WorkBufferSize).IsSuccess());

    // 更新したセーブのハッシュ計算
    nnt::fs::util::Hash hashTmp;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::CalculateSaveDataHash(&hashTmp, uid, appId, workBuffer.get(), WorkBufferSize));

    // ハッシュの不一致確認
    EXPECT_TRUE(memcmp(&hashTmp, &hashSrc, sizeof(hashSrc)) != 0);

    // アップロードしたセーブデータをダウンロード
    NNT_OLSC_EXPECT_RESULT_SUCCESS(TestImportSaveData(sda, nsaIdToken, uid, appId, spaceId,  workBuffer.get(), WorkBufferSize));

    // ダウンロードしたセーブのハッシュ計算
    nnt::fs::util::Hash hashDst;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::CalculateSaveDataHash(&hashDst, uid, appId, workBuffer.get(), WorkBufferSize));

    // ハッシュの一致確認
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(&hashSrc, &hashDst, sizeof(hashDst));

    // ローカルセーブ削除(後始末)
    nnt::fs::util::FindSaveData(&infoSrc, spaceId, [&](fs::SaveDataInfo i) {
        return i.saveDataUserId._data[0] == uid._data[0] && i.saveDataUserId._data[1] == uid._data[1] && i.applicationId.value == appId.value;
    });
    EXPECT_TRUE(infoSrc);
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(spaceId, infoSrc->saveDataId));
}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    nnt::olsc::Initialize();

    ::testing::InitGoogleTest(&argc, argv);
    auto ret = RUN_ALL_TESTS();

    nnt::olsc::Finalize();

    nnt::Exit(ret);
}
