﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/base/testBase_Exit.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include <nn/nn_ApplicationId.h>
#include <nn/ncm/ncm_ContentMetaId.h>

#include <nn/account.h>

#include <nn/util/util_Execution.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_Md5Generator.h>

#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/olsc_Util.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabase.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/transfer/olsc_ComponentFileUpload.h>
#include <nn/olsc/srv/transfer/olsc_ComponentFileDownload.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveUpload.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveDownload.h>

#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTransferVersion2.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_PosixTime.h>


#include <nnt/fsUtil/testFs_util.h>
#include <nnt/fsUtil/testFs_util_function.h>
#include "testOlsc_Util.h"
#include "testOlsc_SaveDataUtil.h"

using namespace nn;

// 新規 export, import, 差分 export, import

namespace
{
    const size_t WorkBufferSize      = 4 * 1024 * 1024;
    const size_t SaveDataSize        = 16 * 1024 * 1024;
    const size_t SaveDataJournalSize = 16 * 1024 * 1024;
    const size_t MaxDivisionCount    = 64;

    nn::Result CreateAndMountSaveData(const char* mountName, nn::ncm::ApplicationId appId, fs::UserId userId)
    {
        NN_RESULT_TRY(nn::fs::MountSaveData(mountName, appId, userId))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NNT_OLSC_RESULT_DO(nn::fs::CreateSaveData(appId, userId, appId.value, SaveDataSize, SaveDataJournalSize, 0));
            NNT_OLSC_RESULT_DO(nn::fs::MountSaveData(mountName, appId, userId));
        }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    nn::Result CreateRandomSaveData(nnt::fs::util::Hash* pOut, const account::Uid& uid, const ApplicationId& appId, void* workBuffer, size_t workbufferSize) NN_NOEXCEPT
    {
        ncm::ApplicationId ncmAppId;
        ncmAppId.value = appId.value;
        fs::UserId userId;
        userId._data[0] = uid._data[0];
        userId._data[1] = uid._data[1];

        NNT_OLSC_RESULT_DO(CreateAndMountSaveData("save", ncmAppId, userId));
        NNT_OLSC_RESULT_DO(nnt::fs::util::CreateTestDirectoryTreeRandomly("save:/", 512 * 1024, 8));
        fs::CommitSaveData("save");
        NN_LOG("\n\n");
        NN_LOG("Src: \n");
        nnt::fs::util::ListDirectoryRecursive("save:/");
        NNT_OLSC_RESULT_DO(nnt::fs::util::CalculateDirectoryTreeHash(pOut, "save:/", workBuffer, WorkBufferSize));
        nnt::fs::util::DumpBuffer(pOut, sizeof(nnt::fs::util::Hash));
        nn::fs::Unmount("save");

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief pred に最初にマッチする ComponentFileInfo を取得します。
    */
    template <typename MetaData, typename Pred>
    void FindMetaFile(nn::util::optional<MetaData>* pOutValue, MetaData* metaList, size_t listCount, Pred pred)
    {
        for(int i = 0; i < static_cast<int>(listCount); i++)
        {
            if(pred(metaList[i]))
            {
                *pOutValue = metaList[i];
                return;
            }
        }
        *pOutValue = nn::util::nullopt;
        return;
    }

    // 命名ロジックここに持ちたくないけど仕方なし
    std::string GetObjectFilePath(olsc::srv::SaveDataArchiveId sdaId, olsc::srv::ComponentFileId cfId) NN_NOEXCEPT
    {
        static const std::string Root("C:/Windows/Temp/NxBackup/");
        static const std::string ObjExtension(".s3obj");
        return Root + std::to_string(sdaId) + "/" + std::to_string(cfId) + ObjExtension;
    }

    Result CalculateHashOnHost(char* pHash, size_t hashSize, olsc::srv::SaveDataArchiveId sdaId, olsc::srv::ComponentFileId cfId, void* workBuffer, size_t workBufferSize)
    {
        NNT_OLSC_RESULT_DO(nn::fs::MountHostRoot());
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::UnmountHostRoot();
        };
        nn::crypto::Md5Generator md5;
        md5.Initialize();

        std::string path = GetObjectFilePath(sdaId, cfId);
        fs::FileHandle handle;
        NNT_OLSC_RESULT_DO(fs::OpenFile(&handle, path.c_str(), fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseFile(handle);
        };

        size_t totalSize = 0;
        while(NN_STATIC_CONDITION(true))
        {
            size_t readSize = 0;
            NNT_OLSC_RESULT_DO(fs::ReadFile(&readSize, handle, totalSize, workBuffer, workBufferSize, fs::ReadOption()));
            if(readSize > 0)
            {
                totalSize += readSize;
                md5.Update(workBuffer, readSize);
            }
            else
            {
                break;
            }
        }
        md5.GetHash(pHash, hashSize);
        NN_RESULT_SUCCESS;
    }

    Result TestExportSaveData(account::Uid uid, ApplicationId appId, const fs::SaveDataInfo& infoSrc, nnt::fs::util::Hash hashSrc1, const olsc::srv::NsaIdToken& nsaIdToken, CURL* curl, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        NN_LOG("%s, %d\n", __FUNCTION__, __LINE__);
        fs::SaveDataTransferManagerForCloudBackUp manager;
        std::unique_ptr<fs::ISaveDataDivisionExporter> exporter;

        // TODO : 差分の場合も作る
        // とりあえず新規作成固定

        nn::olsc::srv::SaveDataArchiveInfo sda;
        olsc::srv::SeriesInfo info = {0, 0};
        time::PosixTime time = { 1413263600 };
        NNT_OLSC_RESULT_DO(nn::olsc::srv::transfer::StartSaveDataArchiveUpload(&sda, uid, appId, info, 128, time, nsaIdToken, curl, workBuffer, WorkBufferSize, nullptr));
        DumpSaveDataArchiveInfo(sda);
        NNT_OLSC_RESULT_DO(manager.OpenSaveDataExporter(&exporter, infoSrc.saveDataSpaceId, infoSrc.saveDataId));
        exporter->SetDivisionCount(sda.numOfPartitions);
        std::unique_ptr<fs::ISaveDataChunkIterator> iter;
        NNT_OLSC_RESULT_DO(exporter->OpenSaveDataDiffChunkIterator(&iter));

        // 差分のある ComponentFileInfo について（新規なら全 cf について）ループ
        size_t offsetNumber = 0;
        for (; !iter->IsEnd(); iter->Next())
        {
            auto index = iter->GetId();

            // CF 作成
            olsc::srv::ComponentFileInfo cf;
            olsc::srv::ComponentFileType type = index == 0 ? olsc::srv::ComponentFileType::Meta : olsc::srv::ComponentFileType::Save;
            olsc::srv::ClientArgument clientArgument;
            clientArgument.chunkId = index;
            NNT_OLSC_RESULT_DO(olsc::srv::transfer::CreateComponentFile(&cf, sda.id, clientArgument, type, nsaIdToken, curl, workBuffer, WorkBufferSize, nullptr));

            size_t size;
            olsc::srv::SaveDataChunkDigest digest;
            NNT_OLSC_RESULT_DO(olsc::srv::transfer::ExportComponentFile(&size, &digest, cf.url, exporter.get(), index, nsaIdToken, curl, workBuffer, WorkBufferSize, nullptr));
            NNT_OLSC_RESULT_DO(olsc::srv::transfer::CompleteComponentFile(cf.id, size, digest,  nsaIdToken, curl, workBuffer, WorkBufferSize, nullptr));

            char md5Dst[nn::crypto::Md5Generator::HashSize];
            // Host での Hash を計算
            NNT_OLSC_RESULT_DO(CalculateHashOnHost(md5Dst, sizeof(md5Dst), sda.id, cf.id, workBuffer, WorkBufferSize));

            NNT_FS_UTIL_EXPECT_MEMCMPEQ(digest.data, &md5Dst, sizeof(digest.data));

            offsetNumber++;
        }
        nn::olsc::srv::KeySeed keySeed = {};
        nn::olsc::srv::InitialDataMac mac = {};
        NNT_OLSC_RESULT_DO(nn::olsc::srv::transfer::FinishSaveDataArchiveUpload(sda.id, keySeed, mac, nsaIdToken, curl, workBuffer, WorkBufferSize, nullptr));
        NN_RESULT_SUCCESS;
    }



    // TODO : SCSI-534 の結論次第でインポート方法を変更する必要あり
    // とりあえず「差分データを DL しながらオンザフライに適用する」
    Result TestImportSaveDataCore(fs::ISaveDataDivisionImporter* importer, olsc::srv::ComponentFileInfo* cfList, size_t listCount, const olsc::srv::NsaIdToken& token, CURL* curl, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        // ここからセーブデータを破壊的に拡張・更新、または作成
        int64_t requiredSize;
        NNT_OLSC_RESULT_DO(importer->InitializeImport(&requiredSize));

        // TODO: セーブデータ新規作成・拡張のための空き容量不足のハンドリング
        NN_UNUSED(requiredSize);

        {
            std::unique_ptr<fs::ISaveDataChunkIterator> iter;
            NNT_OLSC_RESULT_DO(importer->OpenSaveDataDiffChunkIterator(&iter));

            // 差分のある sf について（新規なら全 sf について）ループ
            for (; !iter->IsEnd(); iter->Next())
            {
                auto chunkId = iter->GetId();
                std::unique_ptr<fs::ISaveDataChunkImporter> chunkImporter;
                NNT_OLSC_RESULT_DO(importer->OpenSaveDataChunkImporter(&chunkImporter, chunkId));
                nn::util::optional<olsc::srv::ComponentFileInfo> targetCf;
                FindMetaFile(&targetCf, cfList, listCount, [&chunkId](olsc::srv::ComponentFileInfo i) {
                    return i.clientArgument.chunkId == chunkId;
                });
                EXPECT_TRUE(targetCf);

                NNT_OLSC_RESULT_DO(olsc::srv::transfer::RequestImportComponentFile(chunkImporter.get(), targetCf->url, targetCf->saveDataChunkSize, curl, workBuffer, workBufferSize, nullptr));
            }
        }
        NNT_OLSC_RESULT_DO(importer->FinalizeImport());
        NN_RESULT_SUCCESS;
    }

    nn::util::optional<olsc::srv::SaveDataArchiveInfo> FindSaveDataArchive(account::Uid uid, ApplicationId appId, const olsc::srv::NsaIdToken& nsaIdToken, CURL* curl, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        olsc::srv::util::DefaultMountManager mountManager(nnt::olsc::MountInfoForTest, nnt::olsc::MountInfoForTest, nnt::olsc::MountInfoForTest);
        olsc::srv::database::SaveDataArchiveInfoCache sdaInfoCache(nnt::olsc::GetFirstUserId(), mountManager);

        EXPECT_TRUE(olsc::srv::transfer::RequestSaveDataArchiveInfoList(sdaInfoCache, nsaIdToken, curl, workBuffer, workBufferSize, nullptr).IsSuccess());

        nn::util::optional<olsc::srv::SaveDataArchiveInfo> targetSda;
        for(int i = 0; i < sdaInfoCache.GetCount(); i++)
        {
            olsc::srv::SaveDataArchiveInfo sda;
            sdaInfoCache.List(&sda, 1, i);
            if(sda.applicationId == appId && sda.userId == uid)
            {
                targetSda = sda;
                break;
            }
        }

        {
            olsc::srv::SaveDataArchiveInfo sda[2];
            int count = 0;
            EXPECT_TRUE(olsc::srv::transfer::RequestSaveDataArchiveInfoList(&count, sda, 2, nsaIdToken, appId, curl, workBuffer, workBufferSize, nullptr).IsSuccess());
            EXPECT_TRUE(count > 0);
            bool checked = false;
            for(int i = 0; i < count; i++)
            {
                if(sda[i].status == olsc::srv::SaveDataArchiveStatus::Fixed)
                {
                    EXPECT_TRUE(*targetSda == sda[i]);
                    checked = true;
                }
            }
            EXPECT_TRUE(checked);
        }
        {
            nn::util::optional<olsc::srv::SaveDataArchiveInfo> sda;
            EXPECT_TRUE(olsc::srv::transfer::RequestFixedSaveDataArchiveInfo(&sda, nsaIdToken, appId, curl, workBuffer, workBufferSize, nullptr).IsSuccess());
            EXPECT_TRUE(sda);
            EXPECT_TRUE(*targetSda == *sda);
        }

        {
            nn::util::optional<olsc::srv::SaveDataArchiveInfo> sda;
            EXPECT_TRUE(olsc::srv::transfer::RequestSaveDataArchiveInfo(&sda, nsaIdToken, targetSda->id, curl, workBuffer, workBufferSize, nullptr).IsSuccess());
            EXPECT_TRUE(sda);
            EXPECT_TRUE(*targetSda == *sda);
        }

        return targetSda;
    }

    Result TestImportSaveData(nnt::fs::util::Hash* pOutHashDst1, account::Uid uid, ApplicationId appId, fs::SaveDataSpaceId spaceId, nnt::fs::util::Hash hashSrc1, const olsc::srv::NsaIdToken& token, CURL* curl, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        nn::util::optional<olsc::srv::SaveDataArchiveInfo> sda = FindSaveDataArchive(uid, appId, token, curl, workBuffer, workBufferSize);
        // bool isNewImport = sda ? false : true;
        {
            olsc::srv::DumpSaveDataArchiveInfo(*sda);

            fs::SaveDataTransferManagerForCloudBackUp manager;
            fs::SaveDataTransferManagerForCloudBackUp::Challenge challenge;
            NNT_OLSC_RESULT_DO(manager.GetChallenge(&challenge));

            fs::SaveDataTransferManagerForCloudBackUp::KeySeedPackage keySeedPackage;
            NNT_OLSC_RESULT_DO(olsc::srv::transfer::GetKeySeedPackage(
                &keySeedPackage, sda->id, challenge,
                token, curl, workBuffer, WorkBufferSize, nullptr));
            NNT_OLSC_RESULT_DO(manager.SetKeySeedPackage(keySeedPackage));

            int cfNum = 0;
            auto cfListBuffer = nnt::fs::util::AllocateBuffer(sizeof(olsc::srv::ComponentFileInfo) * MaxDivisionCount);
            olsc::srv::ComponentFileInfo* cfList = reinterpret_cast<olsc::srv::ComponentFileInfo*>(cfListBuffer.get());
            NNT_OLSC_RESULT_DO(olsc::srv::transfer::RequestStartDownloadSaveDataArchive(&cfNum, cfList, MaxDivisionCount, sda->id, token, curl, workBuffer, workBufferSize, nullptr));

            fs::InitialDataForCloudBackUp initialData = {};
            size_t initialDataSize = 0;
            nn::util::optional<olsc::srv::ComponentFileInfo> targetCf;
            FindMetaFile(&targetCf, cfList, cfNum, [](olsc::srv::ComponentFileInfo i) {
                return i.clientArgument.chunkId == 0;
            });
            EXPECT_TRUE(targetCf);
            NNT_OLSC_RESULT_DO(olsc::srv::transfer::RequestDownloadComponentFile(&initialDataSize, &initialData, sizeof(fs::InitialDataForCloudBackUp), targetCf->url, targetCf->saveDataChunkSize, curl, nullptr));

            std::unique_ptr<fs::ISaveDataDivisionImporter> importer;
            NNT_OLSC_RESULT_DO(manager.OpenSaveDataImporter(&importer, initialData, uid, spaceId));

            NNT_OLSC_RESULT_DO(TestImportSaveDataCore(importer.get(), cfList, cfNum, token, curl, workBuffer, workBufferSize));
            NNT_OLSC_RESULT_DO(olsc::srv::transfer::RequestFinishDownloadSaveDataArchive(sda->id, token, curl, workBuffer, workBufferSize, nullptr));
        }
        {
            ncm::ApplicationId ncmAppId;
            ncmAppId.value = appId.value;
            fs::UserId userId;
            userId._data[0] = uid._data[0];
            userId._data[1] = uid._data[1];
            NNT_OLSC_RESULT_DO(CreateAndMountSaveData("save", ncmAppId, userId));
            nnt::fs::util::ListDirectoryRecursive("save:/");
            NNT_OLSC_RESULT_DO( nnt::fs::util::CalculateDirectoryTreeHash(pOutHashDst1, "save:/", workBuffer, workBufferSize));
            nn::fs::Unmount("save");
        }
        NN_RESULT_SUCCESS;
    }
}


TEST(UploadToHost, ExportSimple)
{
    account::Uid uid = { { 0, 0x1234567890abcdefull } };
    ApplicationId appId = { 0x0005000c10000003ULL };
    auto workBuffer = nnt::fs::util::AllocateBuffer(WorkBufferSize);

    auto curlHandle = curl_easy_init();
    olsc::srv::NsaIdToken token;
    memset(&token, 0x00, sizeof(olsc::srv::NsaIdToken));

    nnt::fs::util::Hash hashSrc1;
    nnt::fs::util::DeleteAllTestSaveData();
    nn::util::optional<fs::SaveDataInfo> infoSrc;

    fs::SaveDataSpaceId spaceId = fs::SaveDataSpaceId::User;

    nnt::fs::util::FindSaveData(&infoSrc, spaceId, [&](fs::SaveDataInfo i) {
        return i.saveDataUserId._data[0] == uid._data[0] && i.saveDataUserId._data[1] == uid._data[1] && i.applicationId.value == appId.value;
    });
    if(infoSrc)
    {
        NN_LOG("Delete SaveData\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::DeleteSaveData(spaceId, infoSrc->saveDataId));
        infoSrc = nn::util::nullopt;
    }

    ASSERT_TRUE(CreateRandomSaveData(&hashSrc1, uid, appId, workBuffer.get(), WorkBufferSize).IsSuccess());


    nnt::fs::util::FindSaveData(&infoSrc, spaceId, [&](fs::SaveDataInfo i) {
        return i.saveDataUserId._data[0] == uid._data[0] && i.saveDataUserId._data[1] == uid._data[1] && i.applicationId.value == appId.value;
    });
    EXPECT_TRUE(infoSrc);


    // // 新規エクスポート
    ASSERT_TRUE(TestExportSaveData(uid, appId, *infoSrc, hashSrc1, token, curlHandle, workBuffer.get(), WorkBufferSize).IsSuccess());
    ASSERT_TRUE(nn::fs::DeleteSaveData(spaceId, infoSrc->saveDataId).IsSuccess());

    // // 別のセーブ (dst) にインポート
    nnt::fs::util::Hash hashDst1;
    ASSERT_TRUE(TestImportSaveData(&hashDst1, uid, appId, spaceId, hashSrc1, token, curlHandle, workBuffer.get(), WorkBufferSize).IsSuccess());

    NNT_FS_UTIL_EXPECT_MEMCMPEQ(&hashSrc1, &hashDst1, sizeof(hashDst1));
}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    nn::fssystem::InitializeAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();


    nn::time::Initialize();

#if defined( NN_BUILD_CONFIG_OS_WIN )
    account::InitializeForAdministrator();

    nnt::olsc::CleanupUsers();
    nn::account::Uid uid;
    nnt::olsc::CreateUsers(&uid, 1);
#endif

    account::InitializeForSystemService();


    fs::SetEnabledAutoAbort(false);

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
