﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_ApiForPrivate.h>
#include <nn/olsc/olsc_ApiForSystemService.h>
#include <nn/olsc/olsc_RemoteStorageController.h>
#include <nn/olsc/olsc_TransferTaskListController.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiForApplications.h>

using namespace nn;
using namespace nn::olsc;

namespace nn { namespace olsc {
    void DeleteAllTransferTask() NN_NOEXCEPT;
}}

namespace {
    class Stopwatch
    {
    public:
        Stopwatch(bool enable, const char* fmt, ...) : m_Enable(enable)
        {
            std::va_list vlist;
            va_start(vlist, fmt);
            util::VSNPrintf(m_Title, sizeof(m_Title), fmt, vlist);
            va_end(vlist);

            m_Begin = os::GetSystemTick();
        }
        ~Stopwatch()
        {
            if (m_Enable)
            {
                auto end = os::GetSystemTick();
                NN_LOG("%s: %lld ms\n", m_Title, (end - m_Begin).ToTimeSpan().GetMilliSeconds());
            }
        }
    private:
        char m_Title[128];
        bool m_Enable;
        os::Tick m_Begin;
    };

    class OlscTransferTaskListControllerTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            account::InitializeForSystemService();
            DeleteAllTransferTask();
        }

        static void TearDownTestCase()
        {
        }
    private:

    };

    account::Uid GetFirstUserId() NN_NOEXCEPT
    {
        int userCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetUserCount(&userCount));
        NN_ABORT_UNLESS(userCount > 0);

        account::Uid uid;
        int listCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&listCount, &uid, 1));
        return uid;
    }
}

TEST_F(OlscTransferTaskListControllerTest, Basic)
{
    nn::olsc::TransferTaskListController tc;
    nn::olsc::OpenTransferTaskListController(&tc);

    EXPECT_EQ(0, tc.GetTransferTaskCount());

    auto uid = GetFirstUserId();
    nn::olsc::RemoteStorageController rc;
    nn::olsc::OpenRemoteStorageController(&rc, uid);

    {
        auto stopper = tc.StopTransferTaskExecution();

        const int MaxTaskCount = 256;
        for (int i = 0; i < MaxTaskCount; ++i)
        {
            TransferTaskId tid;
            {
                Stopwatch sw(true, "RegisterUploadTask: %d\n", i);
                NNT_EXPECT_RESULT_SUCCESS(rc.RegisterUploadSaveDataTransferTask(&tid, { static_cast<Bit64>(i) }));
            }
        }
        EXPECT_EQ(MaxTaskCount, tc.GetTransferTaskCount());
    }

    os::TimerEvent timeout(os::EventClearMode_ManualClear);
    timeout.StartOneShot(TimeSpan::FromSeconds(360));
    int taskCount = -1;
    NN_LOG("Waiting task execution.\n");
    while (NN_STATIC_CONDITION(true))
    {
        taskCount = tc.GetTransferTaskCount();
        NN_LOG("Rest task count: %d\n", taskCount);
        if (taskCount == 0 || timeout.TryWait())
        {
            break;
        }
        os::SleepThread(TimeSpan::FromMilliSeconds(500));
    }
    EXPECT_EQ(0, taskCount);
}
