﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_ApiForPrivate.h>
#include <nn/olsc/olsc_ApiForSystemService.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_TransferTaskListController.h>
#include <nn/olsc/olsc_TransferTaskTypes.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

#include <random>
#include "testOlsc_Stopwatch.h"

//#define NN_TEST_OLSC_ERROR_HISTORY_DATABASE_MAX_ENTRY

using namespace nn;

namespace {

    class OlscTransferTaskErrorInfo : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(time::Initialize());
            account::InitializeForSystemService();
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
            olsc::DeleteAllTransferTaskErrorInfo();
        }

        static void TearDownTestCase()
        {
        }
    private:

    };

    Result GetUid(account::Uid* out, int index)
    {
        account::Uid uids[account::UserCountMax];

        int listCount;
        NN_RESULT_DO(account::ListAllUsers(&listCount, uids, account::UserCountMax));
        NN_ABORT_UNLESS(index < listCount);

        *out = uids[index];

        NN_RESULT_SUCCESS;
    }

    void CreateAppIdList(ApplicationId out[], int count)
    {
        for (int i = 0; i < count; ++i)
        {
            out[i] = { static_cast<Bit64>(i) };
        }

        std::random_device seed_gen;
        std::mt19937 engine(seed_gen());
        std::shuffle(&out[0], &out[count], engine);
    }
}

TEST_F(OlscTransferTaskErrorInfo, Basic)
{
    account::Uid uid;
    NNT_EXPECT_RESULT_SUCCESS(GetUid(&uid, 0));


#if defined(NN_TEST_OLSC_ERROR_HISTORY_DATABASE_MAX_ENTRY)
    const int MaxAppCount = olsc::srv::MaxApplicationCount * account::UserCountMax;
#else
    const int MaxAppCount = olsc::srv::MaxApplicationCount / 4;
#endif

    auto tc = olsc::OpenTransferTaskListController();
    // テスト中に転送タスクを動かさない
    auto stopper = tc.StopTransferTaskExecution();

    std::unique_ptr<ApplicationId[]> appIdList(new ApplicationId[MaxAppCount]);
    CreateAppIdList(appIdList.get(), MaxAppCount);

    // デバッグ API を使って登録
    {
        NN_LOG("Importing test data.\n");
        nnt::olsc::Stopwatch s(true, "Imported");
        for (int i = 0; i < MaxAppCount; ++i)
        {
            bool enableLog = (i % 64 == 0);
            auto& appId = appIdList[i];
            {
                nnt::olsc::Stopwatch s2(enableLog, "Register[%d]", i);
                olsc::TransferTaskErrorInfo errorInfo = {
                    uid,
                    appId,
                    olsc::TransferTaskKind::Download,
                    false,
                    { 0 },
                    fs::ResultPathNotFound(),
                };
                olsc::RegisterTransferTaskErrorInfo(errorInfo);
            }
        }
    }

    // 取得
    auto errorInfoCount = olsc::GetTransferTaskErrorInfoCount();
    EXPECT_EQ(MaxAppCount, errorInfoCount);
    std::unique_ptr<olsc::TransferTaskErrorInfo[]> readBuffer(new olsc::TransferTaskErrorInfo[errorInfoCount]);
    {
        int listedCount = 0;
        {
            nnt::olsc::Stopwatch s(true, "Get");

            listedCount = olsc::ListTransferTaskErrorInfo(readBuffer.get(), errorInfoCount, 0);
        }

        EXPECT_EQ(errorInfoCount, listedCount);
        for (int i = 0; i < errorInfoCount; ++i)
        {
            auto& ei = readBuffer[i];

            EXPECT_EQ(static_cast<Bit64>(i), ei.applicationId.value);
            NNT_EXPECT_RESULT_FAILURE(fs::ResultPathNotFound, ei.lastResult);
        }
    }

    // 削除
    {
        NN_LOG("Removing test data.\n");
        nnt::olsc::Stopwatch s(true, "Removed");
        for (int i = 0; i < errorInfoCount; ++i)
        {
            auto& ei = readBuffer[i];

            olsc::RemoveTransferTaskErrorInfo(ei.uid, ei.applicationId);
        }
        auto listedCount = olsc::ListTransferTaskErrorInfo(readBuffer.get(), errorInfoCount, 0);

        EXPECT_EQ(0, listedCount);
    }
}
