﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_TransferTaskContextDatabase.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabaseManager.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabase.h>
#include <nn/olsc/srv/database/olsc_ErrorHistoryDatabase.h>
#include <nn/olsc/srv/olsc_SystemEventManager.h>
#include <nn/olsc/srv/olsc_TransferTask.h>
#include <nn/olsc/srv/olsc_TransferTaskAgent.h>
#include <nn/olsc/srv/olsc_TransferTaskFactory.h>
#include <nn/olsc/srv/util/olsc_Account.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <vector>

#include <nnt/fsUtil/testFs_util.h>
#include <nnt/fsUtil/testFs_util_function.h>

#include "testOlsc_SaveDataUtil.h"
#include "testOlsc_Util.h"

#define NN_TEST_OLSC_UPLOAD_TASK
#define NN_TEST_OLSC_DOWNLOAD_TASK

using namespace nn;
using namespace nn::olsc;
using namespace nn::olsc::srv;
using namespace nn::olsc::srv::database;

namespace {
    const fs::SystemSaveDataId SystemSaveDataId = 0x8000000000004000;
    const int64_t   SystemSaveDataSize = 4 * 1024 * 1024;
    const int64_t   SystemSaveJournalSize = 4 * 1024 * 1024;
    const uint32_t  SystemSaveDataFlags = 0;

    olsc::srv::util::MountInfo TestDeviceSave = {
        SystemSaveDataId,
        SystemSaveDataSize,
        SystemSaveJournalSize,
        SystemSaveDataFlags
    };

    olsc::srv::util::MountInfo TestUserSettingSave = {
        SystemSaveDataId + 1,
        SystemSaveDataSize,
        SystemSaveJournalSize,
        SystemSaveDataFlags
    };

    olsc::srv::util::MountInfo TestUserSeriesInfoSave = {
        SystemSaveDataId + 2,
        SystemSaveDataSize,
        SystemSaveJournalSize,
        SystemSaveDataFlags
    };

    class OlscTransferTaskTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData());
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            nnt::olsc::Initialize();
        }

        static void TearDownTestCase()
        {
            NNT_ASSERT_RESULT_SUCCESS(time::Finalize());
        }
    private:

        static Result InitializeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
            {
            }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }
    };

    struct ServiceResourceForTest
    {
        ServiceResourceForTest()
            : agentExecutionThreadResource(NN_SYSTEM_THREAD_NAME(olsc, TransferTask), NN_SYSTEM_THREAD_PRIORITY(olsc, TransferTask))
        {}
        TransferTaskBase::TransferTaskExecutionResource taskExecutionResource;
        TransferTaskAgent::ThreadResource agentExecutionThreadResource;
    };

    nn::Result CreateAndMountSaveData(const char* mountName, nn::ncm::ApplicationId appId, fs::UserId userId, int64_t saveDataSize, int64_t journalSize)
    {
        NN_RESULT_TRY(nn::fs::MountSaveData(mountName, appId, userId))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NNT_OLSC_RESULT_DO(nn::fs::CreateSaveData(appId, userId, appId.value, saveDataSize, journalSize, 0));
            NNT_OLSC_RESULT_DO(nn::fs::MountSaveData(mountName, appId, userId));
        }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    nn::Result CreateRandomSaveData(nnt::fs::util::Hash* pOut, const account::Uid& uid, const ApplicationId& appId, void* workBuffer, size_t workbufferSize, size_t saveDataSize) NN_NOEXCEPT
    {
        ncm::ApplicationId ncmAppId;
        ncmAppId.value = appId.value;
        fs::UserId userId;
        userId._data[0] = uid._data[0];
        userId._data[1] = uid._data[1];

        NNT_OLSC_RESULT_DO(CreateAndMountSaveData("save", ncmAppId, userId, saveDataSize, saveDataSize));
        NNT_OLSC_RESULT_DO(nnt::fs::util::CreateTestDirectoryTreeRandomly("save:/", 512 * 1024, 8));
        fs::CommitSaveData("save");
        NN_LOG("\n\n");
        NN_LOG("Src: \n");
        nnt::fs::util::ListDirectoryRecursive("save:/");
        NNT_OLSC_RESULT_DO(nnt::fs::util::CalculateDirectoryTreeHash(pOut, "save:/", workBuffer, workbufferSize));
        nnt::fs::util::DumpBuffer(pOut, sizeof(nnt::fs::util::Hash));
        nn::fs::Unmount("save");

        NN_RESULT_SUCCESS;
    }

    using CompleteCondition = std::function<bool()>;
    bool WaitTaskEvent(os::SystemEventType* taskEvent, const CompleteCondition& condition, int timeout) NN_NOEXCEPT
    {
        const int FinishedIndex = 0;
        const int TimeoutIndex = 1;
        os::TimerEvent timeoutEvent(os::EventClearMode::EventClearMode_ManualClear);
        timeoutEvent.StartOneShot(TimeSpan::FromSeconds(timeout));
        while (NN_STATIC_CONDITION(true))
        {
            auto signaled = os::WaitAny(taskEvent, timeoutEvent.GetBase());
            if (signaled == FinishedIndex)
            {
                os::ClearSystemEvent(taskEvent);
                if (condition())
                {
                    return true;
                }
            }
            else if (signaled == TimeoutIndex)
            {
                NN_LOG("Timeout\n");
                return false;
            }
        }
        NN_ABORT("Not come here.\n");
    }

    const int WorkBufferLen = 1 * 1024 * 1024;
    char g_WorkBuffer[WorkBufferLen];
    nnt::fs::util::Hash g_Hash;

    nn::Result GetSaveDataArchive(nn::util::optional<SaveDataArchiveInfo>* pOutValue, account::Uid &uid, ApplicationId& appId) NN_NOEXCEPT
    {
        auto curl = curl_easy_init();
        nn::olsc::srv::NsaIdToken nsaIdToken;
        size_t nsaIdTokenSize;
        nn::util::Cancelable c;
        NNT_OLSC_RESULT_DO(olsc::srv::util::GetNsaIdToken(&nsaIdTokenSize, nsaIdToken.data, sizeof(nsaIdToken.data), uid, c));
        NNT_OLSC_RESULT_DO(olsc::srv::transfer::RequestFixedSaveDataArchiveInfo(pOutValue, nsaIdToken, appId, curl, g_WorkBuffer, WorkBufferLen, nullptr));
        NN_RESULT_SUCCESS;
    }
}
#if defined(NN_TEST_OLSC_UPLOAD_TASK)
TEST_F(OlscTransferTaskTest, UploadTask)
{
    srv::util::DefaultMountManager mountManager(TestDeviceSave, TestUserSettingSave, TestUserSeriesInfoSave);

    static ServiceResourceForTest resource;

    TransferTaskDatabaseManager ttdm(mountManager);
    TransferTaskFactory factory(ttdm);
    TransferTaskCompleteEventManager completeEventManager;
    TransferTaskStartEventManager startEventManager;
    SeriesInfoDatabaseManager seriesInfoDatabaseManager(mountManager);
    SaveDataArchiveInfoCacheManager sdaInfoCacheManager(mountManager);
    ErrorHistoryDatabase errorHistoryDatabase(mountManager);
    TransferTaskAgent agent(ttdm, resource.agentExecutionThreadResource, factory, startEventManager, completeEventManager, seriesInfoDatabaseManager, sdaInfoCacheManager, errorHistoryDatabase, resource.taskExecutionResource);

    account::Uid uid = nnt::olsc::GetFirstUserId();
    seriesInfoDatabaseManager.Acquire(uid)->Clear();
    ApplicationId appId = { nnt::olsc::BaseApplicationIdValue };

    // 同じアプリのセーブが残ってたら削除
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::DeleteTestSaveData(uid, appId, fs::SaveDataSpaceId::User));

    NNT_OLSC_EXPECT_RESULT_SUCCESS(CreateRandomSaveData(&g_Hash, uid, appId, g_WorkBuffer, sizeof(g_WorkBuffer), 4 * 1024 * 1024));

    // サーバーにある既存の Sda を削除
    {
        nn::olsc::srv::NsaIdToken nsaIdToken;
        size_t nsaIdTokenSize;
        nn::util::Cancelable c;
        NNT_OLSC_EXPECT_RESULT_SUCCESS(olsc::srv::util::GetNsaIdToken(&nsaIdTokenSize, nsaIdToken.data, sizeof(nsaIdToken.data), uid, c));
        NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::DeleteAllSaveDataArchiveFromServer(nsaIdToken, uid, g_WorkBuffer, sizeof(g_WorkBuffer)));
    }

    os::SystemEventType startedEvent;
    NNT_ASSERT_RESULT_SUCCESS(startEventManager.AcquireSystemEvent(&startedEvent));

    agent.Start();

    TransferTaskConfig config = {};
    config.kind = TransferTaskKind::Upload;

    TransferTaskId tid;
    NNT_EXPECT_RESULT_SUCCESS(ttdm.RegisterTransferTask(&tid, uid, appId, config, TransferTaskRank::Basic));

    EXPECT_TRUE(WaitTaskEvent(&startedEvent, [&]() -> bool {
        return true;
    }, 5));


    // とりあえず 20 秒待ってキャンセル
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(20));
        auto taskLocker = agent.LockTask();
        auto currentTask = agent.GetCurrentTask();
        if(currentTask)
        {
            currentTask->Cancel();
        }
    }

    os::SystemEventType finishedEvent;
    NNT_ASSERT_RESULT_SUCCESS(completeEventManager.AcquireSystemEvent(&finishedEvent));
    WaitTaskEvent(&finishedEvent, []() -> bool {
        return true;
    }, 30);

    nn::os::ClearSystemEvent(&finishedEvent);

    WaitTaskEvent(&finishedEvent, []() -> bool {
        return true;
    }, 30);

    agent.Stop();
}
#endif
#if defined(NN_TEST_OLSC_DOWNLOAD_TASK)
TEST_F(OlscTransferTaskTest, DownloadTask)
{
    srv::util::DefaultMountManager mountManager(TestDeviceSave, TestUserSettingSave, TestUserSeriesInfoSave);

    static ServiceResourceForTest resource;

    TransferTaskDatabaseManager ttdm(mountManager);
    TransferTaskFactory factory(ttdm);
    TransferTaskCompleteEventManager completeEventManager;
    TransferTaskCompleteEventManager startEventManager;
    SeriesInfoDatabaseManager seriesInfoDatabaseManager(mountManager);
    SaveDataArchiveInfoCacheManager sdaInfoCacheManager(mountManager);
    ErrorHistoryDatabase errorHistoryDatabase(mountManager);
    TransferTaskAgent agent(ttdm, resource.agentExecutionThreadResource, factory, startEventManager, completeEventManager, seriesInfoDatabaseManager, sdaInfoCacheManager, errorHistoryDatabase, resource.taskExecutionResource);

    account::Uid uid = nnt::olsc::GetFirstUserId();
    seriesInfoDatabaseManager.Acquire(uid)->Clear();
    ApplicationId appId = { nnt::olsc::BaseApplicationIdValue };

    // 同じアプリのセーブが残ってたら削除
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::DeleteTestSaveData(uid, appId, fs::SaveDataSpaceId::User));

    os::SystemEventType startedEvent;
    NNT_ASSERT_RESULT_SUCCESS(startEventManager.AcquireSystemEvent(&startedEvent));

    agent.Start();

    TransferTaskConfig config = {};
    config.kind = TransferTaskKind::Download;

    nn::util::optional<SaveDataArchiveInfo> sdaInfo;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(GetSaveDataArchive(&sdaInfo, uid, appId));
    EXPECT_TRUE(sdaInfo);

    config.cachedSi = sdaInfo->seriesInfo;

    TransferTaskId tid;
    NNT_EXPECT_RESULT_SUCCESS(ttdm.RegisterTransferTask(&tid, uid, appId, config, TransferTaskRank::Basic));

    EXPECT_TRUE(WaitTaskEvent(&startedEvent, [&]() -> bool {
        return true;
    }, 5));

    os::SystemEventType finishedEvent;
    NNT_ASSERT_RESULT_SUCCESS(completeEventManager.AcquireSystemEvent(&finishedEvent));
    WaitTaskEvent(&finishedEvent, []() -> bool {
        return true;
    }, 30);

    agent.Stop();

    nnt::fs::util::Hash hashDst;
    NNT_OLSC_EXPECT_RESULT_SUCCESS(nnt::olsc::CalculateSaveDataHash(&hashDst, uid, appId, g_WorkBuffer, WorkBufferLen));
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(&g_Hash, &hashDst, sizeof(nnt::fs::util::Hash));
}
#endif
