﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/sfdl/olsc_IOlscService.sfdl.h>
#include <nn/olsc/srv/database/olsc_TransferTaskContextDatabase.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabaseManager.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabase.h>
#include <nn/olsc/srv/database/olsc_ErrorHistoryDatabase.h>
#include <nn/olsc/srv/olsc_SystemEventManager.h>
#include <nn/olsc/srv/olsc_TransferTask.h>
#include <nn/olsc/srv/olsc_TransferTaskAgent.h>
#include <nn/olsc/srv/olsc_TransferTaskListControllerImpl.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_FormatString.h>

#include "testOlsc_MemoryResource.h"
#include "testOlsc_SaveDataUtil.h"
#include "testOlsc_TransferTaskManagement.h"

#include <vector>

#define NNT_OLSC_TEST_TTLC_FORBID
#define NNT_OLSC_TEST_TTLC_STOP
#define NNT_OLSC_TEST_TTLC_DELETE


using namespace nn;
using namespace nn::olsc;
using namespace nn::olsc::srv;
using namespace nn::olsc::srv::database;

namespace {

    class OlscTransferTaskListControllerImplTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData());
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
            NNT_ASSERT_RESULT_SUCCESS(time::Initialize());

#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::account::InitializeForAdministrator();

            nnt::olsc::CleanupUsers();
            nn::account::Uid uid;
            nnt::olsc::CreateUsers(&uid, 1);
#else
            account::InitializeForSystemService();
#endif
            NNT_ASSERT_RESULT_SUCCESS(nifm::InitializeSystem());
        }

        static void TearDownTestCase()
        {
            NNT_ASSERT_RESULT_SUCCESS(time::Finalize());
        }
    private:

        static Result InitializeSaveData()
        {
            fs::SystemSaveDataId toDelete[] = {
                nnt::olsc::MountInfoForTestDeviceSave.systemSaveDataId,
                nnt::olsc::MountInfoForTestUserSettingSave.systemSaveDataId,
                nnt::olsc::MountInfoForTestUserSeriesInfoSave.systemSaveDataId,
            };
            for (auto& id : toDelete)
            {
                NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, id, {}))
                    NN_RESULT_CATCH(fs::ResultTargetNotFound)
                    {
                    }
                NN_RESULT_END_TRY;
            }

            NN_RESULT_SUCCESS;
        }
    };

    struct TaskParam
    {
        account::Uid        uid;
        ApplicationId       appId;
        TransferTaskKind    kind;
    };

    struct Modules
    {
        Modules(int taskProcessSecond, std::function<Result(const TransferTaskDetailInfo& info)> taskResultArbiter) :
            mountManager(nnt::olsc::MountInfoForTestDeviceSave, nnt::olsc::MountInfoForTestUserSettingSave, nnt::olsc::MountInfoForTestUserSeriesInfoSave),
            ttdm(mountManager),
            seriesInfoDatabaseManager(mountManager),
            sdaInfoCacheManager(mountManager),
            errorHistoryDatabase(mountManager),
            threadResource(
                NN_SYSTEM_THREAD_NAME(olsc, TransferTask),
                NN_SYSTEM_THREAD_PRIORITY(olsc, TransferTask)
            ),
            factory(ttdm, taskDoneList, taskProcessSecond, taskResultArbiter),
            agent(ttdm, threadResource, factory, startEventManager, completeEventManager, seriesInfoDatabaseManager, sdaInfoCacheManager, errorHistoryDatabase, executionResource),
            ttci(nnt::olsc::DefaultMemoryResource::GetInstance(), ttdm, agent, startEventManager, completeEventManager)
        {}
        ~Modules() {}
        srv::util::DefaultMountManager mountManager;
        srv::database::TransferTaskDatabaseManager ttdm;
        TransferTaskCompleteEventManager completeEventManager;
        TransferTaskStartEventManager startEventManager;
        SeriesInfoDatabaseManager seriesInfoDatabaseManager;
        SaveDataArchiveInfoCacheManager sdaInfoCacheManager;
        ErrorHistoryDatabase errorHistoryDatabase;

        TransferTaskAgent::ThreadResource threadResource;
        TransferTaskBase::TransferTaskExecutionResource executionResource;

        std::vector<TransferTaskId> taskDoneList;
        nnt::olsc::TransferTaskFactoryForTest factory;
        TransferTaskAgent agent;
        TransferTaskListControllerImpl ttci;
    };

    std::aligned_storage<sizeof(Modules), 4096>::type g_ModuleStorage;
}

#if defined(NNT_OLSC_TEST_TTLC_FORBID)

TEST_F(OlscTransferTaskListControllerImplTest, StopNextExecution)
{
    olsc::srv::util::DefaultMountManager mountManager(nnt::olsc::MountInfoForTestDeviceSave, nnt::olsc::MountInfoForTestUserSettingSave, nnt::olsc::MountInfoForTestUserSeriesInfoSave);

    auto taskResultArbiter = [](const TransferTaskDetailInfo& info) -> Result
    {
        NN_UNUSED(info);
        NN_RESULT_SUCCESS;
    };

    // 5秒でタスク完了
    auto m = new (&g_ModuleStorage) Modules(5, taskResultArbiter);
    NN_UTIL_SCOPE_EXIT{
        m->~Modules();
    };

    os::SystemEventType finishedEvent;
    NNT_ASSERT_RESULT_SUCCESS(m->completeEventManager.AcquireSystemEvent(&finishedEvent));

    os::SystemEventType startedEvent;
    NNT_ASSERT_RESULT_SUCCESS(m->startEventManager.AcquireSystemEvent(&startedEvent));

    auto uid = nnt::olsc::GetFirstUserId();

    m->agent.Start();
    {
        // 実行禁止中にタスク登録されても動かない
        sf::SharedPointer<srv::IStopperObject> p;
        m->ttci.StopNextTransferTaskExecution(&p);

        os::ClearSystemEvent(&startedEvent);
        os::ClearSystemEvent(&finishedEvent);

        ApplicationId appId = { 1 };
        TransferTaskId tid;
        TransferTaskConfig config;
        config.kind = TransferTaskKind::Upload;
        config.ulInfo.force = false;
        m->ttdm.RegisterTransferTask(&tid, uid, appId, config, TransferTaskRank::Basic);

        EXPECT_FALSE(os::TimedWaitSystemEvent(&startedEvent, TimeSpan::FromSeconds(2)));
    }

    // 禁止解除後には動く
    os::SleepThread(TimeSpan::FromSeconds(1));
    EXPECT_TRUE(os::TryWaitSystemEvent(&startedEvent));
    EXPECT_FALSE(os::TryWaitSystemEvent(&finishedEvent));

    // 動き始めている状態で禁止にしても動き続ける
    {
        sf::SharedPointer<srv::IStopperObject> p;
        m->ttci.StopNextTransferTaskExecution(&p);

        EXPECT_TRUE(os::TimedWaitSystemEvent(&startedEvent, TimeSpan::FromSeconds(3)));
    }
}
#endif

#if defined(NNT_OLSC_TEST_TTLC_STOP)
TEST_F(OlscTransferTaskListControllerImplTest, Suspend)
{
    olsc::srv::util::DefaultMountManager mountManager(nnt::olsc::MountInfoForTestDeviceSave, nnt::olsc::MountInfoForTestUserSettingSave, nnt::olsc::MountInfoForTestUserSeriesInfoSave);

    auto taskResultArbiter = [](const TransferTaskDetailInfo& info) -> Result
    {
        NN_UNUSED(info);
        NN_RESULT_SUCCESS;
    };

    // 5秒でタスク完了
    auto m = new (&g_ModuleStorage) Modules(5, taskResultArbiter);
    NN_UTIL_SCOPE_EXIT{
        m->~Modules();
    };

    os::SystemEventType finishedEvent;
    NNT_ASSERT_RESULT_SUCCESS(m->completeEventManager.AcquireSystemEvent(&finishedEvent));

    os::SystemEventType startedEvent;
    NNT_ASSERT_RESULT_SUCCESS(m->startEventManager.AcquireSystemEvent(&startedEvent));

    auto uid = nnt::olsc::GetFirstUserId();

    m->agent.Start();
    os::ClearSystemEvent(&startedEvent);
    os::ClearSystemEvent(&finishedEvent);

    ApplicationId appId = { 1 };
    TransferTaskId tid;
    TransferTaskConfig config;
    config.kind = TransferTaskKind::Upload;
    config.ulInfo.force = false;
    m->ttdm.RegisterTransferTask(&tid, uid, appId, config, TransferTaskRank::Basic);

    // 動き始めるのを待って stop する
    os::TimedWaitSystemEvent(&startedEvent, TimeSpan::FromSeconds(2));
    os::ClearSystemEvent(&startedEvent);
    {
        // 自動的にリトライが走らないように止めておく
        sf::SharedPointer<IStopperObject> p;
        m->ttci.StopNextTransferTaskExecution(&p);

        m->ttci.SuspendTransferTask(tid);

        // stop はキャンセル扱いなのでエラーが残らない
        EXPECT_TRUE(os::TimedWaitSystemEvent(&finishedEvent, TimeSpan::FromSeconds(2)));
        os::ClearSystemEvent(&finishedEvent);
        auto ei = m->errorHistoryDatabase.GetLastErrorInfo(uid, appId);
        ASSERT_FALSE(ei);
        NNT_EXPECT_RESULT_FAILURE(olsc::ResultCanceled, m->ttci.GetTransferTaskLastResult(tid));
    }

    // stop はキャンセル扱いなのでリトライが走る
    EXPECT_TRUE(os::TimedWaitSystemEvent(&startedEvent, TimeSpan::FromSeconds(2)));
    EXPECT_TRUE(os::TimedWaitSystemEvent(&finishedEvent, TimeSpan::FromSeconds(10)));
    // 完了したら Success
    NNT_EXPECT_RESULT_SUCCESS(m->ttci.GetTransferTaskLastResult(tid));
}
#endif

#if defined(NNT_OLSC_TEST_TTLC_DELETE)
TEST_F(OlscTransferTaskListControllerImplTest, Delete)
{
    olsc::srv::util::DefaultMountManager mountManager(nnt::olsc::MountInfoForTestDeviceSave, nnt::olsc::MountInfoForTestUserSettingSave, nnt::olsc::MountInfoForTestUserSeriesInfoSave);

    auto taskResultArbiter = [](const TransferTaskDetailInfo& info) -> Result
    {
        NN_UNUSED(info);
        NN_RESULT_SUCCESS;
    };

    // 5秒でタスク完了
    auto m = new (&g_ModuleStorage) Modules(10, taskResultArbiter);
    NN_UTIL_SCOPE_EXIT{
        m->~Modules();
    };

    os::SystemEventType finishedEvent;
    NNT_ASSERT_RESULT_SUCCESS(m->completeEventManager.AcquireSystemEvent(&finishedEvent));

    os::SystemEventType startedEvent;
    NNT_ASSERT_RESULT_SUCCESS(m->startEventManager.AcquireSystemEvent(&startedEvent));

    auto uid = nnt::olsc::GetFirstUserId();

    m->agent.Start();
    os::ClearSystemEvent(&startedEvent);
    os::ClearSystemEvent(&finishedEvent);

    ApplicationId appId = { 1 };
    TransferTaskId tid;
    TransferTaskConfig config;
    config.kind = TransferTaskKind::Upload;
    config.ulInfo.force = false;
    m->ttdm.RegisterTransferTask(&tid, uid, appId, config, TransferTaskRank::Basic);

    // 動き始めるのを待って delete する
    os::TimedWaitSystemEvent(&startedEvent, TimeSpan::FromSeconds(2));
    m->ttci.DeleteTransferTask(tid);

    // delete 後はタスクが消えている
    EXPECT_FALSE(m->ttdm.FindDetailInfo([&tid](const TransferTaskDetailInfo& di) -> bool {
        return di.id == tid;
    }));
}
#endif
