﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/fs.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/sfdl/olsc_IOlscService.sfdl.h>
#include <nn/olsc/srv/olsc_SystemEventManager.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/olsc/srv/util/olsc_SeriesPosition.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_FormatString.h>

#include "testOlsc_SaveDataUtil.h"

#include <vector>



using namespace nn;
using namespace nn::olsc;
using namespace nn::olsc::srv;
using namespace nn::olsc::srv::database;

namespace {
    struct Modules
    {
        Modules() :
            mountManager(nnt::olsc::MountInfoForTestDeviceSave, nnt::olsc::MountInfoForTestUserSettingSave, nnt::olsc::MountInfoForTestUserSeriesInfoSave),
            seriesInfoDatabaseManager(mountManager), sdaInfoCacheManager(mountManager)
        {}
        ~Modules() {}
        srv::util::DefaultMountManager mountManager;
        SeriesInfoDatabaseManager seriesInfoDatabaseManager;
        SaveDataArchiveInfoCacheManager sdaInfoCacheManager;
    };

    struct TestParam
    {
        enum class SaveDataExists : Bit8
        {
            SaveDataExists,
            SaveDataNotExists,
        };

        enum class SdaInfoCacheExists : Bit8
        {
            SdaInfoCacheExists,
            SdaInfoCacheNotExists,
        };

        enum class SdaSeriesIdVariation : Bit8
        {
            SdaSeriesId_1,
            SdaSeriesId_2,
        };

        enum class SdaCommitIdVariation : Bit8
        {
            SdaCommitId_SameCurrentSave,
            SdaCommitId_1,
            SdaCommitId_2,
        };

        enum class LastSiSeriesIdVariation : Bit8
        {
            LastSiSeriesId_Invalid,
            LastSiSeriesId_1,
            LastSiSeriesId_2,
        };

        enum class LastSiCommitIdVariation : Bit8
        {
            LastSiCommitId_SameCurrentSave,
            LastSiCommitId_1,
            LastSiCommitId_2,
        };


        SaveDataExists saveDataExists;
        SdaInfoCacheExists sdaInfoCacheExists;

        SdaSeriesIdVariation sdaSeriesId;
        SdaCommitIdVariation sdaCommitId;

        LastSiSeriesIdVariation lastSiSeriesId;
        LastSiCommitIdVariation lastSiCommitId;

        nn::olsc::srv::SeriesPosition expected;
    };

    std::array<Bit8, 1024 * 1024> g_WorkBuffer;
    const size_t SaveDataSize = 256 * 1024;
    const size_t SaveDataJournalSize = 256 * 1024;

    const char* GetSdaSiIdString(TestParam::SdaSeriesIdVariation id)
    {
        switch (id)
        {
        case TestParam::SdaSeriesIdVariation::SdaSeriesId_1:
            return "1";
        case TestParam::SdaSeriesIdVariation::SdaSeriesId_2:
            return "2";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* GetSdaSiCommitIdString(TestParam::SdaCommitIdVariation id)
    {
        switch (id)
        {
        case TestParam::SdaCommitIdVariation::SdaCommitId_1:
            return "1";
        case TestParam::SdaCommitIdVariation::SdaCommitId_2:
            return "2";
        case TestParam::SdaCommitIdVariation::SdaCommitId_SameCurrentSave:
            return "Current save commit id";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* GetLastSiIdString(TestParam::LastSiSeriesIdVariation id)
    {
        switch (id)
        {
        case TestParam::LastSiSeriesIdVariation::LastSiSeriesId_Invalid:
            return "Invalid";
        case TestParam::LastSiSeriesIdVariation::LastSiSeriesId_1:
            return "1";
        case TestParam::LastSiSeriesIdVariation::LastSiSeriesId_2:
            return "2";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* GetLastSiCommitIdString(TestParam::LastSiCommitIdVariation id)
    {
        switch (id)
        {
        case TestParam::LastSiCommitIdVariation::LastSiCommitId_1:
            return "1";
        case TestParam::LastSiCommitIdVariation::LastSiCommitId_2:
            return "2";
        case TestParam::LastSiCommitIdVariation::LastSiCommitId_SameCurrentSave:
            return "Current save commit id";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* GetSeriesPositionString(SeriesPosition sp)
    {
        switch (sp)
        {
        case SeriesPosition::Equal:
            return "Equal";
        case SeriesPosition::LocalDataIsNewer:
            return "LocalDataIsNewer";
        case SeriesPosition::ServerDataIsNewer:
            return "ServerDataIsNewer";
        case SeriesPosition::Discrete:
            return "Discrete";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    nn::util::optional<nn::fs::SaveDataCommitId> FindAndGetSaveDataCommitId(const account::Uid& uid, ApplicationId appId) NN_NOEXCEPT
    {
        auto pSaveDataInfo = srv::util::FindSaveData(uid, appId);
        if (!pSaveDataInfo)
        {
            return nn::util::nullopt;
        }

        fs::SaveDataCommitId commitId;
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetSaveDataCommitId(&commitId, pSaveDataInfo->saveDataSpaceId, pSaveDataInfo->saveDataId));
        return commitId;
    }

    nn::util::optional<fs::SaveDataCommitId> PrepareSaveData(const account::Uid& uid, ApplicationId appId, TestParam::SaveDataExists exists, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        switch (exists)
        {
        case TestParam::SaveDataExists::SaveDataExists:
            NN_ABORT_UNLESS_RESULT_SUCCESS(nnt::olsc::CreateTestSaveData("dummy", uid, appId, SaveDataSize, SaveDataJournalSize, buffer, bufferSize));
            break;
        case TestParam::SaveDataExists::SaveDataNotExists:
            // NOP
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        };
        return FindAndGetSaveDataCommitId(uid, appId);
    }

    nn::util::optional<srv::SaveDataArchiveInfo> PrepareSda(const account::Uid& uid, ApplicationId appId, SaveDataArchiveInfoCacheManager& sdaInfoCacheManager, TestParam::SdaInfoCacheExists exists, TestParam::SdaSeriesIdVariation siId, TestParam::SdaCommitIdVariation siCommitId)
    {
        {
            auto writeMount = sdaInfoCacheManager.AcquireWriteMount(uid);
            sdaInfoCacheManager.Acquire(uid).Get().DeleteAll();
            NN_ABORT_UNLESS_RESULT_SUCCESS(writeMount.Commit());
        }

        if (exists == TestParam::SdaInfoCacheExists::SdaInfoCacheNotExists)
        {
            return nn::util::nullopt;
        }

        fs::SaveDataCommitId commitId;
        switch (siCommitId)
        {
        case TestParam::SdaCommitIdVariation::SdaCommitId_SameCurrentSave:
            {
                auto pCommitId = FindAndGetSaveDataCommitId(uid, appId);
                NN_ABORT_UNLESS(pCommitId);
                commitId = *pCommitId;
            }
            break;

        case TestParam::SdaCommitIdVariation::SdaCommitId_1:
            commitId = 1;
            break;

        case TestParam::SdaCommitIdVariation::SdaCommitId_2:
            commitId = 2;
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }

        Bit64 seriesId;
        switch (siId)
        {
        case TestParam::SdaSeriesIdVariation::SdaSeriesId_1:
            seriesId = 1;
            break;

        case TestParam::SdaSeriesIdVariation::SdaSeriesId_2:
            seriesId = 2;
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }

        SaveDataArchiveInfo sda;
        sda.applicationId = appId;
        sda.userId = uid;
        sda.seriesInfo = {
            seriesId, commitId
        };

        auto sdaWriteMount = sdaInfoCacheManager.AcquireWriteMount(uid);
        auto pSdaInfoCache = sdaInfoCacheManager.Acquire(uid);
        NN_ABORT_UNLESS_RESULT_SUCCESS(pSdaInfoCache->Add(sda));
        NN_ABORT_UNLESS_RESULT_SUCCESS(sdaWriteMount.Commit());

        return pSdaInfoCache->GetSaveDataArchiveInfoByApplicationId(appId);
    }

    SeriesInfo PrepareLastSi(const account::Uid& uid, ApplicationId appId, SeriesInfoDatabaseManager& sidbManager, TestParam::LastSiSeriesIdVariation siId, TestParam::LastSiCommitIdVariation siCommitId)
    {
        fs::SaveDataCommitId commitId;
        switch (siCommitId)
        {
        case TestParam::LastSiCommitIdVariation::LastSiCommitId_SameCurrentSave:
            {
                auto pCommitId = FindAndGetSaveDataCommitId(uid, appId);
                NN_ABORT_UNLESS(pCommitId);
                commitId = *pCommitId;
            }
            break;

        case TestParam::LastSiCommitIdVariation::LastSiCommitId_1:
            commitId = 1;
            break;

        case TestParam::LastSiCommitIdVariation::LastSiCommitId_2:
            commitId = 2;
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }

        Bit64 seriesId;
        switch (siId)
        {
        case TestParam::LastSiSeriesIdVariation::LastSiSeriesId_Invalid:
            seriesId = srv::InvalidSeriesInfoId;
            break;

        case TestParam::LastSiSeriesIdVariation::LastSiSeriesId_1:
            seriesId = 1;
            break;

        case TestParam::LastSiSeriesIdVariation::LastSiSeriesId_2:
            seriesId = 2;
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }

        SeriesInfo si = {
            seriesId, commitId
        };

        auto writeMount = sidbManager.AcquireWriteMount(uid);
        NN_ABORT_UNLESS_RESULT_SUCCESS(sidbManager.Acquire(uid)->Update(appId, si));
        NN_ABORT_UNLESS_RESULT_SUCCESS(writeMount.Commit());
        return si;
    }

    class OlscTransferTaskListControllerImplTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData({}));
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
            NNT_ASSERT_RESULT_SUCCESS(time::Initialize());

#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::account::InitializeForAdministrator();

            nnt::olsc::CleanupUsers();
            nn::account::Uid uid;
            nnt::olsc::CreateUsers(&uid, 1);
#else
            account::InitializeForSystemService();
#endif
            NNT_ASSERT_RESULT_SUCCESS(nifm::InitializeSystem());
        }

        static void TearDownTestCase()
        {
            NNT_ASSERT_RESULT_SUCCESS(time::Finalize());
        }

    protected:
        static Result InitializeSaveData(fs::UserId userId)
        {
            fs::SystemSaveDataId toDelete[] = {
                nnt::olsc::MountInfoForTestDeviceSave.systemSaveDataId,
                nnt::olsc::MountInfoForTestUserSettingSave.systemSaveDataId,
                nnt::olsc::MountInfoForTestUserSeriesInfoSave.systemSaveDataId,
            };
            for (auto& id : toDelete)
            {
                NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, id, userId))
                    NN_RESULT_CATCH(fs::ResultTargetNotFound)
                    {
                    }
                NN_RESULT_END_TRY;
            }

            NN_RESULT_SUCCESS;
        }

        void TestSeriesPosition(TestParam& param)
        {
            auto uid = nnt::olsc::GetFirstUserId();
            ApplicationId appId = { nnt::olsc::BaseApplicationIdValue };

            auto fsUserId = fs::ConvertAccountUidToFsUserId(uid);
            InitializeSaveData(fsUserId);

            nnt::olsc::DeleteTestSaveData(uid, appId, fs::SaveDataSpaceId::User);

            // テスト用セーブデータ, SDA Info Cache の作成
            Modules m;
            auto pCommitId = PrepareSaveData(uid, appId, param.saveDataExists, g_WorkBuffer.data(), g_WorkBuffer.size());
            auto lastSi = PrepareLastSi(uid, appId, m.seriesInfoDatabaseManager, param.lastSiSeriesId, param.lastSiCommitId);
            auto pSdaInfo = PrepareSda(uid, appId, m.sdaInfoCacheManager, param.sdaInfoCacheExists, param.sdaSeriesId, param.sdaCommitId);

            // series position
            auto seriesPosition = srv::util::CalculateSeriesPosition(lastSi, pCommitId, pSdaInfo);

            NN_LOG("----------------------------------------\n");
            NN_LOG("Test summary\n");
            NN_LOG("SaveData: %s\n", param.saveDataExists == TestParam::SaveDataExists::SaveDataExists ? "Exist" : "NOT exist");
            NN_LOG("Sda     : %s\n", param.sdaInfoCacheExists == TestParam::SdaInfoCacheExists::SdaInfoCacheExists ? "Exist" : "NOT exist");
            NN_LOG("\n");
            NN_LOG("Sda\n");
            NN_LOG("  SeriesInfo SeriesId: %s\n", GetSdaSiIdString(param.sdaSeriesId));
            NN_LOG("  SeriesInfo CommitId: %s\n", GetSdaSiCommitIdString(param.sdaCommitId));
            NN_LOG("LastSi\n");
            NN_LOG("  SeriesInfo SeriesId: %s\n", GetLastSiIdString(param.lastSiSeriesId));
            NN_LOG("  SeriesInfo CommitId: %s\n", GetLastSiCommitIdString(param.lastSiCommitId));
            NN_LOG("Expected: %s\n", GetSeriesPositionString(param.expected));
            NN_LOG("Actual  : %s\n", GetSeriesPositionString(seriesPosition));
            NN_LOG("----------------------------------------\n");

            EXPECT_EQ(param.expected, seriesPosition);
        }
    };
}

// -----------------------------------------------------------------------
// Save, Sda どちらも存在しない場合
// -----------------------------------------------------------------------

TEST_F(OlscTransferTaskListControllerImplTest, SaveNotExists_SdaNotExists)
{
    TestParam p = {
        TestParam::SaveDataExists::SaveDataNotExists,
        TestParam::SdaInfoCacheExists::SdaInfoCacheNotExists,
        TestParam::SdaSeriesIdVariation::SdaSeriesId_1,
        TestParam::SdaCommitIdVariation::SdaCommitId_1,
        TestParam::LastSiSeriesIdVariation::LastSiSeriesId_Invalid,
        TestParam::LastSiCommitIdVariation::LastSiCommitId_1,

        SeriesPosition::Discrete,
    };
    TestSeriesPosition(p);
}

TEST_F(OlscTransferTaskListControllerImplTest, SaveNotExists_SdaNotExists_InvalidLastSi)
{
    TestParam p = {
        TestParam::SaveDataExists::SaveDataNotExists,
        TestParam::SdaInfoCacheExists::SdaInfoCacheNotExists,
        TestParam::SdaSeriesIdVariation::SdaSeriesId_1,
        TestParam::SdaCommitIdVariation::SdaCommitId_1,
        TestParam::LastSiSeriesIdVariation::LastSiSeriesId_1,
        TestParam::LastSiCommitIdVariation::LastSiCommitId_1,

        SeriesPosition::Discrete,
    };
    TestSeriesPosition(p);
}

// -----------------------------------------------------------------------
// Save, Sda どちらか一つ存在する場合
// -----------------------------------------------------------------------

TEST_F(OlscTransferTaskListControllerImplTest, SaveExists_SdaNotExists)
{
    TestParam p = {
        TestParam::SaveDataExists::SaveDataExists,
        TestParam::SdaInfoCacheExists::SdaInfoCacheNotExists,
        TestParam::SdaSeriesIdVariation::SdaSeriesId_1,
        TestParam::SdaCommitIdVariation::SdaCommitId_1,
        TestParam::LastSiSeriesIdVariation::LastSiSeriesId_1,
        TestParam::LastSiCommitIdVariation::LastSiCommitId_1,

        SeriesPosition::LocalDataIsNewer,
    };
    TestSeriesPosition(p);
}

TEST_F(OlscTransferTaskListControllerImplTest, SaveNotExists_SdaExists)
{
    TestParam p = {
        TestParam::SaveDataExists::SaveDataNotExists,
        TestParam::SdaInfoCacheExists::SdaInfoCacheExists,
        TestParam::SdaSeriesIdVariation::SdaSeriesId_1,
        TestParam::SdaCommitIdVariation::SdaCommitId_1,
        TestParam::LastSiSeriesIdVariation::LastSiSeriesId_1,
        TestParam::LastSiCommitIdVariation::LastSiCommitId_1,

        SeriesPosition::ServerDataIsNewer,
    };
    TestSeriesPosition(p);
}

TEST_F(OlscTransferTaskListControllerImplTest, SaveNotExists_SdaExists_InvalidLastSi)
{
    TestParam p = {
        TestParam::SaveDataExists::SaveDataNotExists,
        TestParam::SdaInfoCacheExists::SdaInfoCacheExists,
        TestParam::SdaSeriesIdVariation::SdaSeriesId_1,
        TestParam::SdaCommitIdVariation::SdaCommitId_1,
        TestParam::LastSiSeriesIdVariation::LastSiSeriesId_Invalid,
        TestParam::LastSiCommitIdVariation::LastSiCommitId_1,

        SeriesPosition::ServerDataIsNewer,
    };
    TestSeriesPosition(p);
}

// -----------------------------------------------------------------------
// Save, Sda 両方存在する場合
// -----------------------------------------------------------------------

TEST_F(OlscTransferTaskListControllerImplTest, SaveExists_SdaExists_SameSi)
{
    TestParam p = {
        TestParam::SaveDataExists::SaveDataExists,
        TestParam::SdaInfoCacheExists::SdaInfoCacheExists,
        TestParam::SdaSeriesIdVariation::SdaSeriesId_1,
        TestParam::SdaCommitIdVariation::SdaCommitId_SameCurrentSave,
        TestParam::LastSiSeriesIdVariation::LastSiSeriesId_1,
        TestParam::LastSiCommitIdVariation::LastSiCommitId_SameCurrentSave,

        SeriesPosition::Equal,
    };
    TestSeriesPosition(p);
}

TEST_F(OlscTransferTaskListControllerImplTest, SaveExists_SdaExists_LocalUpdated)
{
    TestParam p = {
        TestParam::SaveDataExists::SaveDataExists,
        TestParam::SdaInfoCacheExists::SdaInfoCacheExists,
        TestParam::SdaSeriesIdVariation::SdaSeriesId_1,
        TestParam::SdaCommitIdVariation::SdaCommitId_1,
        TestParam::LastSiSeriesIdVariation::LastSiSeriesId_1,
        TestParam::LastSiCommitIdVariation::LastSiCommitId_1,

        SeriesPosition::LocalDataIsNewer,
    };
    TestSeriesPosition(p);
}

TEST_F(OlscTransferTaskListControllerImplTest, SaveExists_SdaExists_ServerUpdated)
{
    TestParam p = {
        TestParam::SaveDataExists::SaveDataExists,
        TestParam::SdaInfoCacheExists::SdaInfoCacheExists,
        TestParam::SdaSeriesIdVariation::SdaSeriesId_1,
        TestParam::SdaCommitIdVariation::SdaCommitId_1,
        TestParam::LastSiSeriesIdVariation::LastSiSeriesId_1,
        TestParam::LastSiCommitIdVariation::LastSiCommitId_SameCurrentSave,

        SeriesPosition::ServerDataIsNewer,
    };
    TestSeriesPosition(p);
}

TEST_F(OlscTransferTaskListControllerImplTest, SaveExists_SdaExists_ServerUpdated_InvalidLastSi)
{
    TestParam p = {
        TestParam::SaveDataExists::SaveDataExists,
        TestParam::SdaInfoCacheExists::SdaInfoCacheExists,
        TestParam::SdaSeriesIdVariation::SdaSeriesId_1,
        TestParam::SdaCommitIdVariation::SdaCommitId_1,
        TestParam::LastSiSeriesIdVariation::LastSiSeriesId_Invalid,
        TestParam::LastSiCommitIdVariation::LastSiCommitId_SameCurrentSave,

        SeriesPosition::Discrete,
    };
    TestSeriesPosition(p);
}

TEST_F(OlscTransferTaskListControllerImplTest, SaveExists_SdaExists_IdMismatch)
{
    TestParam p = {
        TestParam::SaveDataExists::SaveDataExists,
        TestParam::SdaInfoCacheExists::SdaInfoCacheExists,
        TestParam::SdaSeriesIdVariation::SdaSeriesId_1,
        TestParam::SdaCommitIdVariation::SdaCommitId_SameCurrentSave,
        TestParam::LastSiSeriesIdVariation::LastSiSeriesId_2,
        TestParam::LastSiCommitIdVariation::LastSiCommitId_SameCurrentSave,

        SeriesPosition::Discrete,
    };
    TestSeriesPosition(p);
}

TEST_F(OlscTransferTaskListControllerImplTest, SaveExists_SdaExists_InvalidLastSi)
{
    TestParam p = {
        TestParam::SaveDataExists::SaveDataExists,
        TestParam::SdaInfoCacheExists::SdaInfoCacheExists,
        TestParam::SdaSeriesIdVariation::SdaSeriesId_1,
        TestParam::SdaCommitIdVariation::SdaCommitId_SameCurrentSave,
        TestParam::LastSiSeriesIdVariation::LastSiSeriesId_Invalid,
        TestParam::LastSiCommitIdVariation::LastSiCommitId_SameCurrentSave,

        SeriesPosition::Discrete,
    };
    TestSeriesPosition(p);
}

TEST_F(OlscTransferTaskListControllerImplTest, SaveExists_SdaExists_CommitIdMismatch)
{
    TestParam p = {
        TestParam::SaveDataExists::SaveDataExists,
        TestParam::SdaInfoCacheExists::SdaInfoCacheExists,
        TestParam::SdaSeriesIdVariation::SdaSeriesId_1,
        TestParam::SdaCommitIdVariation::SdaCommitId_1,
        TestParam::LastSiSeriesIdVariation::LastSiSeriesId_1,
        TestParam::LastSiCommitIdVariation::LastSiCommitId_2,

        SeriesPosition::Discrete,
    };
    TestSeriesPosition(p);
}
