﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_ApiForPrivate.h>
#include <nn/olsc/olsc_ApiForSystemService.h>
#include <nn/olsc/olsc_RemoteStorageController.h>
#include <nn/olsc/olsc_TransferTaskListController.h>
#include <nn/olsc/olsc_SystemEventHolder.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiForApplications.h>

#include <nnt/fsUtil/testFs_util.h>
#include <nnt/fsUtil/testFs_util_function.h>

#include "testOlsc_Util.h"
#include "testOlsc_SaveDataUtil.h"

using namespace nn;
using namespace nn::olsc;

namespace {
    class Stopwatch
    {
    public:
        Stopwatch(bool enable, const char* fmt, ...) : m_Enable(enable)
        {
            std::va_list vlist;
            va_start(vlist, fmt);
            util::VSNPrintf(m_Title, sizeof(m_Title), fmt, vlist);
            va_end(vlist);

            m_Begin = os::GetSystemTick();
        }
        ~Stopwatch()
        {
            if (m_Enable)
            {
                auto end = os::GetSystemTick();
                NN_LOG("%s: %lld ms\n", m_Title, (end - m_Begin).ToTimeSpan().GetMilliSeconds());
            }
        }
    private:
        char m_Title[128];
        bool m_Enable;
        os::Tick m_Begin;
    };

    class OlscTransferTaskListControllerTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            account::InitializeForSystemService();
            DeleteAllTransferTask();

            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();

            nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
            nnt::fs::util::ResetAllocateCount();

            nnt::fs::util::DeleteAllTestSaveData();
        }

        static void TearDownTestCase()
        {
        }
    private:

    };

    nn::Result CreateAndMountSaveData(const char* mountName, nn::ncm::ApplicationId appId, fs::UserId userId, int64_t saveDataSize, int64_t journalSize)
    {
        NN_RESULT_TRY(nn::fs::MountSaveData(mountName, appId, userId))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NNT_OLSC_RESULT_DO(nn::fs::CreateSaveData(appId, userId, appId.value, saveDataSize, journalSize, 0));
            NNT_OLSC_RESULT_DO(nn::fs::MountSaveData(mountName, appId, userId));
        }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    nn::Result CalculateSaveDataHash(nnt::fs::util::Hash* pOut, const account::Uid& uid, const ApplicationId& appId, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        ncm::ApplicationId ncmAppId;
        ncmAppId.value = appId.value;
        fs::UserId userId;
        userId._data[0] = uid._data[0];
        userId._data[1] = uid._data[1];

        NN_RESULT_DO(nn::fs::MountSaveData("save", ncmAppId, userId));
        NN_UTIL_SCOPE_EXIT
        {
            fs::Unmount("save");
        };

        NNT_OLSC_RESULT_DO(nnt::fs::util::CalculateDirectoryTreeHash(pOut, "save:/", workBuffer, workBufferSize));
        nnt::fs::util::DumpBuffer(pOut, sizeof(nnt::fs::util::Hash));

        NN_RESULT_SUCCESS;
    }

    nn::Result CreateRandomSaveData(nnt::fs::util::Hash* pOut, const account::Uid& uid, const ApplicationId& appId, void* workBuffer, size_t workbufferSize, size_t saveDataSize, size_t fillSize) NN_NOEXCEPT
    {
        ncm::ApplicationId ncmAppId;
        ncmAppId.value = appId.value;
        fs::UserId userId;
        userId._data[0] = uid._data[0];
        userId._data[1] = uid._data[1];

        NNT_OLSC_RESULT_DO(CreateAndMountSaveData("save", ncmAppId, userId, saveDataSize, saveDataSize));
        NNT_OLSC_RESULT_DO(nnt::fs::util::CreateTestDirectoryTreeRandomly("save:/", fillSize, 8));
        fs::CommitSaveData("save");
        NN_LOG("\n\n");
        NN_LOG("Src: \n");
        nnt::fs::util::ListDirectoryRecursive("save:/");
        NNT_OLSC_RESULT_DO(nnt::fs::util::CalculateDirectoryTreeHash(pOut, "save:/", workBuffer, workbufferSize));
        nnt::fs::util::DumpBuffer(pOut, sizeof(nnt::fs::util::Hash));
        nn::fs::Unmount("save");

        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteUserSaveData(const account::Uid& uid, const ApplicationId& appId)
    {
        std::unique_ptr<fs::SaveDataIterator> iter;
        NNT_OLSC_RESULT_DO(fs::OpenSaveDataIterator(&iter, fs::SaveDataSpaceId::User));

        const int MaxSaveDataCount = 2000;
        std::unique_ptr<fs::SaveDataInfo[]> saveDataInfoBuffer(new fs::SaveDataInfo[MaxSaveDataCount]);

        int64_t listCount;
        NNT_OLSC_RESULT_DO(iter->ReadSaveDataInfo(&listCount, saveDataInfoBuffer.get(), MaxSaveDataCount));
        for (int i = 0; i < listCount; ++i)
        {
            auto& saveDataInfo = saveDataInfoBuffer[i];
            if (saveDataInfo.applicationId.value == appId.value
                && saveDataInfo.saveDataUserId._data[0] == uid._data[0]
                && saveDataInfo.saveDataUserId._data[1] == uid._data[1])
            {
                NNT_OLSC_RESULT_DO(fs::DeleteSaveData(saveDataInfo.saveDataId));
                NN_RESULT_SUCCESS;
            }
        }
        NN_LOG("SaveData not found: Uid = %016llx%016llx, AppId = %016llx\n", uid._data[0], uid._data[1], appId.value);
        NN_RESULT_SUCCESS;
    }

    const int WorkBufferSize = 4 * 1024 * 1024;
    static char g_WorkBuffer[WorkBufferSize];

    void WaitTaskDone(TransferTaskId tid)
    {
        NN_LOG("Tid: %016llx\n", tid);

        auto tc = OpenTransferTaskListController();

        auto taskStartEventHolder = tc.GetTransferTaskStartEvent();
        os::SystemEvent startEvent;
        taskStartEventHolder.GetSystemEvent(&startEvent);
        startEvent.Wait();

        // 進捗を監視するために新たなタスク実行を止める
        auto stopper = tc.StopTransferTaskExecution();

        auto taskEndEventHolder = tc.GetTransferTaskEndEvent();
        os::SystemEvent doneEvent;
        taskEndEventHolder.GetSystemEvent(&doneEvent);

        while (!doneEvent.TryWait())
        {
            olsc::TransferTaskProgress progress;
            NNT_EXPECT_RESULT_SUCCESS(tc.GetTransferTaskProgress(&progress, tid));
            NN_LOG("Progress: %zu / %zu\n", progress.completedSize, progress.totalSize);
            os::SleepThread(TimeSpan::FromSeconds(1));
        }

        doneEvent.Clear();
    }

    Result GetUid(account::Uid* out, int index)
    {
        account::Uid uids[account::UserCountMax];

        int listCount;
        NN_RESULT_DO(account::ListAllUsers(&listCount, uids, account::UserCountMax));
        NN_ABORT_UNLESS(index < listCount);

        *out = uids[index];

        NN_RESULT_SUCCESS;
    }

    Result DeleteBackupOnHost()
    {
        NN_RESULT_DO(fs::MountHostRoot());
        NN_UTIL_SCOPE_EXIT{
            fs::UnmountHostRoot();
        };

        auto deleteResult = fs::DeleteDirectoryRecursively("C:\\Windows\\Temp\\NxBackup");
        NN_RESULT_THROW_UNLESS(deleteResult.IsSuccess() || fs::ResultPathNotFound::Includes(deleteResult), deleteResult);

        NN_RESULT_SUCCESS;
    }
}

TEST_F(OlscTransferTaskListControllerTest, Basic)
{
    NNT_ASSERT_RESULT_SUCCESS(DeleteBackupOnHost());

    const size_t SaveDataSize = 8 * 1024 * 1024;
    const size_t FillSize = static_cast<int64_t>(SaveDataSize) * 80 / 100;
    nnt::fs::util::Hash hash;
    account::Uid uid;
    NNT_ASSERT_RESULT_SUCCESS(GetUid(&uid, 0));
    ApplicationId appId = { nnt::olsc::BaseApplicationIdValue };

    NNT_EXPECT_RESULT_SUCCESS(DeleteUserSaveData(uid, appId));
    olsc::DeleteAllSaveDataArchiveInfoCache(uid);
    olsc::DeleteAllSeriesInfo(uid);

    NNT_ASSERT_RESULT_SUCCESS(CreateRandomSaveData(&hash, uid, appId, g_WorkBuffer, WorkBufferSize, SaveDataSize, FillSize));

    auto rc = OpenRemoteStorageController(uid);

    // Upload
    {
        {
            AsyncRequest asyncRequest;
            NNT_EXPECT_RESULT_SUCCESS(rc.RequestUpdateDataInfoCacheAsync(&asyncRequest));
            os::SystemEvent asyncEvent;
            asyncRequest.GetEvent(&asyncEvent);
            asyncEvent.Wait();
            NNT_EXPECT_RESULT_SUCCESS(asyncRequest.GetResult());
        }
        TransferTaskId tid;
        NNT_EXPECT_RESULT_SUCCESS(rc.RegisterUploadSaveDataTransferTask(&tid, appId));

        WaitTaskDone(tid);
        NN_LOG("Upload done\n");

        auto tc = OpenTransferTaskListController();
        TransferTaskInfo ti = {};
        EXPECT_TRUE(tc.GetTransferTask(&ti, tid));
        EXPECT_EQ(TransferTaskStatus::Completed, ti.status);
    }

    NNT_EXPECT_RESULT_SUCCESS(DeleteUserSaveData(uid, appId));

    {
        olsc::DeleteAllSaveDataArchiveInfoCache(uid);

        EXPECT_EQ(0, rc.GetDataInfoCount());

        AsyncRequest request;
        NNT_EXPECT_RESULT_SUCCESS(rc.RequestUpdateDataInfoCacheAsync(&request));
        os::SystemEvent updateDone;
        request.GetEvent(&updateDone);
        updateDone.Wait();
        NNT_EXPECT_RESULT_SUCCESS(request.GetResult());

        auto cachedCount = rc.GetDataInfoCount();

        ASSERT_TRUE(cachedCount > 0);
    }

    // 新規DL
    {
        {
            AsyncRequest asyncRequest;
            NNT_EXPECT_RESULT_SUCCESS(rc.RequestUpdateDataInfoCacheAsync(&asyncRequest));
            os::SystemEvent asyncEvent;
            asyncRequest.GetEvent(&asyncEvent);
            asyncEvent.Wait();
            NNT_EXPECT_RESULT_SUCCESS(asyncRequest.GetResult());
        }

        auto cachedCount = rc.GetDataInfoCount();

        ASSERT_TRUE(cachedCount > 0);

        std::unique_ptr<DataInfo[]> listBuffer(new DataInfo[cachedCount]);
        auto listedCount = rc.ListDataInfo(listBuffer.get(), cachedCount, 0);
        EXPECT_EQ(listedCount, cachedCount);

        util::optional<DataInfo> dlTarget = util::nullopt;
        for (int i = 0; i < listedCount; ++i)
        {
            auto& di = listBuffer.get()[i];
            if (di.appId == appId)
            {
                dlTarget.emplace(di);
                break;
            }
        }

        ASSERT_TRUE(dlTarget);

        TransferTaskId tid;
        NNT_EXPECT_RESULT_SUCCESS(rc.RegisterDownloadSaveDataTransferTask(&tid, dlTarget->id));

        WaitTaskDone(tid);
        NN_LOG("Download done\n");

        auto tc = OpenTransferTaskListController();
        TransferTaskInfo ti = {};
        EXPECT_TRUE(tc.GetTransferTask(&ti, tid));
        EXPECT_EQ(TransferTaskStatus::Completed, ti.status);

        NNT_EXPECT_RESULT_SUCCESS(tc.DeleteTransferTask(tid));
    }

    nnt::fs::util::Hash downloadedHash;
    NNT_EXPECT_RESULT_SUCCESS(CalculateSaveDataHash(&downloadedHash, uid, appId, g_WorkBuffer, WorkBufferSize));

    NNT_FS_UTIL_EXPECT_MEMCMPEQ(&hash, &downloadedHash, sizeof(hash));
}
