﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_TransferTaskContextDatabase.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabaseManager.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabase.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_FormatString.h>

using namespace nn;
using namespace nn::olsc;
using namespace nn::olsc::srv;

namespace {
    const fs::SystemSaveDataId SystemSaveDataId = 0x8000000000004000;
    const int64_t   SystemSaveDataSize          = 4 * 1024 * 1024;
    const int64_t   SystemSaveJournalSize       = 4 * 1024 * 1024;
    const uint32_t  SystemSaveDataFlags         = 0;

    srv::util::MountInfo TestSaveInfo = {
        SystemSaveDataId,
        SystemSaveDataSize,
        SystemSaveJournalSize,
        SystemSaveDataFlags
    };

    using ManagerBase = srv::database::TransferTaskDatabaseManagerBase;
    using TransferTaskDatabase = olsc::srv::database::TransferTaskDatabase;
    using TransferTaskContextDatabase = olsc::srv::database::TransferTaskContextDatabase;

    class TestManager : public ManagerBase
    {
    public:
        TestManager(
            srv::util::DefaultMountManager& mountManager) NN_NOEXCEPT
            : ManagerBase(mountManager)
        {}
        using ManagerBase::PerformReadTransaction;
        using ManagerBase::PerformWriteTransaction;
    private:
    };

    class OlscTransferTaskDatabaseManagerTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData());
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
            NNT_ASSERT_RESULT_SUCCESS(time::Initialize());
        }

        static void TearDownTestCase()
        {
            NNT_ASSERT_RESULT_SUCCESS(time::Finalize());
        }
    private:

        static Result InitializeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }
    };

    using TransferTaskDatabase = olsc::srv::database::TransferTaskDatabase;
    using TransferTaskContextDatabase = olsc::srv::database::TransferTaskContextDatabase;

    struct TaskParam
    {
        account::Uid        uid;
        ApplicationId       appId;
        TransferTaskKind    kind;
    };


    const int MaxUserCount = account::UserCountMax;

    TaskParam CreateTaskParam(int n) NN_NOEXCEPT
    {
        return TaskParam {
            {{ static_cast<Bit64>(n % MaxUserCount + 1) }},
            { static_cast<Bit64>(n) },
            TransferTaskKind::Download
        };
    }
}

const int MaxTaskCount = 128;

TEST_F(OlscTransferTaskDatabaseManagerTest, Basic)
{
    srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);

    TestManager manager(mountManager);

    static TransferTaskId appendedIds[MaxTaskCount];
    for (int i = 0; i < MaxTaskCount; ++i)
    {
        auto taskAppender = [&i](TransferTaskDatabase& taskDb, TransferTaskContextDatabase& contextDb) -> Result
        {
            NN_UNUSED(contextDb);
            auto p = CreateTaskParam(i);
            TransferTaskConfig config = {};
            config.kind = p.kind;
            return taskDb.AppendTask(&appendedIds[i], p.uid, p.appId, config, TransferTaskRank::Basic);
        };

        manager.PerformWriteTransaction(taskAppender);
    }

    for (int i = 0; i < MaxTaskCount; ++i)
    {
        auto taskReader = [&i](const TransferTaskDatabase& taskDb, const TransferTaskContextDatabase& contextDb) -> Result
        {
            NN_UNUSED(contextDb);
            TransferTaskDetailInfo di;
            NN_RESULT_DO(taskDb.GetTask(&di, appendedIds[i]));

            EXPECT_EQ(appendedIds[i], di.id);

            NN_RESULT_SUCCESS;
        };

        manager.PerformReadTransaction(taskReader);
    }
}


TEST_F(OlscTransferTaskDatabaseManagerTest, RemoveAutoUploadTask)
{
    srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);
    olsc::srv::database::TransferTaskDatabaseManager ttdm(mountManager);

    account::Uid uid = { 1ULL };

    const int AutoUlTaskCount = 5;

    TransferTaskId autoUlTasks[AutoUlTaskCount];
    for (int i = 0; i < AutoUlTaskCount; ++i)
    {
        ApplicationId appId = { static_cast<Bit64>(0x10000000 + i) };
        TransferTaskConfig config = {};
        config.kind = TransferTaskKind::Upload;
        config.ulInfo.force = false;
        NNT_EXPECT_RESULT_SUCCESS(ttdm.RegisterTransferTask(&autoUlTasks[i], uid, appId, config, i % 2 == 0 ? TransferTaskRank::Basic : TransferTaskRank::Postponed));
    }

    const int ManualUlTaskCount = 4;
    TransferTaskId manualUlTasks[ManualUlTaskCount];
    for (int i = 0; i < ManualUlTaskCount; ++i)
    {
        ApplicationId appId = { static_cast<Bit64>(i) };
        TransferTaskConfig config = {};
        config.kind = TransferTaskKind::Upload;
        config.ulInfo.force = true;
        NNT_EXPECT_RESULT_SUCCESS(ttdm.RegisterTransferTask(&manualUlTasks[i], uid, appId, config, i % 2 == 0 ? TransferTaskRank::Basic : TransferTaskRank::Postponed));
    }

    uint32_t taskCount;
    NNT_EXPECT_RESULT_SUCCESS(ttdm.GetTransferTaskCount(&taskCount));
    EXPECT_EQ(AutoUlTaskCount + ManualUlTaskCount, taskCount);

    NNT_EXPECT_RESULT_SUCCESS(ttdm.RemoveAutoUploadTaskByUser(uid));
    NNT_EXPECT_RESULT_SUCCESS(ttdm.GetTransferTaskCount(&taskCount));
    EXPECT_EQ(ManualUlTaskCount, taskCount);

    for (int i = 0; i < ManualUlTaskCount; ++i)
    {
        EXPECT_TRUE(ttdm.GetTransferTaskDetailInfo(manualUlTasks[i]));
    }
}
