﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_Result.private.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabase.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

#include "testOlsc_SaveDataUtil.h"
#include "testOlsc_Stopwatch.h"

#include <vector>

using namespace nn;
using namespace nn::olsc;
using namespace nn::olsc::srv;
using namespace nnt::olsc;

//#define NN_OLSC_TRANSFER_TASK_DATABASE_ENTRY_MAX_TEST

namespace {

    class OlscTransferTaskDatabaseTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData());
        }

        virtual void TearDown()
        {
            NNT_ASSERT_RESULT_SUCCESS(FinalizeSaveData());
        }

        static void SetUpTestCase()
        {
            nnt::olsc::Initialize();
        }

        static void TearDownTestCase()
        {
            nnt::olsc::Finalize();
        }
    private:

        static Result InitializeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, nnt::olsc::MountInfoForTestDeviceSave.systemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

        static Result FinalizeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, nnt::olsc::MountInfoForTestDeviceSave.systemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

    };

    using TransferTaskDatabase = olsc::srv::database::TransferTaskDatabase;

    struct TaskParam
    {
        account::Uid        uid;
        ApplicationId       appId;
        TransferTaskKind    kind;
    };

    void Dump(const char* title, const TransferTaskDetailInfo& detailInfo)
    {
        auto& v = detailInfo;
        NN_LOG("%s:\n", title);
        NN_LOG("    id: %016llx\n", v.id);
        NN_LOG("   uid: %016llx%016llx\n", v.uid._data[0], v.uid._data[1]);
        NN_LOG("   app: %016llx\n", v.appId);
        NN_LOG("  kind: %d\n", v.config.kind);
        NN_LOG("  rank: %s\n", v.rank == TransferTaskRank::Basic ? "Basic" : "Postponed");
    }

    struct Modules
    {
        Modules() :
            mountManager(nnt::olsc::MountInfoForTestDeviceSave, nnt::olsc::MountInfoForTestUserSettingSave, nnt::olsc::MountInfoForTestUserSeriesInfoSave),
            db(mountManager)
        {}
        srv::util::DefaultMountManager mountManager;
        TransferTaskDatabase db;
    };

    struct AppendParam
    {
        account::Uid            uid;
        ApplicationId           appId;
        TransferTaskConfig      config;
        TransferTaskRank        rank;
    };

    struct TestParam
    {
        AppendParam                                 appendParam;
        TransferTaskDetailInfo                      expected;
    };

    TestParam CreateTestParam(int userIndex, int appIdValue, TransferTaskKind kind, TransferTaskRank rank)
    {
        account::Uid uid = { { static_cast<Bit64>(userIndex + 1) } };
        ApplicationId appId = { static_cast<Bit64>(appIdValue) };
        TransferTaskConfig config = {};
        config.kind = kind;

        AppendParam ap = {
            uid,
            appId,
            config,
            rank
        };

        auto expected = TransferTaskDetailInfo::Make(
            0,
            uid,
            appId,
            config,
            TransferTaskDetailInfo::State::NotFinished,
            rank
        );

        return TestParam
        {
            ap,
            expected
        };
    }

    void AppendAndSetExpectedTid(Modules& m, TestParam& p)
    {
        NNT_ASSERT_RESULT_SUCCESS(m.db.AppendTask(&p.expected.id, p.appendParam.uid, p.appendParam.appId, p.appendParam.config, p.appendParam.rank));
    }

    void TestChangePriority(const TestParam& p)
    {
        // 書き換えが発生するのでコピー
        TestParam tp = p;

        Modules m;

        // Basic キューに1つタスクを積んでおく
        auto firstTaskParam = CreateTestParam(0, 1, TransferTaskKind::Upload, TransferTaskRank::Basic);
        AppendAndSetExpectedTid(m, firstTaskParam);

        AppendAndSetExpectedTid(m, tp);

        // Postponed からの優先度変更
        NNT_ASSERT_RESULT_SUCCESS(m.db.MoveTaskToHead(tp.expected.id));
        tp.expected.rank = TransferTaskRank::Basic; // 期待値も Basic に変更しておく

        std::vector<TransferTaskDetailInfo> listBuffer(2);
        auto listedCount = m.db.ListTasks(listBuffer.data(), 2, 0);
        EXPECT_EQ(2, listedCount);

        Dump("Expected", tp.expected);
        Dump("Actual", listBuffer[0]);
        EXPECT_EQ(tp.expected, listBuffer[0]);
        EXPECT_EQ(firstTaskParam.expected, listBuffer[1]);
    }
}

TEST_F(OlscTransferTaskDatabaseTest, BasicCases)
{
    Modules m;

    const int UserCount = 8;
    const int BasicAppCount = 128;
    const int PostponedAppCount = 128;
    const int AppCount = BasicAppCount + PostponedAppCount;
    const int TaskCount = AppCount * UserCount;

    // 登録するタスクの情報を作成
    std::vector<TestParam> testParams(TaskCount);
    for (int i = 0; i < UserCount; ++i)
    {
        for (int j = 0; j < AppCount; ++j)
        {
            auto rank = j < BasicAppCount ? TransferTaskRank::Basic : TransferTaskRank::Postponed;
            testParams[AppCount * i + j] = CreateTestParam(i, j + 1, TransferTaskKind::Upload, rank);
        }
    }

    EXPECT_EQ(0, m.db.GetTaskCount());

    // Append And Get
    auto& updatedEvent = m.db.GetExecutionListUpdatedEvent();
    for (auto& p : testParams)
    {
        TransferTaskId tid;

        updatedEvent.Clear();
        NNT_ASSERT_RESULT_SUCCESS(m.db.AppendTask(&tid, p.appendParam.uid, p.appendParam.appId, p.appendParam.config, p.appendParam.rank));
        p.expected.id = tid;
        EXPECT_TRUE(updatedEvent.TryWait());

        TransferTaskDetailInfo di;
        NNT_ASSERT_RESULT_SUCCESS(m.db.GetTask(&di, tid));
        EXPECT_EQ(p.expected, di);
    }

    EXPECT_EQ(TaskCount, m.db.GetTaskCount());

    // Listing
    std::vector<TransferTaskDetailInfo> listBuffer(TaskCount);
    auto listedCount = m.db.ListTasks(listBuffer.data(), TaskCount, 0);
    EXPECT_EQ(TaskCount, listedCount);

    std::vector<TestParam> expectedListBuffer(TaskCount);
    std::copy(testParams.begin(), testParams.end(), expectedListBuffer.begin());

    std::stable_sort(expectedListBuffer.begin(), expectedListBuffer.end(), [](const TestParam& lhs, const TestParam& rhs) -> int {
        if (lhs.expected.rank == rhs.expected.rank)
        {
            if (lhs.expected.id < rhs.expected.id)
            {
                return true;
            }
            return false;

        }
        else if (lhs.expected.rank == TransferTaskRank::Basic)
        {
            return true;
        }
        return false;
    });

    for (int i = 0; i < TaskCount; ++i)
    {
        if (expectedListBuffer[i].expected != listBuffer[i])
        {
            Dump("expected", expectedListBuffer[i].expected);
            Dump("actual", listBuffer[i]);
        }
        EXPECT_EQ(expectedListBuffer[i].expected, listBuffer[i]);
    }

    // 削除
    for (auto& listed : listBuffer)
    {
        m.db.RemoveTask(listed.id);
    }
    EXPECT_EQ(0, m.db.GetTaskCount());
}

TEST_F(OlscTransferTaskDatabaseTest, ChangePriorityFromPostponed)
{
    auto p = CreateTestParam(0, 2, TransferTaskKind::Upload, TransferTaskRank::Postponed);
    TestChangePriority(p);
}

TEST_F(OlscTransferTaskDatabaseTest, ChangePriorityFromBasic)
{
    auto p = CreateTestParam(0, 2, TransferTaskKind::Upload, TransferTaskRank::Basic);
    TestChangePriority(p);
}

TEST_F(OlscTransferTaskDatabaseTest, AppendDuplicationError)
{
    Modules m;

    auto p = CreateTestParam(0, 1, TransferTaskKind::Upload, TransferTaskRank::Basic);
    NNT_ASSERT_RESULT_SUCCESS(m.db.AppendTask(&p.expected.id, p.appendParam.uid, p.appendParam.appId, p.appendParam.config, p.appendParam.rank));

    auto& updatedEvent = m.db.GetExecutionListUpdatedEvent();
    updatedEvent.Clear();
    NNT_EXPECT_RESULT_FAILURE(olsc::ResultTransferTaskDuplicated, m.db.AppendTask(&p.expected.id, p.appendParam.uid, p.appendParam.appId, p.appendParam.config, p.appendParam.rank));
    EXPECT_FALSE(updatedEvent.TryWait());

    updatedEvent.Clear();
    auto p2 = CreateTestParam(0, 1, TransferTaskKind::Upload, TransferTaskRank::Postponed);
    NNT_EXPECT_RESULT_FAILURE(olsc::ResultTransferTaskDuplicated, m.db.AppendTask(&p2.expected.id, p2.appendParam.uid, p2.appendParam.appId, p2.appendParam.config, p2.appendParam.rank));
    EXPECT_FALSE(updatedEvent.TryWait());
}

#if defined(NN_OLSC_TRANSFER_TASK_DATABASE_ENTRY_MAX_TEST)
TEST_F(OlscTransferTaskDatabaseTest, DbFull)
{
    Modules m;

    for (int i = 0; i < olsc::srv::MaxUserCount; ++i)
    {
        for (int j = 0; j < olsc::srv::MaxApplicationCount; ++j)
        {
            auto p = CreateTestParam(i, j + 1, TransferTaskKind::Upload, TransferTaskRank::Basic);
            nnt::olsc::Stopwatch s(j % 512 == 0, "User %d: App %d\n", i, j);
            NNT_ASSERT_RESULT_SUCCESS(m.db.AppendTask(&p.expected.id, p.appendParam.uid, p.appendParam.appId, p.appendParam.config, p.appendParam.rank));
        }
    }

    // auto p = CreateTestParam(0, 9999, TransferTaskKind::Upload, TransferTaskRank::Basic);
}
#endif

TEST_F(OlscTransferTaskDatabaseTest, RemoveTaskIfMatch)
{
    Modules m;

    auto p1 = CreateTestParam(0, 1, TransferTaskKind::Upload, TransferTaskRank::Basic);
    AppendAndSetExpectedTid(m, p1);

    auto p2 = CreateTestParam(0, 2, TransferTaskKind::Upload, TransferTaskRank::Postponed);
    AppendAndSetExpectedTid(m, p2);

    EXPECT_EQ(2, m.db.GetTaskCount());
    {
        TransferTaskDetailInfo di;
        NNT_EXPECT_RESULT_SUCCESS(m.db.GetTask(&di, p1.expected.id));
    }
    {
        TransferTaskDetailInfo di;
        NNT_EXPECT_RESULT_SUCCESS(m.db.GetTask(&di, p2.expected.id));
    }

    m.db.RemoveTaskIfMatch(p1.appendParam.uid, [](const olsc::srv::TransferTaskDetailInfo& di) -> bool {
        return di.rank == TransferTaskRank::Postponed;
    });

    EXPECT_EQ(1, m.db.GetTaskCount());
    {
        TransferTaskDetailInfo di;
        NNT_EXPECT_RESULT_SUCCESS(m.db.GetTask(&di, p1.expected.id));
    }
    {
        TransferTaskDetailInfo di;
        NNT_EXPECT_RESULT_FAILURE(olsc::ResultTransferTaskNotFoundInExecutionList, m.db.GetTask(&di, p2.expected.id));
    }
}
