﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/base/testBase_Exit.h>

#include <nn/account.h>
#include <nn/nn_ApplicationId.h>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_UserAccountSystemSaveData.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/database/olsc_SaveDataArchiveInfoCache.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiForApplications.h>

#include <cstdio>
#include <cstdlib>
#include <string>
#include <cstring>
#include <random>

#include "testOlsc_Stopwatch.h"

using namespace nn;
using namespace nn::olsc;
using namespace nn::olsc::srv;
using namespace nnt::olsc;

namespace {

    const fs::SystemSaveDataId SystemSaveDataId = 0x8000000000004000;
    const Bit64 BaseApplicationId = 0x0005000c10000001ULL;
    const int64_t              SystemSaveDataSize = 4 * 1024 * 1024;
    const int64_t              SystemSaveJournalSize = 4 * 1024 * 1024;
    const uint32_t             SystemSaveDataFlags = 0;
    const int VariationCount = 2000;

    srv::util::MountInfo TestSaveInfo = {
        SystemSaveDataId,
        SystemSaveDataSize,
        SystemSaveJournalSize,
        SystemSaveDataFlags
    };

    account::Uid GetFirstUserId() NN_NOEXCEPT
    {
        int userCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetUserCount(&userCount));
        NN_ABORT_UNLESS(userCount > 0);

        account::Uid uid;
        int listCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&listCount, &uid, 1));
        return uid;
    }

    // SaveDataArchiveInfo を偽造
    SaveDataArchiveInfo CreateSaveDataArchive(int appIdOffset) NN_NOEXCEPT
    {
        SaveDataArchiveInfo ret;
        nn::os::GenerateRandomBytes(&ret.id, sizeof(ret.id));
        ret.nsaId = {0};
        ret.applicationId = { BaseApplicationId + appIdOffset };
        ret.userId = GetFirstUserId();
        ret.deviceId = static_cast<DeviceId>(0x0);
        ret.dataSize = 1024;
        nn::os::GenerateRandomBytes(&ret.seriesInfo.seriesId, sizeof(ret.seriesInfo.seriesId));
        ret.seriesInfo.commitId = 0;
        ret.status = SaveDataArchiveStatus::Uploading;
        ret.autoBackup = false;
        ret.hasThumbnail = false;
        ret.launchRequiredVersion = 0;
        ret.numOfPartitions = 8;
        time::PosixTime time = { 1413263600 };
        ret.savedAt    = time;
        ret.timeoutAt  = time;
        ret.finishedAt = time;
        ret.createdAt  = time;
        ret.createdAt  = time;
        return ret;
    }

    void shuffle(int array[], int size) NN_NOEXCEPT
    {
        int i = size;
        while( i > 1 )
        {
            int j = rand() % i;
            i--;
            int t = array[i];
            array[i] = array[j];
            array[j] = t;
        }
    }
    void ConfirmSdaList(srv::database::SaveDataArchiveInfoCache* pSdaInfoCache, srv::SaveDataArchiveInfo sda[], int arrayCount, int variationCount) NN_NOEXCEPT
    {
        for(int i = 0; i < variationCount / arrayCount; i++)
        {
            int num;
            {
                Stopwatch s(true, "List: offset = %d, count = %d", i, arrayCount);
                num = pSdaInfoCache->List(sda, arrayCount, i * arrayCount);
            }
            for(int j = 0; j < num; j++)
            {
                auto targetSda = CreateSaveDataArchive(i * arrayCount + j);
                EXPECT_EQ(sda[j].applicationId.value, targetSda.applicationId.value);
            }
        }
    }
    void ConfirmGetSda(srv::database::SaveDataArchiveInfoCache* pSdaInfoCache, srv::SaveDataArchiveInfo sda[], int variationCount) NN_NOEXCEPT
    {
        for(int i = 0; i < variationCount; i++)
        {
            {
                nn::util::optional<DataInfo> dataInfo;
                {
                    Stopwatch s(true, "GetDataInfoByDataId: id = %016llx", sda[i].id);
                    dataInfo = pSdaInfoCache->GetDataInfoByDataId(sda[i].id);
                }
                ASSERT_TRUE(dataInfo);
                EXPECT_EQ(dataInfo->appId.value, sda[i].applicationId.value);
                EXPECT_EQ(dataInfo->id, sda[i].id);
                {
                    Stopwatch s(true, "GetDataInfoByAppId: id = %016llx", sda[i].applicationId);
                    dataInfo = pSdaInfoCache->GetDataInfoByApplicationId(sda[i].applicationId);
                }
                ASSERT_TRUE(dataInfo);
                EXPECT_EQ(dataInfo->appId.value, sda[i].applicationId.value);
                EXPECT_EQ(dataInfo->id, sda[i].id);
            }
            {
                nn::util::optional<SaveDataArchiveInfo> targetSda;
                {
                    Stopwatch s(true, "GetDataArchiveInfoByDataId: id = %016llx", sda[i].id);
                    targetSda = pSdaInfoCache->GetSaveDataArchiveInfoByDataId(sda[i].id);
                }
                ASSERT_TRUE(targetSda);
                EXPECT_EQ(targetSda->applicationId.value, sda[i].applicationId.value);
                EXPECT_EQ(targetSda->id, sda[i].id);
            }
        }
    }

}

TEST(SaveDataArchiveCache, AddListSimple)
{
    nn::fs::DeleteSystemSaveData(SystemSaveDataId, GetFirstUserId());

    SaveDataArchiveInfo sdaList[VariationCount];
    int applicationIdVariation[VariationCount];
    for(int i = 0; i < VariationCount; i++)
    {
        applicationIdVariation[i] = i;
    }
    // 順番をバラバラにして入れるためシャッフルする
    shuffle(applicationIdVariation, VariationCount);

    srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);
    srv::database::SaveDataArchiveInfoCache sdaInfoCache(GetFirstUserId(), mountManager);

    for(int i = 0; i < VariationCount; i++)
    {
        Stopwatch s(true, "Add[%d]", i);
        sdaList[i] = CreateSaveDataArchive(applicationIdVariation[i]);
        // 順不同で Add
        ASSERT_TRUE(sdaInfoCache.Add(sdaList[i]).IsSuccess());
    }

    {
        Stopwatch s(true, "Commit");
        auto writeMount = mountManager.AcquireUserSettingsSaveForWrite(GetFirstUserId());
        ASSERT_TRUE(writeMount.Commit().IsSuccess());
    }
    SaveDataArchiveInfo sda[10];

    ASSERT_TRUE(sdaInfoCache.GetCount() == VariationCount);

    {
        Stopwatch s(true, "List (stride 1)");
        // 1 ずつとりだして順番通りであることを確認する
        ConfirmSdaList(&sdaInfoCache, sda, 1, VariationCount);
    }

    {
        Stopwatch s(true, "List (stride 10)");
        // 10 ずつとりだして順番通りであることを確認する
        ConfirmSdaList(&sdaInfoCache, sda, 10, VariationCount);
    }

    {
        Stopwatch s(true, "Get");
        // Get で取得できることを確認する
        ConfirmGetSda(&sdaInfoCache, sdaList, VariationCount);
    }

    // 順番をバラバラにして入れるためシャッフルする
    shuffle(applicationIdVariation, VariationCount);

    // Add し直す(内部的にはリプレース)
    for(int i = 0; i < VariationCount; i++)
    {
        Stopwatch s(true, "Replace[%d]", i);
        sdaList[i] = CreateSaveDataArchive(applicationIdVariation[i]);
        // 順不同で Add
        NNT_ASSERT_RESULT_SUCCESS(sdaInfoCache.Add(sdaList[i]));
    }

    {
        Stopwatch s(true, "List (stride 1)");
        // 1 ずつとりだして順番通りであることを確認する
        ConfirmSdaList(&sdaInfoCache, sda, 1, VariationCount);
    }

    {
        Stopwatch s(true, "List (stride 10)");
        // 10 ずつとりだして順番通りであることを確認する
        ConfirmSdaList(&sdaInfoCache, sda, 10, VariationCount);
    }

    {
        Stopwatch s(true, "Get");
        // Get で取得できることを確認する
        ConfirmGetSda(&sdaInfoCache, sdaList, VariationCount);
    }

    // 1件ずつ削除
    for (int i = 0; i < VariationCount; i++)
    {
        ASSERT_TRUE(sdaInfoCache.GetDataInfoByApplicationId(sdaList[i].applicationId));
        {
            Stopwatch s(true, "Delete[%d] ", i);
            NNT_ASSERT_RESULT_SUCCESS(sdaInfoCache.Delete(sdaList[i].applicationId));
        }
        ASSERT_FALSE(sdaInfoCache.GetDataInfoByApplicationId(sdaList[i].applicationId));
    }
}

// 空のリストに対する操作
TEST(SaveDataArchiveCache, EmptyListCheck)
{
    nn::fs::DeleteSystemSaveData(SystemSaveDataId, GetFirstUserId());
    srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);
    srv::database::SaveDataArchiveInfoCache sdaInfoCache(GetFirstUserId(), mountManager);

    ASSERT_TRUE(sdaInfoCache.GetCount() == 0);
    auto srcSda = CreateSaveDataArchive(0);
    ASSERT_EQ(sdaInfoCache.GetDataInfoByApplicationId(srcSda.applicationId), false);
    ASSERT_EQ(sdaInfoCache.GetDataInfoByDataId(srcSda.id), false);
    ASSERT_EQ(sdaInfoCache.GetSaveDataArchiveInfoByDataId(srcSda.id), false);
    NNT_ASSERT_RESULT_FAILURE(olsc::ResultSaveDataArchiveInfoCacheNoLongerExists, sdaInfoCache.Delete(0));
}

// 全削除 API の確認
TEST(SaveDataArchiveCache, DeleteAllCheck)
{
    nn::fs::DeleteSystemSaveData(SystemSaveDataId, GetFirstUserId());

    SaveDataArchiveInfo sdaList[VariationCount];
    int applicationIdVariation[VariationCount];
    for(int i = 0; i < VariationCount; i++)
    {
        applicationIdVariation[i] = i;
    }

    srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);
    srv::database::SaveDataArchiveInfoCache sdaInfoCache(GetFirstUserId(), mountManager);

    for(int i = 0; i < VariationCount; i++)
    {
        sdaList[i] = CreateSaveDataArchive(applicationIdVariation[i]);
        ASSERT_TRUE(sdaInfoCache.Add(sdaList[i]).IsSuccess());
    }

    {
        auto writeMount = mountManager.AcquireUserSettingsSaveForWrite(GetFirstUserId());
        ASSERT_TRUE(writeMount.Commit().IsSuccess());
    }
    ASSERT_TRUE(sdaInfoCache.GetCount() == VariationCount);

    ASSERT_TRUE(sdaInfoCache.DeleteAll().IsSuccess());

    ASSERT_TRUE(sdaInfoCache.GetCount() == 0);
}


// 時間がかかるためコメントアウト
// TEST(SaveDataArchiveCache, AddTooMuchSave)
// {
//     srv::util::UserAccountSaveDataManager saveDataManager(SystemSaveDataId, GetFirstUserId(), SystemSaveDataSize, SystemSaveJournalSize, SystemSaveDataFlags, MountName);
//     srv::database::SaveDataArchiveInfoCache sdaInfoCache(&saveDataManager);
//     nn::fs::DeleteSystemSaveData(SystemSaveDataId, GetFirstUserId());
//     ASSERT_TRUE(saveDataManager.Ensure().IsSuccess());
//     ASSERT_TRUE(saveDataManager.Mount().IsSuccess());
//     for(int i = 0; i < MaxApplicationCount; i++)
//     {
//         auto sda = CreateSaveDataArchive(MaxApplicationCount - i);
//         ASSERT_TRUE(sdaInfoCache.Add(sda).IsSuccess());
//     }
//     auto sda = CreateSaveDataArchive(MaxApplicationCount + 1);
//     NNT_EXPECT_RESULT_FAILURE(ResultSdaInfoCacheMaxCount, sdaInfoCache.Add(sda));
//     ASSERT_TRUE(saveDataManager.Commit().IsSuccess());
//     saveDataManager.Unmount();
// }

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    account::InitializeForSystemService();
    fs::SetEnabledAutoAbort(false);

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
