﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_Utility.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/olsc_AutoUploadSettingManager.h>
#include <nn/olsc/srv/database/olsc_SaveDataArchiveInfoCache.h>
#include <nn/olsc/srv/database/olsc_SeriesInfoDatabase.h>
#include <nn/olsc/srv/database/olsc_TransferTaskContextDatabase.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabaseManager.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabase.h>
#include <nn/olsc/srv/olsc_SystemEventManager.h>
#include <nn/olsc/srv/olsc_TransferTask.h>
#include <nn/olsc/srv/olsc_TransferTaskAgent.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_FormatString.h>

#include "testOlsc_SaveDataUtil.h"

#include <vector>

#define NNT_OLSC_TTA_BASIC
#define NNT_OLSC_TTA_CANCEL

using namespace nn;
using namespace nn::olsc;
using namespace nn::olsc::srv;
using namespace nn::olsc::srv::database;


namespace {
    const fs::SystemSaveDataId SystemSaveDataId = 0x8000000000004000;
    const int64_t   SystemSaveDataSize = 4 * 1024 * 1024;
    const int64_t   SystemSaveJournalSize = 4 * 1024 * 1024;
    const uint32_t  SystemSaveDataFlags = 0;

    olsc::srv::util::MountInfo TestDeviceSave = {
        SystemSaveDataId,
        SystemSaveDataSize,
        SystemSaveJournalSize,
        SystemSaveDataFlags
    };

    olsc::srv::util::MountInfo TestUserSettingSave = {
        SystemSaveDataId + 1,
        SystemSaveDataSize,
        SystemSaveJournalSize,
        SystemSaveDataFlags
    };

    olsc::srv::util::MountInfo TestUserSeriesInfoSave = {
        SystemSaveDataId + 2,
        SystemSaveDataSize,
        SystemSaveJournalSize,
        SystemSaveDataFlags
    };

    class OlscAutoUploadSettingManagerTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData({}));
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
            NNT_ASSERT_RESULT_SUCCESS(time::Initialize());

#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::account::InitializeForAdministrator();

            nnt::olsc::CleanupUsers();
            nn::account::Uid uid;
            nnt::olsc::CreateUsers(&uid, 1);
#else
            account::InitializeForSystemService();
#endif
            NNT_ASSERT_RESULT_SUCCESS(nifm::InitializeSystem());
        }

        static void TearDownTestCase()
        {
            NNT_ASSERT_RESULT_SUCCESS(time::Finalize());
        }

        static Result InitializeSaveData(fs::UserId fsUser)
        {
            fs::SystemSaveDataId toDelete[] = {
                TestDeviceSave.systemSaveDataId,
                TestUserSettingSave.systemSaveDataId,
                TestUserSeriesInfoSave.systemSaveDataId,
            };
            for (auto id : toDelete)
            {
                NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, id, fsUser))
                    NN_RESULT_CATCH(fs::ResultTargetNotFound)
                    {
                    }
                NN_RESULT_END_TRY;
            }

            NN_RESULT_SUCCESS;
        }
    private:

    };


    account::Uid GetFirstUserId() NN_NOEXCEPT
    {
        int userCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetUserCount(&userCount));
        NN_ABORT_UNLESS(userCount > 0);

        account::Uid uid;
        int listCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&listCount, &uid, 1));
        return uid;
    }

    void CreateAppIdList(ApplicationId out[], int count) NN_NOEXCEPT
    {
        for (int i = 0; i < count; ++i)
        {
            out[i] = { static_cast<Bit64>(i) };
        }
    }
}

TEST_F(OlscAutoUploadSettingManagerTest, AppSetting)
{
    olsc::srv::util::DefaultMountManager mountManager(TestDeviceSave, TestUserSettingSave, TestUserSeriesInfoSave);

    auto uid = GetFirstUserId();
    OlscAutoUploadSettingManagerTest::InitializeSaveData(fs::ConvertAccountUidToFsUserId(uid));

    AutoUploadSettingManager ulSettingManager(mountManager);
    // 初期値
    EXPECT_FALSE(ulSettingManager.GetGlobalSetting(uid));

    // アプリ別に設定を追加
    const int MaxAppCount = 100;
    std::unique_ptr<ApplicationId[]> appIdList(new ApplicationId[MaxAppCount]);
    CreateAppIdList(appIdList.get(), MaxAppCount);
    for (int i = 0; i < MaxAppCount; ++i)
    {
        auto& appId = appIdList[i];
        EXPECT_EQ(AutoUploadSettingDatabase::ApplicationAutoUploadSetting::NotSet, ulSettingManager.GetApplicationSetting(uid, appId));
        ulSettingManager.SetApplicationSetting(uid, appId, AutoUploadSettingDatabase::ApplicationAutoUploadSetting::Enabled);
        EXPECT_EQ(AutoUploadSettingDatabase::ApplicationAutoUploadSetting::Enabled, ulSettingManager.GetApplicationSetting(uid, appId));
    }
}

TEST_F(OlscAutoUploadSettingManagerTest, GlobalSetting)
{
    olsc::srv::util::DefaultMountManager mountManager(TestDeviceSave, TestUserSettingSave, TestUserSeriesInfoSave);

    auto uid = GetFirstUserId();
    OlscAutoUploadSettingManagerTest::InitializeSaveData(fs::ConvertAccountUidToFsUserId(uid));

    AutoUploadSettingManager ulSettingManager(mountManager);

    // 初期値
    EXPECT_FALSE(ulSettingManager.GetGlobalSetting(uid));

    // 全体設定をオフからオンにする場合の挙動
    ulSettingManager.SetGlobalSetting(uid, true);
    EXPECT_TRUE(ulSettingManager.GetGlobalSetting(uid));

    // 全体設定をオンからオフにする場合の挙動
    ulSettingManager.SetGlobalSetting(uid, false);
    EXPECT_FALSE(ulSettingManager.GetGlobalSetting(uid));

}

