﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/mem.h>
#include <nn/vi.h>
#include <nvn/nvn.h>
#include <nv/nv_MemoryManagement.h>

#include <nvn/nvn_Cpp.h>
#include <nvn/nvn_CppMethods.h>
#include <nvn/nvn_CppFuncPtr.h>
#include <nvn/nvn_CppFuncPtrImpl.h>

#include "AppClass.h"

namespace {
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    nn::mem::StandardAllocator s_Allocator;

    void* Allocate(size_t size)
    {
        void* ptr = s_Allocator.Allocate(size);
        NN_ASSERT_NOT_NULL(ptr);

        return ptr;
    }

    void* AlignedAllocate(size_t size, size_t alignment)
    {
        void* ptr = s_Allocator.Allocate(size, alignment);
        NN_ASSERT_NOT_NULL(ptr);

        return ptr;
    }

    void Free(void* ptr)
    {
        return s_Allocator.Free(ptr);
    }
}

AppClass::AppClass(int width, int height, const char* title) :
    AppBase(width, height, title),
    m_GpuHangRequested(false)
{
    NN_UNUSED(reinterpret_cast<void*>(Allocate));
    NN_UNUSED(reinterpret_cast<void*>(AlignedAllocate));
    NN_UNUSED(reinterpret_cast<void*>(Free));
}

AppClass::~AppClass()
{
}

void AppClass::OnInit(PFNNVNBOOTSTRAPLOADERPROC loader, NVNnativeWindow nativeWindow)
{
    m_NativeWindow = nativeWindow;

    s_Allocator.Initialize(malloc(MemoryHeapSize), MemoryHeapSize);

    nvn::DeviceGetProcAddressFunc getProcAddress = reinterpret_cast<nvn::DeviceGetProcAddressFunc>(loader("nvnDeviceGetProcAddress"));
    nvn::nvnLoadCPPProcs(NULL, getProcAddress);

    {
        nvn::Device* pDevice = NULL;
        int majorVersion, minorVersion;
        pDevice->GetInteger(nvn::DeviceInfo::API_MAJOR_VERSION, &majorVersion);
        pDevice->GetInteger(nvn::DeviceInfo::API_MINOR_VERSION, &minorVersion);

        if (majorVersion != NVN_API_MAJOR_VERSION || minorVersion < NVN_API_MINOR_VERSION)
        {
            NN_ASSERT(false, "NVN SDK not supported by current driver.");
        }
    }

    {
        nvn::DeviceFlagBits flags = 0;
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
        flags |= nvn::DeviceFlagBits::DEBUG_ENABLE;
#endif

        nvn::DeviceBuilder deviceBuilder;
        deviceBuilder.SetDefaults();
        deviceBuilder.SetFlags(flags);

        if (!m_Device.Initialize(&deviceBuilder))
        {
            NN_ASSERT(false, "Failed to initialize nvn::Device");
        }
    }

    nvn::nvnLoadCPPProcs(&m_Device, getProcAddress);

    {
        nvn::QueueBuilder queueBuilder;
        queueBuilder.SetDevice(&m_Device);
        queueBuilder.SetDefaults();

        if (!m_Queue.Initialize(&queueBuilder))
        {
            NN_ASSERT(false, "Failed to initialize Queue");
        }
    }

    {
        size_t memorySize = nn::util::align_up(RenderTargetMemorySize, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
        void* memory = AlignedAllocate(memorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);

        nvn::MemoryPoolBuilder memoryPoolBuilder;
        memoryPoolBuilder.SetDevice(&m_Device);
        memoryPoolBuilder.SetDefaults();
        memoryPoolBuilder.SetFlags(nvn::MemoryPoolFlags::COMPRESSIBLE | nvn::MemoryPoolFlags::CPU_NO_ACCESS | nvn::MemoryPoolFlags::GPU_CACHED);
        memoryPoolBuilder.SetStorage(memory, memorySize);

        if (!m_RenderTargetMemoryPool.Initialize(&memoryPoolBuilder))
        {
            NN_ASSERT(false, "Failed to initialize MemoryPool for RenderTarget");
        }

        nvn::TextureBuilder textureBuilder;
        textureBuilder.SetDevice(&m_Device);
        textureBuilder.SetDefaults();
        textureBuilder.SetSize2D(m_Width, m_Height);
        textureBuilder.SetFlags(nvn::TextureFlags::COMPRESSIBLE | nvn::TextureFlags::DISPLAY);
        textureBuilder.SetFormat(nvn::Format::RGBA8);

        size_t textureSize = textureBuilder.GetStorageSize();
        size_t textureAlignment = textureBuilder.GetStorageAlignment();

        ptrdiff_t memoryPoolOffset = 0;
        for (int i = 0; i < FrameCount; i++)
        {
            textureBuilder.SetStorage(&m_RenderTargetMemoryPool, memoryPoolOffset);

            if (!m_RenderTargets[i].Initialize(&textureBuilder))
            {
                NN_ASSERT(false, "Failed to initialize Texture for RenderTarget%d", i);
            }

            memoryPoolOffset += textureSize;
            memoryPoolOffset = nn::util::align_up(memoryPoolOffset, textureAlignment);
        }
    }

    {
        nvn::Texture* textures[FrameCount];

        for (int i = 0; i < FrameCount; i++)
        {
            textures[i] = &m_RenderTargets[i];
        }

        nvn::WindowBuilder windowBuilder;
        windowBuilder.SetDevice(&m_Device);
        windowBuilder.SetDefaults();
        windowBuilder.SetTextures(FrameCount, textures);
        windowBuilder.SetNativeWindow(m_NativeWindow);

        if (!m_Window.Initialize(&windowBuilder))
        {
            NN_ASSERT(false, "Failed to initialize Window");
        }
    }

    {
        m_TextureAvailableSync.Initialize(&m_Device);
    }

    {
        size_t commandMemorySize = nn::util::align_up(CommandMemorySize, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
        void* commandMemory = AlignedAllocate(commandMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);

        nvn::MemoryPoolBuilder memoryPoolBuilder;
        memoryPoolBuilder.SetDevice(&m_Device);
        memoryPoolBuilder.SetDefaults();
        memoryPoolBuilder.SetFlags(nvn::MemoryPoolFlags::CPU_UNCACHED | nvn::MemoryPoolFlags::GPU_CACHED);
        memoryPoolBuilder.SetStorage(commandMemory, commandMemorySize);

        m_CommandMemoryInfo.size = commandMemorySize;
        m_CommandMemoryInfo.memory = commandMemory;

        if (!m_CommandMemoryPool.Initialize(&memoryPoolBuilder))
        {
            NN_ASSERT(false, "Failed to initialize MemoryPool for CommandBuffer");
        }

        int controlMemoryMinimumSize;
        m_Device.GetInteger(nvn::DeviceInfo::COMMAND_BUFFER_MIN_CONTROL_SIZE, &controlMemoryMinimumSize);
        int controlMemoryAlignment;
        m_Device.GetInteger(nvn::DeviceInfo::COMMAND_BUFFER_CONTROL_ALIGNMENT, &controlMemoryAlignment);

        m_ControlMemoryInfo.size = nn::util::align_up(ControlMemorySize, controlMemoryMinimumSize);
        m_ControlMemoryInfo.memory = AlignedAllocate(m_ControlMemoryInfo.size, controlMemoryAlignment);

        if (!m_CommandBuffer.Initialize(&m_Device))
        {
            NN_ASSERT(false, "Failed to initialize CommandBuffer");
        }
    }
} // NOLINT(impl/function_size)

void AppClass::OnUpdate()
{
}

void AppClass::OnRender()
{
    int renderTargetIndex;

    nvn::WindowAcquireTextureResult acquireResult = m_Window.AcquireTexture(&m_TextureAvailableSync, &renderTargetIndex);
    NN_ASSERT_EQUAL(acquireResult, nvn::WindowAcquireTextureResult::SUCCESS);

    m_CommandBuffer.AddCommandMemory(&m_CommandMemoryPool, 0, m_CommandMemoryPool.GetSize());
    m_CommandBuffer.AddControlMemory(m_ControlMemoryInfo.memory, m_ControlMemoryInfo.size);

    m_CommandBuffer.BeginRecording();
    {
        nvn::Texture* renderTargets[1] = { &m_RenderTargets[renderTargetIndex] };
        m_CommandBuffer.SetRenderTargets(1, renderTargets, NULL, NULL, NULL);

        float clearColor[4] = { 0.f, 0.2f, 0.4f, 1.f };
        m_CommandBuffer.ClearColor(0, clearColor, nvn::ClearColorMask::RGBA);
    }
    nvn::CommandHandle commandHandle = m_CommandBuffer.EndRecording();

    if (m_GpuHangRequested)
    {
        // コマンドリストを破壊し、GPU ハングを発生させます。
        std::memset(m_CommandMemoryInfo.memory, -1, m_CommandMemoryInfo.size);
        std::memset(m_ControlMemoryInfo.memory, -1, m_ControlMemoryInfo.size);
    }

    m_Queue.WaitSync(&m_TextureAvailableSync);
    m_Queue.SubmitCommands(1, &commandHandle);
    m_Queue.PresentTexture(&m_Window, renderTargetIndex);
    m_Queue.Finish();
}

void AppClass::OnDestroy()
{
    m_Queue.Finish();

    m_CommandBuffer.Finalize();
    m_CommandMemoryPool.Finalize();
    m_TextureAvailableSync.Finalize();
    m_Window.Finalize();
    for (int i = 0; i < FrameCount; i++)
    {
        m_RenderTargets[i].Finalize();
    }
    m_RenderTargetMemoryPool.Finalize();
    m_Queue.Finalize();

    s_Allocator.Finalize();
}
