﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nsd/detail/nsd_FqdnResolver.h>
#include <nn/nsd/detail/nsd_DetailApi.h>
#include <nn/nsd/nsd_Version.h>

using namespace nn::nsd;

namespace
{
    void GenerateTestDefaultSaveData(SaveData* pOut) NN_NOEXCEPT
    {
        const DeviceId deviceId = {"1234567890"};
        const EnvironmentIdentifier env = {"dd1"};
        detail::GetDefaultSettings(pOut, deviceId, env);
    }

    void ExecuteAllApiExpectSuccess(detail::FqdnResolver* resolver) NN_NOEXCEPT
    {
        // Resolve(Ex)
        {
            const Fqdn in = { "hoge-%.com" };
            Fqdn out;
            NNT_EXPECT_RESULT_SUCCESS(resolver->Resolve(&out, in));
            EXPECT_STREQ("hoge-dd1.com", out.value);
        }
        {
            const Fqdn in = { "hoge-%.com" };
            Fqdn out;
            nn::Result result;
            NNT_EXPECT_RESULT_SUCCESS(resolver->ResolveEx(&result, &out, in));
            EXPECT_STREQ("hoge-dd1.com", out.value);
        }

        // GetNasApiFqdn(Ex) / GetNasRequestFqdn(Ex)
        {
            Fqdn out;
            NNT_EXPECT_RESULT_SUCCESS(resolver->GetNasApiFqdn(&out));
        }
        {
            nn::Result result;
            Fqdn out;
            NNT_EXPECT_RESULT_SUCCESS(resolver->GetNasApiFqdnEx(&result, &out));
            EXPECT_TRUE(result.IsSuccess());
        }
        {
            Fqdn out;
            NNT_EXPECT_RESULT_SUCCESS(resolver->GetNasRequestFqdn(&out));
        }
        {
            nn::Result result;
            Fqdn out;
            NNT_EXPECT_RESULT_SUCCESS(resolver->GetNasRequestFqdnEx(&result, &out));
            EXPECT_TRUE(result.IsSuccess());
        }

        // GetNasServiceSetting(Ex)
        const NasServiceName testServiceName[] =
        {
            NasServiceNameOfNxAccount,
            NasServiceNameOfNxShop
        };
        for (const auto& name : testServiceName)
        {
            {
                NasServiceSetting setting;
                NNT_EXPECT_RESULT_SUCCESS(resolver->GetNasServiceSetting(&setting, name));
            }
            {
                nn::Result result;
                NasServiceSetting setting;
                NNT_EXPECT_RESULT_SUCCESS(resolver->GetNasServiceSettingEx(&result, &setting, name));
                EXPECT_TRUE(result.IsSuccess());
            }
        }
    }

    template <typename ErrorResult>
    void ExecuteAllApiExpectFailure(detail::FqdnResolver* resolver) NN_NOEXCEPT
    {
        // Resolve(Ex)
        {
            const Fqdn in = { "hoge-%.com" };
            Fqdn out;
            NNT_EXPECT_RESULT_FAILURE(ErrorResult, resolver->Resolve(&out, in));
        }
        {
            const Fqdn in = { "hoge-%.com" };
            Fqdn out;
            nn::Result result;
            NNT_EXPECT_RESULT_SUCCESS(resolver->ResolveEx(&result, &out, in));
            NNT_EXPECT_RESULT_FAILURE(ErrorResult, result);
        }

        // GetNasApiFqdn(Ex) / GetNasRequestFqdn(Ex)
        {
            Fqdn out;
            NNT_EXPECT_RESULT_FAILURE(ErrorResult, resolver->GetNasApiFqdn(&out));
        }
        {
            nn::Result result;
            Fqdn out;
            NNT_EXPECT_RESULT_SUCCESS(resolver->GetNasApiFqdnEx(&result, &out));
            NNT_EXPECT_RESULT_FAILURE(ErrorResult, result);
        }
        {
            Fqdn out;
            NNT_EXPECT_RESULT_FAILURE(ErrorResult, resolver->GetNasRequestFqdn(&out));
        }
        {
            nn::Result result;
            Fqdn out;
            NNT_EXPECT_RESULT_SUCCESS(resolver->GetNasRequestFqdnEx(&result, &out));
            NNT_EXPECT_RESULT_FAILURE(ErrorResult, result);
        }

        // GetNasServiceSetting(Ex)
        const NasServiceName testServiceName[] =
        {
            NasServiceNameOfNxAccount,
            NasServiceNameOfNxShop
        };
        for (const auto& name : testServiceName)
        {
            {
                NasServiceSetting setting;
                NNT_EXPECT_RESULT_FAILURE(ErrorResult, resolver->GetNasServiceSetting(&setting, name));
            }
            {
                nn::Result result;
                NasServiceSetting setting;
                NNT_EXPECT_RESULT_SUCCESS(resolver->GetNasServiceSettingEx(&result, &setting, name));
                NNT_EXPECT_RESULT_FAILURE(ErrorResult, result);
            }
        }
    }
}

TEST(NsdUnitTest_FqdnResolver, Default)
{
    static SaveData saveData;
    GenerateTestDefaultSaveData(&saveData);
    detail::FqdnResolver resolver;
    resolver.SetSaveData(&saveData);
    resolver.SetFatalError(nn::ResultSuccess()); // 現状は Success をセットしないとダメ

    SCOPED_TRACE("");
    ExecuteAllApiExpectSuccess(&resolver);
}

TEST(NsdUnitTest_FqdnResolver, VersionMissmatch)
{
    static SaveData saveData;
    GenerateTestDefaultSaveData(&saveData);
    saveData.version = 0; // version 不一致にする
    NN_STATIC_ASSERT(0 < nn::nsd::Version);

    detail::FqdnResolver resolver;
    resolver.SetSaveData(&saveData);
    resolver.SetFatalError(nn::ResultSuccess()); // 現状は Success をセットしないとダメ

    SCOPED_TRACE("");
    ExecuteAllApiExpectFailure<ResultInvalidSettingVersion>(&resolver);
}

TEST(NsdUnitTest_FqdnResolver, InvalidState)
{
    static SaveData saveData;
    GenerateTestDefaultSaveData(&saveData);

    detail::FqdnResolver resolver;
    resolver.SetSaveData(&saveData);

    // Invalid な状態にする
    resolver.SetFatalError(nn::nsd::ResultInitializeFailed()); // 適当なエラーを入れる

    SCOPED_TRACE("");
    ExecuteAllApiExpectFailure<ResultInitializeFailed>(&resolver);
}

TEST(NsdUnitTest_FqdnResolver, ForcibleSuccessFqdn1)
{
    static SaveData saveData;
    GenerateTestDefaultSaveData(&saveData);
    saveData.version = 0; // version 不一致にする
    NN_STATIC_ASSERT(0 < nn::nsd::Version);

    detail::FqdnResolver resolver;
    resolver.SetSaveData(&saveData);
    resolver.SetFatalError(nn::nsd::ResultInitializeFailed()); // 適当なエラーを入れる

    {
        const Fqdn in = {"api.sect.srv.nintendo.net"}; // Sirius FQDN
        Fqdn out;
        NNT_EXPECT_RESULT_SUCCESS(resolver.Resolve(&out, in));
        EXPECT_STREQ("api.sect.srv.nintendo.net", out.value);
    }
    {
        const Fqdn in = {"ctest.cdn.nintendo.net"}; // 接続テスト FQDN
        Fqdn out;
        NNT_EXPECT_RESULT_SUCCESS(resolver.Resolve(&out, in));
        EXPECT_STREQ("ctest.cdn.nintendo.net", out.value);
    }
}

TEST(NsdUnitTest_FqdnResolver, ForcibleSuccessFqdn2)
{
    static SaveData saveData;
    GenerateTestDefaultSaveData(&saveData);

    detail::FqdnResolver resolver;
    resolver.SetSaveData(&saveData);

    // Invalid な状態にする
    resolver.SetFatalError(nn::nsd::ResultInitializeFailed()); // 適当なエラーを入れる

    {
        const Fqdn in = {"api.sect.srv.nintendo.net"}; // Sirius FQDN
        Fqdn out;
        NNT_EXPECT_RESULT_SUCCESS(resolver.Resolve(&out, in));
        EXPECT_STREQ("api.sect.srv.nintendo.net", out.value);
    }
    {
        const Fqdn in = {"ctest.cdn.nintendo.net"}; // 接続テスト FQDN
        Fqdn out;
        NNT_EXPECT_RESULT_SUCCESS(resolver.Resolve(&out, in));
        EXPECT_STREQ("ctest.cdn.nintendo.net", out.value);
    }
}



