﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nsd/nsd_ApiForMiddleware.h>
#include "./testNsd_Constant.h"

#if NN_BUILD_CONFIG_OS_WIN

// Windows.h マクロと衝突しないかの確認

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif

#ifndef NOMINMAX
#define NOMINMAX
#endif

#include <nn/nn_Windows.h>

#endif // NN_BUILD_CONFIG_OS_WIN

using namespace nn::nsd;

//
// TODO:環境が dd1 とは限らないのをどうにかする. GetEnvironmentIdentifier で取得して TEST するとか.
//

TEST(NsdTest_ForMiddleWare, Resolve1)
{
    Fqdn in = {{"hoge-%.com"}};
    Fqdn out;
    NNT_ASSERT_RESULT_SUCCESS( Resolve(&out, in) );
    EXPECT_STREQ("hoge-dd1.com", out.value);
}

TEST(NsdTest_ForMiddleWare, Resolve2)
{
    Fqdn in;
    Fqdn out = {};

    // NULL 終端してない FQDN
    for(auto& c : in.value)
    {
        c = 'a';
    }

    NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultFqdnIsNotNullTerminated, Resolve(&out, in));

    // エラー時、out には不定な値が入れられるので返り値チェックはしない
}

TEST(NsdTest_ForMiddleWare, Resolve3)
{
    Fqdn in = {};
    Fqdn out = {};

    // 254 文字以上の % を含む FQDN
    for(auto& c : in.value)
    {
        c = 'a';
    }
    in.value[in.Size - 1] = '\0'; // NULL終端
    in.value[5] = '%'; // 適当な箇所に % を仕込む

    NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultFqdnTooLong, Resolve(&out, in));

    // エラー時、out には不定な値が入れられるので返り値チェックはしない
}

TEST(NsdTest_ForMiddleWare, Resolve4)
{
    Fqdn in = {};
    Fqdn out = {};

    // 253 文字でぎりぎり解決できるFQDN
    for(auto& c : in.value)
    {
        c = 'a';
    }
    in.value[253] = '\0'; // NULL終端
    in.value[5] = '%'; // 適当な箇所に % を仕込む

    NNT_ASSERT_RESULT_SUCCESS( Resolve(&out, in) );
}

TEST(NsdTest_ForMiddleWare, Resolve5)
{
    Fqdn in = {};
    Fqdn out = {};

    // 253 文字でぎりぎり解決できるFQDN(末尾に %)
    for(auto& c : in.value)
    {
        c = 'a';
    }
    in.value[253] = '\0'; // NULL終端
    in.value[252] = '%'; // 末尾

    NNT_ASSERT_RESULT_SUCCESS( Resolve(&out, in) );
}

TEST(NsdTest_ForMiddleWare, ResolveEx1)
{
    Fqdn in = {{"hoge-%.com"}};
    Fqdn out = {};
    NNT_ASSERT_RESULT_SUCCESS( ResolveEx(&out, in) );
    EXPECT_STREQ("hoge-dd1.com", out.value);
}

TEST(NsdTest_ForMiddleWare, ResolveEx2)
{
    Fqdn in = {};
    Fqdn out = {};

    // NULL 終端してない FQDN
    for(auto& c : in.value)
    {
        c = 'a';
    }

    NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultFqdnIsNotNullTerminated, ResolveEx(&out, in));
    EXPECT_STREQ(nnt::nsd::constant::InvalidFqdn.value, out.value); // 失敗文字列(仮)
}

TEST(NsdTest_ForMiddleWare, ResolveEx3)
{
    Fqdn in = {};
    Fqdn out = {};

    // 254 文字以上の % を含む FQDN
    for(auto& c : in.value)
    {
        c = 'a';
    }
    in.value[in.Size - 1] = '\0'; // NULL終端
    in.value[5] = '%'; // 適当な箇所に % を仕込む

    NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultFqdnTooLong, ResolveEx(&out, in));    // 内部では nn::nsd::ResultFqdnTooLong だが ResolveEx は失敗しない
    EXPECT_STREQ(nnt::nsd::constant::InvalidFqdn.value, out.value); // 失敗文字列(仮)
}

TEST(NsdTest_ForMiddleWare, ResolveEx4)
{
    Fqdn in = {};
    Fqdn out = {};

    // 253 文字でぎりぎり解決できるFQDN
    for(auto& c : in.value)
    {
        c = 'a';
    }
    in.value[253] = '\0'; // NULL終端
    in.value[5] = '%'; // 適当な箇所に % を仕込む

    NNT_ASSERT_RESULT_SUCCESS( ResolveEx(&out, in) );
}

TEST(NsdTest_ForMiddleWare, ResolveEx5)
{
    Fqdn in = {};
    Fqdn out = {};

    // 253 文字でぎりぎり解決できるFQDN(末尾に %)
    for(auto& c : in.value)
    {
        c = 'a';
    }
    in.value[253] = '\0'; // NULL終端
    in.value[252] = '%'; // 末尾

    NNT_ASSERT_RESULT_SUCCESS( ResolveEx(&out, in) );
}

TEST(NsdTest_ForMiddleWare, ResolveNasFqdn)
{
    {
        Fqdn out;
        NNT_EXPECT_RESULT_SUCCESS(Resolve(&out, nnt::nsd::constant::Dd1.nasApiFqdn));
        EXPECT_STREQ(nnt::nsd::constant::Dd1.nasApiFqdn.value, out.value); // dd1 の FQDN はそのまま出てくる
    }
    {
        Fqdn out;
        NNT_EXPECT_RESULT_SUCCESS(Resolve(&out, nnt::nsd::constant::Dd1.nasRequestFqdn));
        EXPECT_STREQ(nnt::nsd::constant::Dd1.nasRequestFqdn.value, out.value); // dd1 の FQDN はそのまま出てくる
    }
}
