﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using CsTestAssistants;
using System.Reflection;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace testNs_SystemUpdateVariationCs
{
    using System.Linq;
    using ExpectEvaluator = ThrowEvaluator<UnexpectFailureException>;

    [TestClass]
    public class UnitMain : TestClassBase
    {
        // `TestContext` プロパティを定義しておけば、UnitTestフレームワークで勝手にインスタンスが提供される.
        public TestContext TestContext { get; set; }
        private const string LogPrefix = "[NsSystemUpdate]";

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが生成され、呼び出される前に一度だけ実施。
        /// </summary>
        /// <param name="context"></param>
        //!----------------------------------------------------------------------------
        [ClassInitialize]
        public static void TestClassInitialize( TestContext context )
        {
            ClassInitialize( context, LogPrefix );
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが破棄されるタイミングで一度だけ実施。
        /// </summary>
        //!----------------------------------------------------------------------------
        [ClassCleanup]
        public static void TestClassCleanup()
        {
            ClassCleanup();
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// `TestProperty属性の "JIRA" キーの値` と `Sigloコンフィギュレーションコンテキスト` に依存した作業用ワーキングディレクトリパスを生成します.
        /// </summary>
        /// <param name="context">Sigloコンフィギュレーションコンテキスト</param>
        /// <param name="methodName">
        /// 参照対象TestProperty属性を保持するメソッド名.
        /// 指定を省略した場合、呼び出されたカレントメソッド名が自動で選択されます.
        /// 明示的な null 指定の場合、もしくは "JIRA" キーの TestPropertyが見つからない場合には、TestPropertyに依存しないベース作業用ディレクトリが返却されます。
        /// </param>
        /// <returns>
        /// TestProperty( "JIRA", * ) に依存した作業用ディレクトリパス、もしくはベース作業用ディレクトリパス。
        /// </returns>
        //!----------------------------------------------------------------------------
        private string GenerateIntermediateDirectoryAsMethod( SigloHelper.Configuration.Context context,
            [System.Runtime.CompilerServices.CallerMemberName] string methodName = null )
        {
            var runAssembly = Assembly.GetExecutingAssembly();
            var asmTitle = ( AssemblyTitleAttribute )System.Attribute.GetCustomAttribute( runAssembly, typeof( AssemblyTitleAttribute ) );
            return GenerateIntermediateDirectory( context, asmTitle.Title, methodName );
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 「ダウンロード済本体更新の永続化機能」の検証
        /// `testNs_NupAbortOnUpdate` テストで生成/アップロードされた NUPコンテンツの再利用を想定しています。
        /// テスト実施期間: 約13分
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty( "JIRA", "SIGLO-60967" )]
        public void TestForNupPersistenceOfDownloadedPackages()
        {
            using ( new TestMethodLog() )
            {
                var method = MethodBase.GetCurrentMethod();
                var executor = new SigloHelper.CommodityExecutor.Context( ActiveConfiguration );

                var contentsPath = System.IO.Path.Combine( SigloHelper.Path.GetTestsRoot(), "Outputs", "AnyCPU", "Tests", "testNs_NupAbortOnUpdate", executor.TargetBuildType );
                var testerPath = System.IO.Path.Combine( SigloHelper.Path.GetTestsRoot(), "Outputs", executor.TargetPlatform, "Tests", "testNs_NupPersistence", executor.TargetBuildType, "testNs_NupPersistence.nsp" );
                var nspArg = $"--testns_MountHost={contentsPath} --testns_ActionCase={{0}}";

                //-----------------------------------
                // ゲームカード消去
                //-----------------------------------
                var hasInsertedGameCard = executor.RunDevMenuCommandSystem( $"gamecard status", @"^Inserted$" );
                if ( hasInsertedGameCard ) { executor.RunDevMenuCommandSystem( "gamecard erase" ); }

                //-----------------------------------
                // 途中経過残っていない系
                //-----------------------------------
                // ターゲット初期化
                InitializeSdev( executor );

                // ダウンロードが完了していない状態で強制再起動したら、途中経過は保存されない
                ExpectEvaluator.IsTrue( executor.RunOnTarget( testerPath, null, string.Format( nspArg, "ExitWithinDownloading" ), 5 * 60 ) );
                ExpectEvaluator.IsTrue( executor.ResetTarget() );
                ExpectEvaluator.IsTrue( executor.RunOnTarget( testerPath, null, string.Format( nspArg, "Expect_NeedsDownload" ) ) );

                // ダウンロードが完了していない状態で正常再起動したら、途中経過は保存されない
                ExpectEvaluator.IsTrue( executor.RunOnTarget( testerPath, null, string.Format( nspArg, "ExitWithinDownloading" ), 5 * 60 ) );
                ExpectEvaluator.IsTrue( RebootTargetByPowerButton( executor ) );
                ExpectEvaluator.IsTrue( executor.RunOnTarget( testerPath, null, string.Format( nspArg, "Expect_NeedsDownload" ) ) );

                // ダウンロードが完了している状態で強制再起動したら、途中経過は保存されない
                ExpectEvaluator.IsTrue( executor.RunOnTarget( testerPath, null, string.Format( nspArg, "ExitAtDownloadDone" ), 10 * 60 ) );
                ExpectEvaluator.IsTrue( executor.ResetTarget() );
                ExpectEvaluator.IsTrue( executor.RunOnTarget( testerPath, null, string.Format( nspArg, "Expect_NeedsDownload" ) ) );

                //-----------------------------------
                // 途中経過残ってる系
                //-----------------------------------

                // ダウンロードが完了している状態で正常再起動したら、途中経過が保存される
                ExpectEvaluator.IsTrue( executor.RunOnTarget( testerPath, null, string.Format( nspArg, "ExitAtDownloadDone" ), 10 * 60 ) );
                ExpectEvaluator.IsTrue( RebootTargetByPowerButton( executor ) );
                ExpectEvaluator.IsTrue( executor.RunOnTarget( testerPath, null, string.Format( nspArg, "Expect_Downloaded" ) ) );

                // ダウンロード完了状態を適用( インストール )する
                ExpectEvaluator.IsTrue( executor.RunOnTarget( testerPath, null, string.Format( nspArg, "ApplyDownloadedUpdate" ) ) );

                // 適用してから正常再起動しても途中経過は残っていない
                ExpectEvaluator.IsTrue( RebootTargetByPowerButton( executor ) );
                ExpectEvaluator.IsTrue( executor.RunOnTarget( testerPath, null, string.Format( nspArg, "Expect_UpToDate" ) ) );

                // 適用してから強制再起動しても途中経過は残っていない
                ExpectEvaluator.IsTrue( executor.ResetTarget() );
                ExpectEvaluator.IsTrue( executor.RunOnTarget( testerPath, null, string.Format( nspArg, "Expect_UpToDate" ) ) );
            }
        }
    }
}
