﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <utility>
#include <string>
#include <vector>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/fs.h>

#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/nifm.h>
#include "libraries/testNs_MountHost.h"
#include "libraries/testNs_CommandOptions.h"
#include "libraries/testNs_InstantMessageThread.h"
#include "testNs_NcmStorageVerify.h"

#define LOG_OUT( ... )  NN_LOG( "[NsSystemUpdate] " __VA_ARGS__ )

namespace {

    //!--------------------------------------------------------------------------------------
    //! 配列要素数算出.
    //!--------------------------------------------------------------------------------------
    template< typename TYPE, size_t SIZE >
    inline static unsigned CountOf( const TYPE( & )[ SIZE ] ) {
        return SIZE;
    }

    //!--------------------------------------------------------------------------------------
    //! Result出力.
    //!--------------------------------------------------------------------------------------
    static void LogoutResultValue( const nn::Result& result, const char* const pTitle )
    {
        if ( nullptr != pTitle )
        {
            LOG_OUT( "%s\n", pTitle );
        }
        LOG_OUT( "-------- Result ---------\n" );
        LOG_OUT( "module = %u\n", result.GetModule() );
        LOG_OUT( "description = %u\n", result.GetDescription() );
        LOG_OUT( "inner value = 0x%08x\n", result.GetInnerValueForDebug() );
        LOG_OUT( "-------------------------\n" );
    }

    //!--------------------------------------------------------------------------------------
    //! 並列キャンセラ
    //!--------------------------------------------------------------------------------------
    static const size_t CancellerThreadStackSize = nn::os::ThreadStackAlignment * 4;
    typedef testns::InstantMessenger< CancellerThreadStackSize >::ThreadContext ParallelCanceller;

    //!--------------------------------------------------------------------------------------
    //! Case Setup
    //!--------------------------------------------------------------------------------------
    class NupCase : public testing::Test, public nnt::nsutil::ApplicationInstaller
    {
    protected:
        virtual void SetUp()
        {
            nn::ncm::Initialize();
            nn::ns::Initialize();

            auto& verifierInstance = nn::util::Get( m_NcmStorageVerifier );
            new ( &verifierInstance ) NcmStorageVerifier( nn::ncm::StorageId::BuildInSystem );
            verifierInstance.Initialize();

            auto& cancellerInstance = nn::util::Get( m_ParallelCanceller );
            new ( &cancellerInstance ) ParallelCanceller();
            cancellerInstance.Initialize();
        }

        virtual void TearDown()
        {
            nn::util::Get( m_ParallelCanceller ).Finalize();
            nn::util::Get( m_NcmStorageVerifier ).Finalize();
            nn::ns::Finalize();
            nn::ncm::Finalize();
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::nifm::Initialize() );
            NN_ABORT_UNLESS_RESULT_SUCCESS( testns::MountHostFileSystem( "Contents" ) );
            testns::InitializeCommandLineOptions();
        }

        static void TearDownTestCase()
        {
            testns::FinalizeCommandLineOptions();
            testns::UnmountHostFileSystem();
        }

    protected:
        nn::util::TypedStorage< ParallelCanceller, sizeof( ParallelCanceller ), NN_ALIGNOF( ParallelCanceller ) > m_ParallelCanceller;
        nn::util::TypedStorage< NcmStorageVerifier, sizeof( NcmStorageVerifier ), NN_ALIGNOF( NcmStorageVerifier ) > m_NcmStorageVerifier;

    public:
        NcmStorageVerifier& GetNcmStorageVerifier() NN_NOEXCEPT
        {
            return nn::util::Get( m_NcmStorageVerifier );
        }

        ParallelCanceller& GetParallelCanceller() NN_NOEXCEPT
        {
            return nn::util::Get( m_ParallelCanceller );
        }

        typedef bool( *RunnableDownloadActionType )( NupCase& context, nn::ns::SystemUpdateControl& control );
        void DoDownloadAction( RunnableDownloadActionType doRunnable, const nn::ns::LatestSystemUpdate expectLatestState, bool ignoreInstall = true ) NN_NOEXCEPT;
    };

    //!--------------------------------------------------------------------------------------
    //! Codition of configuration for exit.
    //!--------------------------------------------------------------------------------------
    template< typename KindCategoryEnumrate, typename KindElapsedContainer = uint32_t >
    class ExitCondition
    {
    public:
        typedef KindCategoryEnumrate    CategoryEnumrateType;   // for enumrate.
        typedef KindElapsedContainer    ElapsedContainerType;   // for millisecond span.

        explicit ExitCondition( const CategoryEnumrateType& category, const ElapsedContainerType& elapsedTime ) NN_NOEXCEPT
            : m_ElapsedTime( elapsedTime ),
            m_Category( category )
        {
        }

        const CategoryEnumrateType& GetCategory() const NN_NOEXCEPT
        {
            return m_Category;
        }

        const ElapsedContainerType& GetElapsedTime() const NN_NOEXCEPT
        {
            return m_ElapsedTime;
        }

    private:
        const ElapsedContainerType  m_ElapsedTime;
        const CategoryEnumrateType  m_Category;
    };

    //!--------------------------------------------------------------------------------------
    //! RequestSynchronizer
    //!--------------------------------------------------------------------------------------
    template< typename THandle, typename TExitCondition = void >
    class RequestSynchronizer
    {
    public:
        typedef ParallelCanceller               CancellerType;
        typedef TExitCondition                  ExitConditionType;
        typedef nn::ns::SystemUpdateControl     ControlType;
        typedef THandle                         HandleType;

        HandleType& GetHandle() NN_NOEXCEPT
        {
            return m_Handle;
        }

        const nn::TimeSpan GetElapsedTime() const NN_NOEXCEPT
        {
            return m_ElapsedTime;
        }

        nn::Result WaitResponse( const ExitConditionType* pExitCondition = nullptr ) NN_NOEXCEPT
        {
            const auto before = nn::os::GetSystemTick();
            auto result = OnWaitResponse( m_Control, m_Handle, pExitCondition );
            m_ElapsedTime = ( nn::os::GetSystemTick() - before ).ToTimeSpan();
            LOG_OUT( "Response wait : %lld ms.\n", m_ElapsedTime.GetMilliSeconds() );
            return result;
        }


    protected:
        explicit RequestSynchronizer( ControlType& control ) NN_NOEXCEPT : m_Control( control )
        {
        }

        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle, const ExitConditionType* pExitCondition ) NN_NOEXCEPT = 0;

    private:
        ControlType&    m_Control;
        HandleType      m_Handle;
        nn::TimeSpan    m_ElapsedTime;
    };

    //!--------------------------------------------------------------------------------------
    //! CheckLatestUpdate
    //!--------------------------------------------------------------------------------------
    class NewlyUpdateInquirer : public RequestSynchronizer< nn::ns::AsyncLatestSystemUpdate >
    {
    public:
        static const nn::ns::LatestSystemUpdate INVALID_LATEST_VALUE = static_cast< nn::ns::LatestSystemUpdate >( -1 );
        typedef RequestSynchronizer< nn::ns::AsyncLatestSystemUpdate > BaseType;

        explicit NewlyUpdateInquirer( ControlType& control ) NN_NOEXCEPT
            : BaseType( control ), m_LatestStatus( INVALID_LATEST_VALUE )
        {
        }

        const nn::ns::LatestSystemUpdate& GetLatestStatus() const NN_NOEXCEPT
        {
            return m_LatestStatus;
        }

        const char* GetLatestStatusString() const NN_NOEXCEPT
        {
            switch ( m_LatestStatus )
            {
            case nn::ns::LatestSystemUpdate::Downloaded:    return "Downloaded";
            case nn::ns::LatestSystemUpdate::NeedsDownload: return "NeedsDownload";
            case nn::ns::LatestSystemUpdate::UpToDate:      return "UpToDate";
            default:                                        return "Unknown";
            }
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle, const ExitConditionType* pExitCondition ) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED( pExitCondition );
            NN_UNUSED( control );

            while ( !handle.TryWait() )
            {
                LOG_OUT( "Wating check latest update...\n" );
                nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
            }
            return handle.Get( &m_LatestStatus );
        }

    private:
        nn::ns::LatestSystemUpdate m_LatestStatus;
    };

    //!--------------------------------------------------------------------------------------
    //! ApplyDownloadedUpdate
    //!--------------------------------------------------------------------------------------
    class NewlyUpdateInstaller : public RequestSynchronizer< int >
    {
    public:
        typedef RequestSynchronizer< int > BaseType;

        explicit NewlyUpdateInstaller( ControlType& control ) NN_NOEXCEPT : BaseType( control )
        {
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle, const ExitConditionType* pExitCondition ) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED( pExitCondition );
            NN_UNUSED( handle );

            control.ApplyDownloadedUpdate();
            NN_RESULT_SUCCESS;
        }
    };

    //!--------------------------------------------------------------------------------------
    //! Exit category for DownloadLatestUpdate.
    //!--------------------------------------------------------------------------------------
    enum class ExitCategoryForDownload : nn::Bit8
    {
        None,
        InDownloadPrepared,
        InDownloading,
    };

    //!--------------------------------------------------------------------------------------
    //! DownloadLatestUpdate
    //!--------------------------------------------------------------------------------------
    class NewlyUpdateDownloader : public RequestSynchronizer< nn::ns::AsyncResult, ExitCondition< ExitCategoryForDownload > >
    {
    public:
        typedef RequestSynchronizer< nn::ns::AsyncResult, ExitConditionType > BaseType;
        typedef nn::ns::SystemUpdateProgress ProgressValueType;

        explicit NewlyUpdateDownloader( ControlType& control ) NN_NOEXCEPT : BaseType( control )
        {
        }

        static const char* GetBackStateString( const nn::ns::BackgroundNetworkUpdateState state ) NN_NOEXCEPT
        {
            switch ( state )
            {
            case nn::ns::BackgroundNetworkUpdateState::None:        return "None";
            case nn::ns::BackgroundNetworkUpdateState::InProgress:  return "InProgress";
            case nn::ns::BackgroundNetworkUpdateState::Ready:       return "Ready";
            default:                                                return "Unknown";
            }
        }

        static const char* GetBackStateString() NN_NOEXCEPT
        {
            return GetBackStateString( nn::ns::GetBackgroundNetworkUpdateState() );
        }

        inline const ProgressValueType& GetExitedProgressValue() const NN_NOEXCEPT
        {
            return exitedProgressValue;
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle, const ExitConditionType* pExitCondition ) NN_NOEXCEPT NN_OVERRIDE
        {
            const auto exitCategory = ( nullptr != pExitCondition ) ? pExitCondition->GetCategory() : ExitCategoryForDownload::None;
            const int64_t spanMillisForExit = ( nullptr != pExitCondition ) ? pExitCondition->GetElapsedTime() : 0;
            auto beginForExit = nn::os::GetSystemTick();
            auto resetBeginAtTotalAny = false;

            while ( !handle.TryWait() )
            {
                const auto progress = control.GetDownloadProgress();
                LOG_OUT( "state( %s ), loaded %lld, total %lld\n", GetBackStateString(), progress.loaded, progress.total );

                if ( false == resetBeginAtTotalAny && progress.total > 0 )
                {
                    resetBeginAtTotalAny = true;
                    beginForExit = nn::os::GetSystemTick();
                }
                const auto span = ( nn::os::GetSystemTick() - beginForExit ).ToTimeSpan();

                switch ( exitCategory )
                {
                case ExitCategoryForDownload::InDownloadPrepared:
                    if ( span.GetMilliSeconds() >= spanMillisForExit )
                    {
                        LOG_OUT( "Detected the request of exit from download process, at => [ %lld / %lld ]\n", progress.loaded, progress.total );
                        exitedProgressValue = progress;
                        NN_RESULT_SUCCESS;
                    }
                    break;
                case ExitCategoryForDownload::InDownloading:
                    if ( progress.total > 0 && span.GetMilliSeconds() >= spanMillisForExit )
                    {
                        LOG_OUT( "Detected the request of exit from download process, at => [ %lld / %lld ]\n", progress.loaded, progress.total );
                        exitedProgressValue = progress;
                        NN_RESULT_SUCCESS;
                    }
                    break;
                default:
                    break;
                }

                // 厳密に途中退出( exit )の実施精度上げるには、
                //   - 別スレッド化してイベントによるループ終了.
                //   - MultiWaitでタイマ && AsyncResult のイベント検知.
                // の手法が必要なんだが、このテストではそんな精度いらんので寝かす方式で.
                nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
            }
            const auto progress = control.GetDownloadProgress();
            LOG_OUT( "state( %s ), loaded %lld, total %lld\n", GetBackStateString(), progress.loaded, progress.total );
            return handle.Get();
        }

    protected:
        ProgressValueType exitedProgressValue;
    };

    //!--------------------------------------------------------------------------------------
    //! Network connection request.
    //! 本クライアントアプリケーションがネットワークリソースを占有する事を要求します。
    //!--------------------------------------------------------------------------------------
    static bool SubmitConnectionNetwork() NN_NOEXCEPT
    {
        LOG_OUT( "Request network connection ...\n" );
        const nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
        nn::nifm::SubmitNetworkRequestAndWait();
        const nn::TimeSpan spanOnConnection = nn::os::GetSystemTick().ToTimeSpan() - before;
        LOG_OUT( "Response wait : %lld ms.\n", spanOnConnection.GetMilliSeconds() );
        const auto availableNetwork = nn::nifm::IsNetworkAvailable();
        LOG_OUT( "Network connection %s.\n", ( availableNetwork ? "succeeded" : "failed" ) );
        if ( false == availableNetwork )
        {
            LogoutResultValue( nn::nifm::HandleNetworkRequestResult(), "Network connection error." );
        }
        return availableNetwork;
    }

    //!--------------------------------------------------------------------------------------
    //! Install procedure
    //!--------------------------------------------------------------------------------------
    static bool ExecuteInstall( nn::ns::SystemUpdateControl& control ) NN_NOEXCEPT
    {
        // インストール. ApplyDownloadedUpdate
        LOG_OUT( "Start install downloaded update... On API [nn::ns::SystemUpdateControl::ApplyDownloadedUpdate]\n" );
        NewlyUpdateInstaller installer( control );
        const auto result = installer.WaitResponse();
        NNT_EXPECT_RESULT_SUCCESS( result );
        LOG_OUT( "Complete install downloaded update.\n" );
        return result.IsSuccess();
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ダウンロード部以外の共通化
    //! @param[in] doRunnable           ダウンロード部の実装処理を指定します。
    //!                                 呼び出し時は Occupy() 済であることが保証されます。
    //!                                 処理中に Relieve()を行った場合、処理終了時には Occupy() 状態に復帰しておく必要があります。
    //! @param[in] expectLatestState    RequestCheckLatestUpdate で取得する期待値。
    //!                                 期待値と違う場合ASSERTします。
    //! @param[in] ignoreInstall        true の場合、インストール / ncmベリファイは実施しない.
    //!--------------------------------------------------------------------------------------
    void NupCase::DoDownloadAction( RunnableDownloadActionType doRunnable,
        const nn::ns::LatestSystemUpdate expectLatestState,
        bool ignoreInstall ) NN_NOEXCEPT
    {
        // ストレージ上のインストールコンテンツ及び本体更新対象コンテンツを含むコレクション生成
        auto& verifier = GetNcmStorageVerifier();
        verifier.PrepareUnexpectedContentOnStorage( ignoreInstall );

        // ネットワーク接続確立.
        ASSERT_EQ( true, SubmitConnectionNetwork() );

        // 本体更新制御の占有開始.
        nn::ns::SystemUpdateControl control;
        NNT_ASSERT_RESULT_SUCCESS( control.Occupy() );

        nn::ns::LatestSystemUpdate latest = NewlyUpdateInquirer::INVALID_LATEST_VALUE;
        {
            // 最新の更新の検出. RequestCheckLatestUpdate
            LOG_OUT( "Start latest update checking... On API [nn::ns::SystemUpdateControl::RequestCheckLatestUpdate]\n" );
            NewlyUpdateInquirer inquirer( control );
            NNT_EXPECT_RESULT_SUCCESS( control.RequestCheckLatestUpdate( &inquirer.GetHandle() ) );
            NNT_EXPECT_RESULT_SUCCESS( inquirer.WaitResponse() );
            LOG_OUT( "Complete latest update checking [ %s ].\n", inquirer.GetLatestStatusString() );

            latest = inquirer.GetLatestStatus();
            ASSERT_EQ( expectLatestState, latest );
        }

        bool newlyInstalled = false;
        if ( nn::ns::LatestSystemUpdate::NeedsDownload == latest && nullptr != doRunnable )
        {
            // download 成功及び完了したら true が返される想定。
            if ( true == doRunnable( *this, control ) )
            {
                // インストール. ApplyDownloadedUpdate
                newlyInstalled = ExecuteInstall( control );
            }
        }
        else if ( false == ignoreInstall && nn::ns::LatestSystemUpdate::Downloaded == latest )
        {
            // インストール. ApplyDownloadedUpdate
            newlyInstalled = ExecuteInstall( control );
        }

        // 無事インストールできたなら本体更新完了確認( latest status, already installed, ncmベリファイ )を実施。
        if ( true == newlyInstalled )
        {
            latest = NewlyUpdateInquirer::INVALID_LATEST_VALUE;
            {
                // インストール済確認. RequestCheckLatestUpdate == nn::ns::LatestSystemUpdate::UpToDate
                LOG_OUT( "Start verify latest update status... On API [nn::ns::SystemUpdateControl::RequestCheckLatestUpdate]\n" );
                NewlyUpdateInquirer inquirer( control );
                NNT_EXPECT_RESULT_SUCCESS( control.RequestCheckLatestUpdate( &inquirer.GetHandle() ) );
                NNT_EXPECT_RESULT_SUCCESS( inquirer.WaitResponse() );
                latest = inquirer.GetLatestStatus();
                EXPECT_EQ( nn::ns::LatestSystemUpdate::UpToDate, latest );
                LOG_OUT( "Complete verify latest update status [ %s ].\n", inquirer.GetLatestStatusString() );
            }

            if ( nn::ns::LatestSystemUpdate::UpToDate == latest )
            {
                // NCMデータベースベリファイ
                verifier.VerifyHasContentOnStorage();
                verifier.VerifyUnnecessaryPlaceHolderOnStorage();
                verifier.VerifyUnexpectedContentOnStorage();

                // ダウンロード済コンテンツの消去確認. RequestDownloadLatestUpdate != nn::ns::ResultAlreadyUpToDate
                LOG_OUT( "Verify whether the downloaded contents be clean... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate]\n" );
                NewlyUpdateDownloader downloader( control );
                NNT_EXPECT_RESULT_SUCCESS( control.RequestDownloadLatestUpdate( &downloader.GetHandle() ) );
                NNT_EXPECT_RESULT_FAILURE( nn::ns::ResultAlreadyUpToDate, downloader.WaitResponse() );
                LOG_OUT( "Complete, verify whether the downloaded contents be clean.\n" );
            }
        }
    }

} // NOLINT(impl/function_size)

//!--------------------------------------------------------------------------------------
//! Google テストメソッド
//!--------------------------------------------------------------------------------------
TEST_F( NupCase, PersistenceOfDownloadedPackages )
{
    char pCaseValue[ 64 ] = "Nothing";
    testns::QueryCommandLineOptions( "--testns_ActionCase", sizeof( pCaseValue ), pCaseValue );
    LOG_OUT( "+++++ PersistenceOfDownloadedPackages [ %s ] +++++\n", pCaseValue );

    if ( 0 == std::strcmp( pCaseValue, "ExitWithinDownloading" ) )
    {
        // ダウンロード && ( total > 0 ) && 1秒経過で終了します。
        DoDownloadAction( []( NupCase& context, nn::ns::SystemUpdateControl& control )->bool
        {
            NN_UNUSED( context );
            LOG_OUT( "===== Start update downloading with exit at downloading... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate] =====\n" );
            NewlyUpdateDownloader downloader( control );
            NewlyUpdateDownloader::ExitConditionType exitCondition( ExitCategoryForDownload::InDownloading, 1000 );
            NNT_EXPECT_RESULT_SUCCESS( control.RequestDownloadLatestUpdate( &downloader.GetHandle() ) );
            NNT_EXPECT_RESULT_SUCCESS( downloader.WaitResponse( &exitCondition ) );
            LOG_OUT( "Complete update downloading with exit at downloading.\n" );
            return false;
        }, nn::ns::LatestSystemUpdate::NeedsDownload );
    }
    else if ( 0 == std::strcmp( pCaseValue, "ExitAtDownloadDone" ) )
    {
        // ダウンロード & 完了まで, Install はしない。
        DoDownloadAction( []( NupCase& context, nn::ns::SystemUpdateControl& control )->bool
        {
            NN_UNUSED( context );
            LOG_OUT( "===== Start latest update downloading... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate] =====\n" );
            NewlyUpdateDownloader downloader( control );
            NNT_EXPECT_RESULT_SUCCESS( control.RequestDownloadLatestUpdate( &downloader.GetHandle() ) );
            NNT_EXPECT_RESULT_SUCCESS( downloader.WaitResponse() );
            LOG_OUT( "Complete latest update downloading.\n" );
            EXPECT_EQ( true, control.HasDownloaded() );
            return false;
        }, nn::ns::LatestSystemUpdate::NeedsDownload );
    }
    else if ( 0 == std::strcmp( pCaseValue, "ApplyDownloadedUpdate" ) )
    {
        // RequestCheckLatestUpdate が LatestSystemUpdate::Downloaded であること。
        // 一致しなかった場合は ASSERT, 一致すれば、インストール＆ncmベリファイを行う。
        DoDownloadAction( nullptr, nn::ns::LatestSystemUpdate::Downloaded, false );
    }
    else if ( 0 == std::strcmp( pCaseValue, "Expect_NeedsDownload" ) )
    {
        // RequestCheckLatestUpdate が LatestSystemUpdate::NeedsDownload であること。
        // 一致しなかった場合は ASSERT, 一致すれば CheckLatestUpdate だけで終了する。
        DoDownloadAction( nullptr, nn::ns::LatestSystemUpdate::NeedsDownload );
    }
    else if ( 0 == std::strcmp( pCaseValue, "Expect_Downloaded" ) )
    {
        // RequestCheckLatestUpdate が LatestSystemUpdate::Downloaded であること。
        // 一致しなかった場合は ASSERT, 一致すれば CheckLatestUpdate だけで終了する。
        DoDownloadAction( nullptr, nn::ns::LatestSystemUpdate::Downloaded );
    }
    else if ( 0 == std::strcmp( pCaseValue, "Expect_UpToDate" ) )
    {
        // RequestCheckLatestUpdate が LatestSystemUpdate::UpToDate であること。
        // 一致しなかった場合は ASSERT, 一致すれば CheckLatestUpdate だけで終了する。
        DoDownloadAction( nullptr, nn::ns::LatestSystemUpdate::UpToDate );
    }
    else
    {
        // ActionCase の指定がないなら、普通にダウンロードしてインストールしてしまう。
        DoDownloadAction( []( NupCase& context, nn::ns::SystemUpdateControl& control )->bool
        {
            NN_UNUSED( context );
            LOG_OUT( "===== Start latest update downloading... On API [nn::ns::SystemUpdateControl::RequestDownloadLatestUpdate] =====\n" );
            NewlyUpdateDownloader downloader( control );
            NNT_EXPECT_RESULT_SUCCESS( control.RequestDownloadLatestUpdate( &downloader.GetHandle() ) );
            NNT_EXPECT_RESULT_SUCCESS( downloader.WaitResponse() );
            LOG_OUT( "Complete latest update downloading.\n" );
            bool downloaded = control.HasDownloaded();
            EXPECT_EQ( true, downloaded );
            return downloaded;
        }, nn::ns::LatestSystemUpdate::NeedsDownload, false );
    }

} // NOLINT(impl/function_size)
