﻿<#
    .SYNOPSIS
        Private option utilities module

    .DESCRIPTION
        This file defines functions for private operations
#>

# 基本パス関連 ( 全部文字列型らしい [string] )
$MyScriptPath          = $MyInvocation.MyCommand.Path
$MyScriptDirectoryPath = [System.IO.Path]::GetDirectoryName($MyScriptPath)

# -----------------------------------------------------------------------
# スレッドプール作成
# $defMinSpace : コンテキスト最小数
# $defMaxSpace : 一度にオープン可能なコンテキスト最大数
# return : [System.Management.Automation.Runspaces.RunspacePool]
# -----------------------------------------------------------------------
function Open-RunspacePool( [int]$defMinSpace = 1, [int]$defMaxSpace = 1 )
{
    $defRsPool = [System.Management.Automation.Runspaces.RunspaceFactory]::CreateRunspacePool( $defMinSpace, $defMaxSpace )
    $defRsPool.Open() | Out-Null
    return $defRsPool
}

# -----------------------------------------------------------------------
# スレッドプール破棄
# -----------------------------------------------------------------------
function Close-RunspacePool( [System.Management.Automation.Runspaces.RunspacePool]$RsPoolObject )
{
    if ( $null -ne $RsPoolObject )
    {
        $RsPoolObject.Close() | Out-Null
    }
}

# -----------------------------------------------------------------------
# スレッド開始( Runspace )
# -----------------------------------------------------------------------
function Open-Runspace()
{
    Param (
        $ScriptBlock,
        $ScriptArguments = $null,
        [System.Management.Automation.Runspaces.RunspacePool]$RsPoolObject = $null
    )
    [System.Management.Automation.Runspaces.RunspacePool]$defPrivateRsp = $null
    [System.Management.Automation.Runspaces.RunspacePool]$defActiveRsp = ${RsPoolObject}
    if ( $null -eq $defActiveRsp )
    {
        $defPrivateRsp = $(Open-RunspacePool 1 1)
        $defActiveRsp = ${defPrivateRsp}
    }
    $psh = [PowerShell]::Create()
    $psh.RunspacePool = ${defActiveRsp}
    # ユーザ定義プロパティオブジェクト
    $handle = New-Object PSObject -Property @{
        vPsh = ${psh}
        vPool = ${defPrivateRsp}
        vAsyncResult = $null
    }
    $psh.AddScript( ${ScriptBlock} ) | Out-Null
    $psh.AddArgument( $handle ) | Out-Null
    $psh.AddArgument( $ScriptArguments ) | Out-Null
    #$psh.AddCommand( "Out-String" ) | Out-Null
    # [System.Management.Automation.PowerShellAsyncResult]
    $handle.vAsyncResult = $psh.BeginInvoke()
    return $handle
}

# -----------------------------------------------------------------------
# スレッド破棄( Runspace )
# 強制終了します
# -----------------------------------------------------------------------
function Close-Runspace( $RsHandle )
{
    $defResult = $null
    if ( $null -ne $RsHandle )
    {
        $defShell = $RsHandle.vPsh
        $defAsyncResult = $RsHandle.vAsyncResult
        if ( $null -ne ${defAsyncResult} -and $null -ne ${defShell} )
        {
            $defResult = $defShell.EndInvoke( $defAsyncResult )
            $defShell.Dispose() | Out-Null
            $RsHandle.vAsyncResult = $null
            $RsHandle.vPsh = $null
        }
        elseif ( $null -ne ${defShell} )
        {
            $defShell.Dispose() | Out-Null
            $RsHandle.vPsh = $null
        }
        Close-RunspacePool( $RsHandle.vPool ) | Out-Null
    }
    return $defResult
}

# -----------------------------------------------------------------------
# スレッド待機( Runspace )
# -----------------------------------------------------------------------
function Wait-Runspace( $RsHandle, [int]$TimeoutMillis = [System.Threading.Timeout]::Infinite )
{
    $defResult = $null
    if ( $null -ne ${RsHandle} -and $null -ne $RsHandle.vAsyncResult -and $null -ne $RsHandle.vPsh )
    {
        $defAsyncResult = $RsHandle.vAsyncResult
        $defWaitHandle = $defAsyncResult.AsyncWaitHandle # [System.Threading.WaitHandle]
        if ( $true -eq $defWaitHandle.WaitOne( ${TimeoutMillis}, $false ) )
        {
            # EndInvoke 返値はスレッド結果のオブジェクト配列 [PSDataCollection<PSObject>]
            $defShell = $RsHandle.vPsh
            $defResult = $defShell.EndInvoke( $defAsyncResult )
            $defShell.Dispose() | Out-Null
            $RsHandle.vAsyncResult = $null
            $RsHandle.vPsh = $null
        }
    }
    return ${defResult}
}
Export-ModuleMember -Function *
