﻿<#
    .SYNOPSIS
        Private option utilities module

    .DESCRIPTION
        This file defines functions for private operations
#>

# 基本パス関連 ( 全部文字列型らしい [string] )
$MyScriptPath          = $MyInvocation.MyCommand.Path
$MyScriptDirectoryPath = [System.IO.Path]::GetDirectoryName($MyScriptPath)
$BasicModulePath       = "${MyScriptDirectoryPath}\..\..\..\SystemUpdateBasic\Scripts\Modules"

Import-Module "${BasicModulePath}\PlatformUtilities"

# SDK ROOTパス
$NintendoSdkRootPath = $(Find-NintendoSdkRootPath)

Import-Module "${BasicModulePath}\FileUtilities"
Import-Module "${BasicModulePath}\SdkToolUtilities"
Import-Module "${BasicModulePath}\XmlFileUtilities"

# -----------------------------------------------------------------------
# [xml] コンテンツディレクティブ作成 ( コンテンツメタ )
# 返値 : [System.Xml.XmlElement], $defXmlRoot 経由で生成したディレクティブ要素
# $defMetaType : ディレクティブ型名, SystemProgram, SystemData
# $defMetaId : "0x"不要
# $defMetaVersion : 正の整数値
# -----------------------------------------------------------------------
function New-CreateContentMetaDirectiveAsContent( [ref]$defXmlDocRef, [string]$defMetaType, [string]$defMetaId, [int]$defMetaVersion )
{
    $defRoot = $defXmlDocRef.Value
    $defDirective = $defRoot.CreateElement( ${defMetaType} )

    $defElementId = $defRoot.CreateElement( "Id" )
    $defElementId.set_InnerText( "0x${defMetaId}" )

    $defElementVersion = $defRoot.CreateElement( "Version" )
    $defElementVersion.set_InnerText( "${defMetaVersion}" )

    $defDirective.AppendChild( ${defElementId} ) | Out-Null
    $defDirective.AppendChild( ${defElementVersion} ) | Out-Null
    return [System.Xml.XmlElement] ${defDirective}
}
Export-ModuleMember -Function New-CreateContentMetaDirectiveAsContent

# -----------------------------------------------------------------------
# [xml] コンテンツディレクティブ作成 ( システムアップデートメタ )
# 返値 : [System.Xml.XmlElement], $defXmlRoot 経由で生成したディレクティブ要素
# $defMetaType : ディレクティブ型名, SystemProgram, SystemData
# $defMetaId : "0x"不要
# $defMetaVersion : 正の整数値
# -----------------------------------------------------------------------
function New-CreateContentMetaDirectiveAsSystemUpdate( [ref]$defXmlDocRef, [string]$defMetaType, [string]$defMetaId, [int]$defMetaVersion )
{
    $defRoot = $defXmlDocRef.Value
    $defDirective = $defRoot.CreateElement( "ContentMeta" )

    $defElementType = $defRoot.CreateElement( "Type" )
    $defElementType.set_InnerText( "${defMetaType}" )

    $defElementId = $defRoot.CreateElement( "Id" )
    $defElementId.set_InnerText( "0x${defMetaId}" )

    $defElementVersion = $defRoot.CreateElement( "Version" )
    $defElementVersion.set_InnerText( "${defMetaVersion}" )

    $defDirective.AppendChild( ${defElementType} ) | Out-Null
    $defDirective.AppendChild( ${defElementId} ) | Out-Null
    $defDirective.AppendChild( ${defElementVersion} ) | Out-Null
    return [System.Xml.XmlElement] ${defDirective}
}
Export-ModuleMember -Function New-CreateContentMetaDirectiveAsSystemUpdate

# -----------------------------------------------------------------------
# MakeTestApplication 複数nsp一括生成用 config xml 作成
# $defMetaId            : 対象nsp群が含まれる SystemUpdateMetaのID ( "0x"不要 ), xmlファイル名に使用されます。
# $defMetaVersion       : 対象nsp群が含まれる SystemUpdateMetaのバージョン ( 正の整数値 )。
# $defContentType       : コンテンツ型名, SystemProgram, SystemData。
# $defContentSize       : ダミーROMデータサイズ( byte )。
# $defContentBeginId    : 対象nsp群の開始メタID, このメタIDから ${defContentCount} の個数分 +1 しながらnsp用メタIDを生成します。
# $defContentCount      : 生成するnsp 個数。
# -----------------------------------------------------------------------
function Out-CreateTestApplicationConfigXml(
    [string]$defIntermediateDirectory,
    [string]$defMetaId,
    [int]$defMetaVersion,
    [string]$defContentType,
    [System.UInt64]$defContentSize,
    [System.UInt64]$defContentBeginId,
    [System.UInt32]$defContentCount
)
{
    [string] $defStringContentSize = [string] ${defContentSize}
    [string] $defFileNameBase = $(Get-ContentNspFileBaseName ${defMetaId} ${defMetaVersion} "")
    [string] $defIntermediateFile = "${defIntermediateDirectory}\${defFileNameBase}.config.xml"
    $(Remove-ExistFile ${defIntermediateFile})

    [xml] $defXmlDoc = New-Object xml
    [ref] $refXmlDoc = ${defXmlDoc}

    $defDocRoot = $defXmlDoc.CreateElement( "TestApplicationSetting" )
    $defXmlDoc.AppendChild( ${defDocRoot} ) | Out-Null

    for ( [System.UInt32] $defCount = 0; ${defCount} -lt ${defContentCount}; ++$defCount )
    {
        # <TestApplication>
        $defContentRoot = $defXmlDoc.CreateElement( "TestApplication" )

        # <MetaString>
        $defMetaString = $defXmlDoc.CreateElement( "MetaSetting" )
        [System.UInt64] $defSystemDataMetaId = ${defContentBeginId} + ${defCount}
        [string] $defStringSystemDataMetaId = $defSystemDataMetaId.ToString( "X16" )
        $defMetaBody = $(New-CreateContentMetaDirectiveAsContent ${refXmlDoc} ${defContentType} ${defStringSystemDataMetaId} ${defMetaVersion})
        $defMetaString.AppendChild( ${defMetaBody} ) | Out-Null

        # <Type><Size><FileName>
        $defNspBaseFileName = $(Get-ContentNspFileBaseName ${defStringSystemDataMetaId} ${defMetaVersion} "")
        $defKeyContentFile = $defXmlDoc.CreateElement( "FileName" )
        $defKeyContentFile.set_InnerText( ${defNspBaseFileName} )
        $defKeyContentType = $defXmlDoc.CreateElement( "Type" )
        $defKeyContentType.set_InnerText( ${defContentType} )
        $defKeyContentSize = $defXmlDoc.CreateElement( "Size" )
        $defKeyContentSize.set_InnerText( ${defStringContentSize} )

        $defContentRoot.AppendChild( ${defKeyContentType} ) | Out-Null
        $defContentRoot.AppendChild( ${defKeyContentSize} ) | Out-Null
        $defContentRoot.AppendChild( ${defKeyContentFile} ) | Out-Null
        $defContentRoot.AppendChild( ${defMetaString} ) | Out-Null
        $defDocRoot.AppendChild( ${defContentRoot} ) | Out-Null
    }

    $(Out-WriteXml ${refXmlDoc} ${defIntermediateFile})
    return ${defIntermediateFile}
}
Export-ModuleMember -Function Out-CreateTestApplicationConfigXml

# -----------------------------------------------------------------------
# AuthoringTool用 複数nsp群を保持する system update meta file 作成。
#
# $defMetaId            : SystemUpdateMetaのID ( "0x"不要 )。
# $defMetaVersion       : SystemUpdateMetaのバージョン ( 正の整数値 )。
# $defContentType       : コンテンツ型名, SystemProgram, SystemData。
# $defContentBeginId    : SystemUpdateMeta に含まれる対象nsp群の開始メタID, このメタIDから ${defContentCount} の個数分 +1 しながらnsp用メタIDを生成します。
# $defContentCount      : SystemUpdateMeta に含まれる対象nsp群の個数。
# -----------------------------------------------------------------------
function Out-CreateSystemUpdateMetaXml(
    [string]$defIntermediateDirectory,
    [string]$defMetaId,
    [int]$defMetaVersion,
    [string]$defContentType,
    [System.UInt64]$defContentBeginId,
    [System.UInt32]$defContentCount
)
{
    [string] $defFileNameBase = $(Get-ContentNspFileBaseName ${defMetaId} ${defMetaVersion} "")
    [string] $defIntermediateFile = "${defIntermediateDirectory}\${defFileNameBase}.nmeta"
    $(Remove-ExistFile ${defIntermediateFile})

    [xml] $defXmlDoc = New-Object xml
    [ref] $refXmlDoc = ${defXmlDoc}

    # <NintendoSdkMeta>
    #    <SystemUpdate>
    #       <ContentMeta>
    $defDocRoot = $defXmlDoc.CreateElement( "NintendoSdkMeta" )
    $defXmlDoc.AppendChild( ${defDocRoot} ) | Out-Null
    $defMetaRoot = $defXmlDoc.CreateElement( "SystemUpdate" )
    $defDocRoot.AppendChild( ${defMetaRoot} ) | Out-Null
    $defDirectiveRoot = $(New-CreateContentMetaDirectiveAsSystemUpdate ${refXmlDoc} "SystemUpdate" ${defMetaId} ${defMetaVersion})
    $defMetaRoot.AppendChild( ${defDirectiveRoot} ) | Out-Null

    #         <ContentMeta>
    #           <Type>SystemData</Type>
    #                   :
    for ( [System.UInt32] $defCount = 0; ${defCount} -lt ${defContentCount}; ++$defCount )
    {
        [System.UInt64] $defSystemDataMetaId = ${defContentBeginId} + ${defCount}
        $defItemContent = $(New-CreateContentMetaDirectiveAsSystemUpdate ${refXmlDoc} ${defContentType} $defSystemDataMetaId.ToString( "X16" ) ${defMetaVersion})
        $defDirectiveRoot.AppendChild( ${defItemContent} ) | Out-Null
    }
    $(Out-WriteXml ${refXmlDoc} ${defIntermediateFile})
    return ${defFileNameBase}
}
Export-ModuleMember -Function Out-CreateSystemUpdateMetaXml

# -----------------------------------------------------------------------
# MakeTestApplication 複数 SystemData nsp 一括生成用 config xml 作成。
# content-size "4", meta-type "SystemData" 固定。
#
# $defMetaId            : 対象nsp群が含まれる SystemUpdateMetaのID ( "0x"不要 ), xmlファイル名に使用されます。
# $defMetaVersion       : 対象nsp群が含まれる SystemUpdateMetaのバージョン ( 正の整数値 )。
# $defContentBeginId    : 対象nsp群の開始メタID, このメタIDから ${defContentCount} の個数分 +1 しながらnsp用メタIDを生成します。
# $defContentCount      : 生成するnsp 個数。
# -----------------------------------------------------------------------
function Out-CreateTestApplicationConfigForSystemData( [string]$defIntermediateDirectory, [string]$defMetaId, [int]$defMetaVersion, [System.UInt64]$defContentBeginId, [System.UInt32]$defContentCount )
{
    return $(Out-CreateTestApplicationConfigXml ${defIntermediateDirectory} ${defMetaId} ${defMetaVersion} "SystemData" 4 ${defContentBeginId} ${defContentCount})
}
Export-ModuleMember -Function Out-CreateTestApplicationConfigForSystemData

# -----------------------------------------------------------------------
# AuthoringTool用 複数nsp群を保持する system update meta file 作成。
# SystemDataコンテンツ用
#
# $defMetaId            : SystemUpdateMetaのID ( "0x"不要 )。
# $defMetaVersion       : SystemUpdateMetaのバージョン ( 正の整数値 )。
# $defContentType       : コンテンツ型名, SystemProgram, SystemData。
# $defContentBeginId    : SystemUpdateMeta に含まれる対象nsp群の開始メタID, このメタIDから ${defContentCount} の個数分 +1 しながらnsp用メタIDを生成します。
# $defContentCount      : SystemUpdateMeta に含まれる対象nsp群の個数。
# -----------------------------------------------------------------------
function Out-CreateSystemUpdateMetaForSystemData( [string]$defIntermediateDirectory, [string]$defMetaId, [int]$defMetaVersion, [System.UInt64]$defContentBeginId, [System.UInt32]$defContentCount )
{
    return $(Out-CreateSystemUpdateMetaXml ${defIntermediateDirectory} ${defMetaId} ${defMetaVersion} "SystemData" ${defContentBeginId} ${defContentCount})
}
Export-ModuleMember -Function Out-CreateSystemUpdateMetaForSystemData
