﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_StringUtil.h>
#include "testNs_NcmStorageVerify.h"
#include "libraries/testNs_MountHost.h"
#include "libraries/testNs_RawFileLoader.h"
#include "libraries/testNs_PerformanceMeasure.h"

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <algorithm>

#define LOG_OUT( ... )      NN_LOG( "[NsSystemUpdate] " __VA_ARGS__ )
#define LOG_OUT_NO_PREFIX   NN_LOG

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2012
#define strtoull _strtoui64
#endif

namespace {

    //!--------------------------------------------------------------------------------------
    //! @brief 配列要素数算出.
    //!--------------------------------------------------------------------------------------
    template< typename TYPE, size_t SIZE >
    inline static int CountOf( const TYPE( & )[ SIZE ] ) {
        return static_cast< int >( SIZE );
    }
    //!--------------------------------------------------------------------------------------
    //! @brief 文字変換ユーティリティ
    //!--------------------------------------------------------------------------------------
    template< unsigned SIZE = 64 >
    class StringConverter
    {
    public:
        //!--------------------------------------------------------------------------------------
        //! @brief 変換済文字列の取得
        //!--------------------------------------------------------------------------------------
        const char* GetValue() const NN_NOEXCEPT
        {
            return m_Buffer;
        }

        //!--------------------------------------------------------------------------------------
        //! @brief ContentId の文字列化
        //!--------------------------------------------------------------------------------------
        explicit StringConverter( const nn::ncm::ContentId& id ) NN_NOEXCEPT
        {
            ToString( m_Buffer, id );
        }

        //!--------------------------------------------------------------------------------------
        //! @brief ContentId の文字列化
        //!--------------------------------------------------------------------------------------
        static const char* ToString( char* pOutBuffer, const nn::ncm::ContentId& id ) NN_NOEXCEPT
        {
            char* pNext = pOutBuffer;
            int n = CountOf( id.data );
            for ( int i = 0; i < n; ++i )
            {
                pNext = ToString( pNext, id.data[ i ] );
            }
            pNext[ 0 ] = '\0';
            return pOutBuffer;
        }

        //!--------------------------------------------------------------------------------------
        //! @brief 8bit 数値の 数字コード化
        //!--------------------------------------------------------------------------------------
        static char* ToString( char* outBuffer, const nn::Bit8 value ) NN_NOEXCEPT
        {
            const unsigned upper = ( ( value >> 4 ) & 0x0f );
            const unsigned lower = ( ( value >> 0 ) & 0x0f );
            outBuffer[ 0 ] = static_cast< char >( ( upper < 0x0a ) ? ( '0' + upper ) : ( 'a' + ( upper - 0x0a ) ) );
            outBuffer[ 1 ] = static_cast< char >( ( lower < 0x0a ) ? ( '0' + lower ) : ( 'a' + ( lower - 0x0a ) ) );
            return &outBuffer[ 2 ];
        }
    private:
        char m_Buffer[ SIZE ];
    };

    //!--------------------------------------------------------------------------------------
    //! @brief 文字列を数値配列化.
    //!--------------------------------------------------------------------------------------
    static bool ConvertBinaryFromString( nn::Bit8* buffer, const size_t bufferSize, const char* pString, const size_t length ) NN_NOEXCEPT
    {
        if ( ( length % 2 != 0 ) || ( bufferSize * 2 < length ) )
        {
            return false;
        }
        for ( size_t i = 0; i < length; i += 2 )
        {
            char byteString[ 3 ];
            char* errptr = NULL;
            nn::util::Strlcpy( byteString, &pString[ i ], sizeof( byteString ) );
            buffer[ i / 2 ] = static_cast< nn::Bit8 >( std::strtoul( byteString, &errptr, 16 ) );
            if ( *errptr != '\0' )
            {
                return false;
            }
        }
        return true;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 文字列からコンテンツID生成.
    //!--------------------------------------------------------------------------------------
    static const bool GetContentIdFromString( nn::ncm::ContentId& outValue, const char* pString, const size_t length ) NN_NOEXCEPT
    {
        static const size_t s_ContentIdStringLength = sizeof( nn::ncm::ContentId ) * 2;
        if ( length < s_ContentIdStringLength )
        {
            return false;
        }
        return ConvertBinaryFromString( outValue.data, sizeof( outValue.data ), pString, s_ContentIdStringLength );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief インストール済コンテンツメタ数取得
    //!--------------------------------------------------------------------------------------
    static const unsigned GetInstalledContentMetaCount( nn::ncm::ContentMetaDatabase& db ) NN_NOEXCEPT
    {
        nn::ncm::ContentMetaKey pBuffer[ 8 ];
        const auto count = db.ListContentMeta( pBuffer, CountOf( pBuffer ), nn::ncm::ContentMetaType::Unknown );
        return static_cast< unsigned >( count.total );
    }
}

//!---------------------------------------------------------------------------------
//! @brief コンストラクタ
//!---------------------------------------------------------------------------------
ContentIdCollection::ContentIdCollection( const unsigned reservedCapacity ) NN_NOEXCEPT
{
    CollectionType::reserve( reservedCapacity );
    Clean();
}

//!---------------------------------------------------------------------------------
//! @brief デストラクタ
//!---------------------------------------------------------------------------------
ContentIdCollection::~ContentIdCollection() NN_NOEXCEPT
{
    Clean();
}

//!---------------------------------------------------------------------------------
//! @brief クリーン
//!---------------------------------------------------------------------------------
void ContentIdCollection::Clean() NN_NOEXCEPT
{
    CollectionType::clear();
}

//!---------------------------------------------------------------------------------
//! @brief 任意のコンテンツIDの追加
//!---------------------------------------------------------------------------------
void ContentIdCollection::Entry( const nn::ncm::ContentId& contentId ) NN_NOEXCEPT
{
    if ( false == Find( contentId ) )
    {
        std::string key( StringConverter<>( contentId ).GetValue() );
        ( *this )[ key ] = contentId;
    }
}

//!---------------------------------------------------------------------------------
//! @brief ストレージ上のコンテンツIDを取得
//!---------------------------------------------------------------------------------
void ContentIdCollection::LoadFrom( nn::ncm::ContentStorage& storage ) NN_NOEXCEPT
{
    LOG_OUT( "Loading an installed content from the built-in system storages...\n" );

    Clean();
    unsigned nAccumCount = 0;
    nn::ncm::ContentId outBuffer[ 8192 ];
    const int nBufferCount = CountOf( outBuffer );
    for ( int offset = 0, nActualCount = nBufferCount; nActualCount >= nBufferCount; offset += nBufferCount )
    {
        // 読み込み
        auto beginTickLoad = testns::SectionPerformanceMeasureLite::Begin();
        nActualCount = 0;   // nActualCountは、outBuffer内の有効なカウント数。
        NNT_ASSERT_RESULT_SUCCESS( storage.ListContentId( &nActualCount, outBuffer, nBufferCount, offset ) );

        nAccumCount += nActualCount;
        LOG_OUT( "%d Loaded ( %lld ms ).\n", nAccumCount, testns::SectionPerformanceMeasureLite::End( beginTickLoad ) );

        // 登録
        auto beginTickEntry = testns::SectionPerformanceMeasureLite::Begin();
        for ( int i = 0; i < nActualCount; ++i )
        {
            Entry( outBuffer[ i ] );
        }
        LOG_OUT( "%d Entried ( %lld ms ).\n", nAccumCount, testns::SectionPerformanceMeasureLite::End( beginTickEntry ) );
    }

    LOG_OUT( "Loading an installed content from the built-in system storages, [ %u ] : done.\n", nAccumCount );
}

//!---------------------------------------------------------------------------------
//! @brief 検索
//!---------------------------------------------------------------------------------
const bool ContentIdCollection::Find( const nn::ncm::ContentId& contentId ) const NN_NOEXCEPT
{
    std::string key( StringConverter<>( contentId ).GetValue() );
    return Find( key );
}

//!---------------------------------------------------------------------------------
//! @brief 検索
//!---------------------------------------------------------------------------------
const bool ContentIdCollection::Find( const std::string& contentId ) const NN_NOEXCEPT
{
    auto it = CollectionType::find( contentId );
    return ( it != CollectionType::end() );
}

//!---------------------------------------------------------------------------------
//! @brief 比較
//!---------------------------------------------------------------------------------
const bool ContentIdCollection::Compare( const ContentIdCollection& other ) const NN_NOEXCEPT
{
    size_t mySize, otherSize;
    if ( ( mySize = size() ) != ( otherSize = other.size() ) )
    {
        LOG_OUT( "There is a difference of the collection size [ %u / %u ]\n",
            static_cast< unsigned >( mySize ),
            static_cast< unsigned >( otherSize )
        );
        return false;
    }

    bool result = true;
    for ( auto it = other.cbegin(); it != other.cend(); ++it )
    {
        auto& element = *it;
        if ( false == Find( element.first ) )
        {
            LOG_OUT( "Detect unexpected the content [ %s ]\n", element.first.c_str() );
            result = false;
        }
    }
    return result;
}

//!---------------------------------------------------------------------------------
//! @brief コレクションダンプ
//!---------------------------------------------------------------------------------
void ContentIdCollection::Dump( const char* pPrefix ) const NN_NOEXCEPT
{
    LOG_OUT( "%s {\n", ( ( nullptr != pPrefix ) ? pPrefix : "ContentIdCollection" ) );
    for ( auto it = CollectionType::cbegin(); it != CollectionType::cend(); ++it )
    {
        auto& element = *it;
        LOG_OUT( "  [ %s ]\n", element.first.c_str() );
    }
}



//!---------------------------------------------------------------------------------
//! @brief コンストラクタ
//!---------------------------------------------------------------------------------
VerifyContentMetaKey::VerifyContentMetaKey( const unsigned reservedCapacity ) NN_NOEXCEPT
{
    CollectionType::reserve( reservedCapacity );
    Clean();
}

//!---------------------------------------------------------------------------------
//! @brief デストラクタ
//!---------------------------------------------------------------------------------
VerifyContentMetaKey::~VerifyContentMetaKey() NN_NOEXCEPT
{
    Clean();
}

//!---------------------------------------------------------------------------------
//! @brief クリーン
//!---------------------------------------------------------------------------------
void VerifyContentMetaKey::Clean() NN_NOEXCEPT
{
    EntryMetaKey( 0, 0, nn::ncm::ContentMetaType::Unknown );
}

//!---------------------------------------------------------------------------------
//! @brief 設定
//!---------------------------------------------------------------------------------
void VerifyContentMetaKey::EntryMetaKey( const nn::Bit64& metaId, const uint32_t metaVersion,
    const nn::ncm::ContentMetaType metaType, const nn::ncm::ContentInstallType metaInstallType ) NN_NOEXCEPT
{
    nn::ncm::ContentMetaKey* const pThisKey = static_cast< nn::ncm::ContentMetaKey* >( this );
    pThisKey[ 0 ] = nn::ncm::ContentMetaKey::Make( metaId, metaVersion, metaType, metaInstallType );
    CollectionType::clear();
}

//!---------------------------------------------------------------------------------
//! @brief コンテンツID追加
//!---------------------------------------------------------------------------------
void VerifyContentMetaKey::AppendContent( const nn::ncm::ContentId& contentId, const nn::ncm::ContentType contentType ) NN_NOEXCEPT
{
    if ( contentId.IsValid() )
    {
        CollectionType::push_back( nn::ncm::ContentInfo::Make( contentId, 0, contentType, 0 ) );
    }
}

//!---------------------------------------------------------------------------------
//! @brief コンテンツIDが含まれるか
//!---------------------------------------------------------------------------------
const bool VerifyContentMetaKey::HasContent( const nn::ncm::ContentId& contentId ) const NN_NOEXCEPT
{
    for ( auto it = CollectionType::cbegin(); it != CollectionType::cend(); ++it )
    {
        auto& info = *it;
        if ( contentId == info.GetId() )
        {
            return true;
        }
    }
    return false;
}



//!---------------------------------------------------------------------------------
//! @brief コンストラクタ
//!---------------------------------------------------------------------------------
ContentMetaKeyCollection::ContentMetaKeyCollection( const unsigned reservedCapacity ) NN_NOEXCEPT
    : m_pXmlParser( nullptr ),
    m_DetectedMetaKey( false )
{
    ParentType::reserve( reservedCapacity );
}

//!---------------------------------------------------------------------------------
//! @brief デストラクタ
//!---------------------------------------------------------------------------------
ContentMetaKeyCollection::~ContentMetaKeyCollection() NN_NOEXCEPT
{
    Finalize();
}

//!---------------------------------------------------------------------------------
//! @brief ファイナライザ
//!---------------------------------------------------------------------------------
void ContentMetaKeyCollection::Finalize() NN_NOEXCEPT
{
    XmlParser* pParser;
    if ( nullptr != ( pParser = m_pXmlParser ) )
    {
        m_pXmlParser = nullptr;
        XmlParser::Finalize( pParser );
    }
    ParentType::clear();
}

//!---------------------------------------------------------------------------------
//! @brief コレクションローダー ( from File )
//!---------------------------------------------------------------------------------
void ContentMetaKeyCollection::LoadFromPropertyFile( const char* pFilePath ) NN_NOEXCEPT
{
    // XML リソースファイルを読み込み
    XmlParser* pParser;
    if ( nullptr == ( pParser = m_pXmlParser ) )
    {
        m_pXmlParser = pParser = XmlParser::CreateNewParser();
    }
    const char* pStream;
    RawFileLoader loader;
    if ( nullptr != ( pStream = loader.LoadFromFile< char >( pFilePath ) ) )
    {
        ParentType::clear();
        m_DetectedMetaKey = false;
        pParser->Parse( *this, pStream, loader.GetFileSize() );
    }
}

//!---------------------------------------------------------------------------------
//! @brief
//!---------------------------------------------------------------------------------
void ContentMetaKeyCollection::OnElementBegin( const XmlParser::StringType& name, const XmlParser::Attribute::CollectionType& attributes ) NN_NOEXCEPT
{
    if ( 0 == name.compare( "MetaKey" ) && attributes.size() > 0 )
    {
        nn::Bit64 id = 0;
        uint32_t version = 0;
        nn::ncm::ContentMetaType type = nn::ncm::ContentMetaType::Unknown;
        for ( AttrListType::const_iterator it = attributes.begin(); it != attributes.end(); ++it )
        {
            auto& attrName = it->GetName();
            if ( 0 == attrName.compare( "type" ) )
            {
                type = LookupContentMetaType( it->GetValue() );
            }
            else if ( 0 == attrName.compare( "id" ) )
            {
                char* pEndPoint = nullptr;
                const nn::Bit64 t = static_cast< nn::Bit64 >( ::strtoull( it->GetValue().c_str(), &pEndPoint, 16 ) );
                if ( nullptr != pEndPoint && '\0' == pEndPoint[ 0 ] )
                {
                    id = t;
                }
            }
            else if ( 0 == attrName.compare( "version" ) )
            {
                version = std::strtol( it->GetValue().c_str(), nullptr, 10 );
            }
        }
        m_FocusedKey.EntryMetaKey( id, version, type );
        m_DetectedMetaKey = true;
    }
    else if ( true == m_DetectedMetaKey && 0 == name.compare( "Content" ) && attributes.size() > 0 )
    {
        nn::ncm::ContentId id;
        nn::ncm::ContentType type = nn::ncm::ContentType::Meta;
        for ( AttrListType::const_iterator it = attributes.begin(); it != attributes.end(); ++it )
        {
            auto& attrName = it->GetName();
            if ( 0 == attrName.compare( "id" ) )
            {
                const std::string& value = it->GetValue();
                GetContentIdFromString( id, value.c_str(), value.size() );
            }
            else if ( 0 == attrName.compare( "type" ) )
            {
                type = LookupContentType( it->GetValue() );
            }
        }
        m_FocusedKey.AppendContent( id, type );
    }
}

//!---------------------------------------------------------------------------------
//! @brief
//!---------------------------------------------------------------------------------
void ContentMetaKeyCollection::OnText( const XmlParser::StringType& text ) NN_NOEXCEPT
{
    NN_UNUSED( text );
}

//!---------------------------------------------------------------------------------
//! @brief
//!---------------------------------------------------------------------------------
void ContentMetaKeyCollection::OnElementEnd( const XmlParser::StringType& name ) NN_NOEXCEPT
{
    if ( 0 == name.compare( "MetaKey" ) )
    {
        if ( true == m_DetectedMetaKey )
        {
            ParentType::push_back( m_FocusedKey );
        }
        m_FocusedKey.Clean();
        m_DetectedMetaKey = false;
    }
}

//!---------------------------------------------------------------------------------
//! @brief 文字列シンボルから ContentMetaType へ変換。
//!---------------------------------------------------------------------------------
const nn::ncm::ContentMetaType ContentMetaKeyCollection::LookupContentMetaType( const std::string& expectTypeString ) NN_NOEXCEPT
{
    std::string symbol( expectTypeString );
    std::transform( symbol.begin(), symbol.end(), symbol.begin(), []( char c )
        {
            return static_cast< char >( ::tolower( c ) );
        }
    );

    nn::ncm::ContentMetaType result = nn::ncm::ContentMetaType::Unknown;
    if ( 0 == symbol.compare( "systemprogram" ) )
    {
        result = nn::ncm::ContentMetaType::SystemProgram;
    }
    else if ( 0 == symbol.compare( "systemdata" ) )
    {
        result = nn::ncm::ContentMetaType::SystemData;
    }
    else if ( 0 == symbol.compare( "systemupdate" ) )
    {
        result = nn::ncm::ContentMetaType::SystemUpdate;
    }
    else if ( 0 == symbol.compare( "bootimagepackage" ) )
    {
        result = nn::ncm::ContentMetaType::BootImagePackage;
    }
    else if ( 0 == symbol.compare( "bootimagepackagesafe" ) )
    {
        result = nn::ncm::ContentMetaType::BootImagePackageSafe;
    }
    else if ( 0 == symbol.compare( "application" ) )
    {
        result = nn::ncm::ContentMetaType::Application;
    }
    else if ( 0 == symbol.compare( "patch" ) )
    {
        result = nn::ncm::ContentMetaType::Patch;
    }
    else if ( 0 == symbol.compare( "addoncontent" ) )
    {
        result = nn::ncm::ContentMetaType::AddOnContent;
    }
    else
    {
        LOG_OUT( "Unknown meta type : %s\n", symbol.c_str() );
    }
    return result;
}

//!---------------------------------------------------------------------------------
//! @brief ContentMetaType から文字列シンボルへ変換。
//!---------------------------------------------------------------------------------
const char* ContentMetaKeyCollection::ToStringFromContentMetaType( const nn::ncm::ContentMetaType& type ) NN_NOEXCEPT
{
    switch ( type )
    {
    case nn::ncm::ContentMetaType::SystemProgram:
        return "SystemProgram";
    case nn::ncm::ContentMetaType::SystemData:
        return "SystemData";
    case nn::ncm::ContentMetaType::SystemUpdate:
        return "SystemUpdate";
    case nn::ncm::ContentMetaType::BootImagePackage:
        return "BootImagePackage";
    case nn::ncm::ContentMetaType::BootImagePackageSafe:
        return "BootImagePackageSafe";
    case nn::ncm::ContentMetaType::Application:
        return "Application";
    case nn::ncm::ContentMetaType::Patch:
        return "Patch";
    case nn::ncm::ContentMetaType::AddOnContent:
        return "AddOnContent";
    case nn::ncm::ContentMetaType::Unknown:
        return "Unknown";
    default:
        break;
    }
    return "InvalidType";
}

//!---------------------------------------------------------------------------------
//! @brief 文字列シンボルから ContentType へ変換。
//!---------------------------------------------------------------------------------
const nn::ncm::ContentType ContentMetaKeyCollection::LookupContentType( const std::string& expectTypeString ) NN_NOEXCEPT
{
    std::string symbol( expectTypeString );
    std::transform( symbol.begin(), symbol.end(), symbol.begin(), []( char c )
        {
            return static_cast< char >( ::tolower( c ) );
        }
    );

    nn::ncm::ContentType result = nn::ncm::ContentType::Meta;
    if ( 0 == symbol.compare( "meta" ) )
    {
        result = nn::ncm::ContentType::Meta;
    }
    else if ( 0 == symbol.compare( "program" ) )
    {
        result = nn::ncm::ContentType::Program;
    }
    else if ( 0 == symbol.compare( "data" ) )
    {
        result = nn::ncm::ContentType::Data;
    }
    else if ( 0 == symbol.compare( "control" ) )
    {
        result = nn::ncm::ContentType::Control;
    }
    else if ( 0 == symbol.compare( "htmldocument" ) )
    {
        result = nn::ncm::ContentType::HtmlDocument;
    }
    else if ( 0 == symbol.compare( "legalinformation" ) )
    {
        result = nn::ncm::ContentType::LegalInformation;
    }
    else
    {
        LOG_OUT( "Unknown content type : %s\n", symbol.c_str() );
    }
    return result;
}

//!---------------------------------------------------------------------------------
//! @brief ContentType から文字列シンボルへ変換。
//!---------------------------------------------------------------------------------
const char* ContentMetaKeyCollection::ToStringFromContentType( const nn::ncm::ContentType& type ) NN_NOEXCEPT
{
    switch ( type )
    {
    case nn::ncm::ContentType::Meta:
        return "Meta";
    case nn::ncm::ContentType::Program:
        return "Program";
    case nn::ncm::ContentType::Data:
        return "Data";
    case nn::ncm::ContentType::Control:
        return "Control";
    case nn::ncm::ContentType::HtmlDocument:
        return "HtmlDocument";
    case nn::ncm::ContentType::LegalInformation:
        return "LegalInformation";
    default:
        break;
    }
    return "InvalidType";
}

//!---------------------------------------------------------------------------------
//! @brief コレクションダンプ
//!---------------------------------------------------------------------------------
void ContentMetaKeyCollection::Dump( const char* pPrefix ) const NN_NOEXCEPT
{
    LOG_OUT( "%s {\n", ( ( nullptr != pPrefix ) ? pPrefix : "ContentMetaKeyCollection" ) );
    for ( auto it = cbegin(); it != cend(); ++it )
    {
        auto& metaKey = *it;
        LOG_OUT( "  MetaKey {\n" );
        LOG_OUT( "    id=\"%016llx\", version=\"%u\", type=\"%s\"\n",
            metaKey.id, metaKey.version, ToStringFromContentMetaType( metaKey.type )
        );
        for ( auto itc = metaKey.cbegin(); itc != metaKey.cend(); ++itc )
        {
            auto& content = *itc;
            LOG_OUT( "    <Content id=\"%s\", type=\"%s\" />\n",
                StringConverter<>( content.GetId() ).GetValue(),
                ToStringFromContentType( content.type )
            );
        }

        LOG_OUT( "  }\n" );
    }
    LOG_OUT( "}\n" );
}

//!---------------------------------------------------------------------------------
//! @brief メタキーコレクション取得
//!---------------------------------------------------------------------------------
void ContentMetaKeyCollection::ObtainMetaKeys( std::vector< nn::ncm::ContentMetaKey >& outCollection ) const NN_NOEXCEPT
{
    outCollection.clear();
    for ( auto it = cbegin(); it != cend(); ++it )
    {
        outCollection.push_back( *it );
    }
}




//!--------------------------------------------------------------------------------------
//! @brief コンストラクタ
//!--------------------------------------------------------------------------------------
NcmStorageVerifier::NcmStorageVerifier( const nn::ncm::StorageId storageId ) NN_NOEXCEPT
    : m_ExpectContents( 32 ),
    m_StorageId( storageId ),
    m_IgnoreExpectedContents( false )
{
}

//!--------------------------------------------------------------------------------------
//! @brief 初期化
//!
//! @pre
//!     - nn::ncm::Initialize が呼び出し済で nn::ncmコンテキストが利用可能な状態である。
//!     - NcmStorageVerifier::Initialize が呼び出されていない。
//!--------------------------------------------------------------------------------------
void NcmStorageVerifier::Initialize() NN_NOEXCEPT
{
    NNT_ASSERT_RESULT_SUCCESS( nn::ncm::OpenContentMetaDatabase( &m_ContentMetaDatabase, m_StorageId ) );
    NNT_ASSERT_RESULT_SUCCESS( nn::ncm::OpenContentStorage( &m_ContentStorage, m_StorageId ) );

    char pOutPath[ 256 ];
    if ( true == testns::MakeHostFileSystemPath( pOutPath, "/ExpectContents.xml" ) )
    {
        m_ExpectContents.LoadFromPropertyFile( pOutPath );
        m_ExpectContents.Dump( pOutPath );
    }
    else
    {
        LOG_OUT( "Can not load the verification properties files, because did not mount the host file system.\n" );
    }
    m_IgnoreExpectedContents = false;
}

//!--------------------------------------------------------------------------------------
//! @brief 終了
//!
//! @pre
//!     - nn::ncm::Initialize が呼び出し済で nn::ncmコンテキストが利用可能な状態である。
//!     - NcmStorageVerifier::Initialize が呼び出されている。
//!--------------------------------------------------------------------------------------
void NcmStorageVerifier::Finalize() NN_NOEXCEPT
{
    m_ExpectContents.Finalize();
    m_PreContentCollection.Clean();
    // プロセス側で開きっぱのはずなので、クローズしないはず。
    // NNT_ASSERT_RESULT_SUCCESS( nn::ncm::CloseContentStorageForcibly( m_StorageId ) );
    // NNT_ASSERT_RESULT_SUCCESS( nn::ncm::CloseContentMetaDatabaseForcibly( m_StorageId ) );
}

//!--------------------------------------------------------------------------------------
//! @brief 検証実施
//! @details
//!     - 期待するContentMetaKeyが全てデータベースに存在しているかチェックする。
//!     - 期待するContentMetaKeyが持つコンテンツがストレージ上に存在するかチェックする。
//!     - 期待するContentMetaKeyが持つコンテンツがデータベース上に存在するかチェックする。
//!
//! @pre
//!     - nn::ncm::Initialize が呼び出し済で nn::ncmコンテキストが利用可能な状態である。
//!     - NcmStorageVerifier::Initialize が呼び出されている。
//!--------------------------------------------------------------------------------------
void NcmStorageVerifier::VerifyHasContentOnStorage() NN_NOEXCEPT
{
    if ( true == m_IgnoreExpectedContents )
    {
        // 不整合が確定しているなどの状態が予測されるため検証をスキップします.
        LOG_OUT( "=========================\n" );
        LOG_OUT( "Skip the verification as integrity of content on storages, because expect the content mismatched.\n" );
        return;
    }
    ContentMetaKeyCollection& expectContents = m_ExpectContents;
    if ( 0 >= expectContents.size() )
    {
        LOG_OUT( "Could not found the target verification contents.\n" );
        LOG_OUT( "Check the host file system mounting.\n" );
        return;
    }
    nn::ncm::ContentMetaDatabase& db = m_ContentMetaDatabase;
    nn::ncm::ContentStorage& storage = m_ContentStorage;

    // メタキー配列の作成。
    std::vector< nn::ncm::ContentMetaKey > metaKeys;
    metaKeys.reserve( expectContents.size() );
    expectContents.ObtainMetaKeys( metaKeys );

    // 指定のContentMetaKey配列のメタキー全てがデータベースに存在しているかチェックする。
    bool result = false;
    NNT_ASSERT_RESULT_SUCCESS( db.HasAll( &result, metaKeys.data(), static_cast< int >( metaKeys.size() ) ) );
    ASSERT_EQ( true, result );

    // 各メタキーが持つコンテンツの検証.
    for ( auto it = expectContents.cbegin(); it != expectContents.cend(); ++it )
    {
        auto& metaKey = *it;    // 検証対象メタキー
        LOG_OUT( "=========================\n" );
        LOG_OUT( "Start verify the content on storage, [ %s:%016llx, ver< %u > ]\n",
            ContentMetaKeyCollection::ToStringFromContentMetaType( metaKey.type ),
            metaKey.id, metaKey.version
        );

        // 期待するコンテンツがデータベース上に存在するか。
        for ( auto itc = metaKey.cbegin(); itc != metaKey.cend(); ++itc )
        {
            auto& content = *itc;
            const auto& contentId = content.GetId();
            LOG_OUT( "Check exist the content on the database[ %s ]: ", StringConverter<>( contentId ).GetValue() );
            result = false;
            const auto rHasContent = db.HasContent( &result, metaKey, contentId );
            LOG_OUT_NO_PREFIX( ( ( true == rHasContent.IsSuccess() && true == result ) ? "OK\n" : "NG\n" ) );
            NNT_EXPECT_RESULT_SUCCESS( rHasContent );
            EXPECT_EQ( true, result );
        }

        // 期待するコンテンツがストレージ上に全て存在するかどうか。
        // 事前に期待コンテンツのデータベース存在チェックしていますが、
        // 念のため ListContentInfo で改めてDBから再取得したコンテンツIDでストレージ検証しています。
        nn::ncm::ContentInfo outBuffer[ 128 ];
        const int nBufferCount = CountOf( outBuffer );
        for ( int offset = 0, nActualCount = 1; nActualCount > 0; offset += nBufferCount )
        {
            // nActualCountは、outBuffer内の有効なカウント数。
            nActualCount = 0;
            NNT_ASSERT_RESULT_SUCCESS( db.ListContentInfo( &nActualCount, outBuffer, nBufferCount, metaKey, offset ) );
            for ( int i = 0; i < nActualCount; ++i )
            {
                const nn::ncm::ContentId& id = outBuffer[ i ].GetId();
                LOG_OUT( "Check exist the content on the storage [ %s ]: ", StringConverter<>( id ).GetValue() );
                result = false;
                const auto rHas = storage.Has( &result, id );
                LOG_OUT_NO_PREFIX( ( ( true == rHas.IsSuccess() && true == result ) ? "OK\n" : "NG\n" ) );
                NNT_EXPECT_RESULT_SUCCESS( rHas );
                EXPECT_EQ( true, result );
            }
        }
    }
}

//!--------------------------------------------------------------------------------------
//! @brief 検証実施
//! @details プレースホルダが残っていないかチェックする。
//!
//! @pre
//!     - nn::ncm::Initialize が呼び出し済で nn::ncmコンテキストが利用可能な状態である。
//!     - NcmStorageVerifier::Initialize が呼び出されている。
//!--------------------------------------------------------------------------------------
void NcmStorageVerifier::VerifyUnnecessaryPlaceHolderOnStorage() NN_NOEXCEPT
{
    nn::ncm::PlaceHolderId outPlaceHolder[ 128 ];
    nn::ncm::ContentStorage& storage = m_ContentStorage;
    LOG_OUT( "Check exist an unnecessary place holder on the storage: " );
    int nPlaceHolder = 0;
    const auto result = storage.ListPlaceHolder( &nPlaceHolder, outPlaceHolder, CountOf( outPlaceHolder ) );
    LOG_OUT_NO_PREFIX( ( ( true == result.IsSuccess() && 0 == nPlaceHolder ) ? "OK\n" : "NG\n" ) );
    NNT_EXPECT_RESULT_SUCCESS( result );
    EXPECT_EQ( nPlaceHolder, 0 );
}

//!--------------------------------------------------------------------------------------
//! @brief 検証準備
//! @details 事前のインストールコンテンツを除外コンテンツとしてリストアップする。
//!          本APIの除外コンテンツには、本体更新時に新たにインストールされる期待コンテンツも含まれる。
//!
//! @param[in]  ignoreExpectedContents  trueを指定した場合、以降の期待対象コンテンツに依存した検証をスキップします。
//!                                     これは、不整合が確定しているなどの状態が予測される場合に使用します。
//!
//! @pre
//!     - nn::ncm::Initialize が呼び出し済で nn::ncmコンテキストが利用可能な状態である。
//!     - NcmStorageVerifier::Initialize が呼び出されている。
//! @see
//!     - void VerifyUnexpectedContentOnStorage()
//!--------------------------------------------------------------------------------------
void NcmStorageVerifier::PrepareUnexpectedContentOnStorage( const bool ignoreExpectedContents ) NN_NOEXCEPT
{
    m_IgnoreExpectedContents = ignoreExpectedContents;

    LOG_OUT( "=========================\n" );
    LOG_OUT( "Count of an installed system content meta: %u\n", GetInstalledContentMetaCount( m_ContentMetaDatabase ) );
    LOG_OUT( "Count of an expected system content meta: %u\n", static_cast< unsigned >( m_ExpectContents.size() ) );

    // 追加無視指定が無ければ
    if ( false == ignoreExpectedContents )
    {
        // 既存ストレージコンテンツ( コンテンツID )の取得
        m_PreContentCollection.LoadFrom( m_ContentStorage );

        // 本体更新対象の期待コンテンツの追加
        // バージョン更新の場合でも、再起動までは前バージョンと新バージョンが同居するため.
        ContentMetaKeyCollection& expectContents = m_ExpectContents;
        for ( auto it = expectContents.cbegin(); it != expectContents.cend(); ++it )
        {
            auto& metaKey = *it;    // 検証対象メタキー
            for ( auto itc = metaKey.cbegin(); itc != metaKey.cend(); ++itc )
            {
                m_PreContentCollection.Entry( ( *itc ).GetId() );
            }
        }
    }
}

//!--------------------------------------------------------------------------------------
//! @brief 検証実施
//! @details 予期しないコンテンツが存在しないかチェックする。
//!          事前に PrepareUnexpectedContentOnStorage によって除外コンテンツをリストアップしておく。
//!          除外コンテンツ以外が検出された場合、NGとなる。
//!
//! @pre
//!     - nn::ncm::Initialize が呼び出し済で nn::ncmコンテキストが利用可能な状態である。
//!     - NcmStorageVerifier::Initialize が呼び出されている。
//! @see
//!     - void PrepareUnexpectedContentOnStorage()
//!--------------------------------------------------------------------------------------
void NcmStorageVerifier::VerifyUnexpectedContentOnStorage() NN_NOEXCEPT
{
    LOG_OUT( "=========================\n" );
    LOG_OUT( "Count of the current system storage contents, [ meta : %u ]\n", GetInstalledContentMetaCount( m_ContentMetaDatabase ) );

    if ( false == m_IgnoreExpectedContents )
    {
        // 現在のストレージコンテンツの取得
        ContentIdCollection collection;
        collection.LoadFrom( m_ContentStorage );

        // 比較
        LOG_OUT( "=========================\n" );
        LOG_OUT( "Start verify the unexpected content on storage\n" );
        LOG_OUT( "Count of the expected content on storage, [ content : %u ]\n", static_cast< unsigned >( m_PreContentCollection.size() ) );
        LOG_OUT( "Count of the current content on storage, [ content : %u ]\n", static_cast< unsigned >( collection.size() ) );
        const bool result = m_PreContentCollection.Compare( collection );
        if ( true == result )
        {
            LOG_OUT( "Not found the unexpected content on storage, that is OK.\n" );
        }
        EXPECT_EQ( true, result );
    }
}
