﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include "testNs_MountHost.h"
#include <nn/util/util_Optional.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include "testNs_RawFileLoader.h"
#include <string>

#define LOG_OUT( ... )      NN_LOG( "[NsSystemUpdate] " __VA_ARGS__ )

namespace testns { namespace {

    static nn::util::optional< std::string > g_MountedVolumeName = nn::util::nullopt;

}

//!--------------------------------------------------------------------------------------
//! @brief ホストPCのファイルシステムマウント
//! @details 実行時の引数からマウント先を指定します。
//! 引数にマウント先がなければマウントしません。
//! RunOnTarget.exe *.nsp "--testns_MountHost=path"
//!--------------------------------------------------------------------------------------
nn::Result MountHostFileSystem( const char* mountVolumeName ) NN_NOEXCEPT
{
    UnmountHostFileSystem();

    static const char* const s_OptionPrefixMountHost = "--testns_MountHost=";
    const size_t OptionPrefixMountHostLength = std::strlen( s_OptionPrefixMountHost );
    const int argc = ::nnt::GetHostArgc();
    if ( argc > 1 )
    {
        char** const argv = ::nnt::GetHostArgv();
        for ( int i = 1; i < argc; ++i )
        {
            const char* const arg = argv[ i ];
            if ( 0 == std::strncmp( arg, s_OptionPrefixMountHost, OptionPrefixMountHostLength ) )
            {
                char bMountPath[ 768 ];
                std::strcpy( bMountPath, &arg[ OptionPrefixMountHostLength ] );
                const auto result = nn::fs::MountHost( mountVolumeName, bMountPath );
                if ( true == result.IsSuccess() )
                {
                    g_MountedVolumeName = std::string( mountVolumeName );
                }
                else
                {
                    g_MountedVolumeName = nn::util::nullopt;
                }
                LOG_OUT( "Request mount host path=[ %s ]\n", bMountPath );
                LOG_OUT( "%s mount the host file system.\n", ( ( true == result.IsSuccess() ) ? "Did" : "Did not" ) );
                return result;
            }
        }
    }
    g_MountedVolumeName = nn::util::nullopt;
    LOG_OUT( "Did not mount the host file system.\n" );
    NN_RESULT_SUCCESS;
}

//!--------------------------------------------------------------------------------------
//! @brief ホストPCのファイルシステムアンマウント
//!--------------------------------------------------------------------------------------
void UnmountHostFileSystem() NN_NOEXCEPT
{
    if ( nn::util::nullopt != g_MountedVolumeName )
    {
        nn::fs::Unmount( g_MountedVolumeName.value().c_str() );
        g_MountedVolumeName = nn::util::nullopt;
    }
}

//!--------------------------------------------------------------------------------------
//! @brief ホストPCマウントボリュームのスキーム付きパスを生成
//! @param pOutValue        マウントスキームを付与したフルパス。
//!                         MountHostFileSystem() で指定したマウントボリューム + ':' +  ファイルパス + '\0' 分のバッファサイズが必要です。
//! @param pExpectFilePath  '/' をマウントルートとしたマウントスキームを除外したファイルパス。
//! @return     成功した場合 trueが返されます。@n
//!             マウントされていないなど、ホストファイルアクセスが出来ない場合は false が返されます。
//!
//! @details MakeHostFileSystemPath( outValue, "/filename.txt" ) とした場合、
//! outValue には、"Scheme:/filename.txt" が生成されます。
//!--------------------------------------------------------------------------------------
const bool MakeHostFileSystemPath( char* const pOutValue, const char* const pExpectFilePath ) NN_NOEXCEPT
{
    if ( nullptr != pOutValue && nn::util::nullopt != g_MountedVolumeName )
    {
        const std::string& scheme = g_MountedVolumeName.value();
        std::strcpy( pOutValue, scheme.c_str() );
        if ( nullptr != pExpectFilePath )
        {
            char* const pPathSegment = &pOutValue[ scheme.length() ];
            pPathSegment[ 0 ] = ':';
            std::strcpy( &pPathSegment[ 1 ], pExpectFilePath );
        }
        return true;
    }
    return false;
}

//!--------------------------------------------------------------------------------------
//! @brief ホストPCマウントボリュームの取得
//! @param pOutValue    マウントボリューム文字列を格納するバッファ。
//!                     MountHostFileSystem() で指定したマウントボリューム + '\0' 分のバッファサイズが必要です。
//!
//! @return     成功した場合 trueが返されます。@n
//!             マウントされていないなど、ホストファイルアクセスが出来ない場合は false が返されます。
//!--------------------------------------------------------------------------------------
const bool GetMountedHostFileSystemVolume( char* const pOutValue ) NN_NOEXCEPT
{
    return MakeHostFileSystemPath( pOutValue, nullptr );
}

} // namespace testns
